package com.instabug.early_crash.network

import com.instabug.commons.logging.getOrReportError
import com.instabug.commons.logging.logVerbose
import com.instabug.crash.Constants
import com.instabug.crash.settings.CrashSettings
import com.instabug.early_crash.caching.IEarlyCrashCacheHandler
import com.instabug.early_crash.threading.ExecutionMode
import com.instabug.library.diagnostics.IBGDiagnostics
import com.instabug.library.factory.ParameterizedFactory
import com.instabug.library.networkv2.NetworkManager
import com.instabug.library.networkv2.RateLimitedException
import com.instabug.library.networkv2.RequestResponse
import com.instabug.library.networkv2.request.Request
import com.instabug.library.networkv2.request.RequestType
import com.instabug.library.util.InstabugSDKLogger
import com.instabug.library.util.TimeUtils
import com.instabug.library.util.threading.SimpleCompletableFuture
import org.json.JSONObject
import java.util.concurrent.Callable
import java.util.concurrent.ExecutorService
import java.util.concurrent.Future


fun interface SingleEarlyCrashUploader<T> {
    operator fun invoke(
        id: String,
        jsonObject: JSONObject,
        cacheExecMode: ExecutionMode,
        successHandler: ((RequestResponse?) -> Unit)?
    ): T
}

class SynchronousSingleEarlyCrashUploader(
    private val cacheHandler: IEarlyCrashCacheHandler,
    private val requestFactory: ParameterizedFactory<Request?, JSONObject>,
    private val networkManager: NetworkManager,
    private val crashSettings: CrashSettings,
) : SingleEarlyCrashUploader<Runnable?> {

    override operator fun invoke(
        id: String,
        jsonObject: JSONObject,
        cacheExecMode: ExecutionMode,
        successHandler: ((RequestResponse?) -> Unit)?
    ): Runnable? =
        runCatching {
            if (crashSettings.isRateLimited) {
                Runnable { handleRateLimit(id, cacheExecMode) }
            } else {
                syncCrash(id, jsonObject, cacheExecMode, successHandler).get()
            }
        }.getOrReportError(null, "Error while syncing early crashes", true)

    private fun handleRateLimit(
        id: String,
        cacheExecMode: ExecutionMode
    ) {
        cacheHandler.delete(id, cacheExecMode)
        InstabugSDKLogger.d(
            Constants.LOG_TAG,
            String.format(RateLimitedException.RATE_LIMIT_REACHED, Constants.FEATURE_NAME)
        )
    }

    private fun handleRateLimitResponse(
        id: String,
        cacheExecMode: ExecutionMode,
        exception: RateLimitedException
    ) {
        crashSettings.setLimitedUntil(exception.period)
        handleRateLimit(id, cacheExecMode)
    }

    private fun syncCrash(
        id: String,
        jsonObject: JSONObject,
        cacheExecMode: ExecutionMode,
        successHandler: ((RequestResponse?) -> Unit)?
    ): Future<Runnable> {
        val completableFuture = SimpleCompletableFuture<Runnable>()
        requestFactory.create(jsonObject)?.let { request ->
            crashSettings.setLastRequestStartedAt(TimeUtils.currentTimeMillis())
            networkManager.doRequestOnSameThread(
                RequestType.NORMAL,
                request,
                true,
                object : Request.Callbacks<RequestResponse, Throwable> {
                    override fun onSucceeded(response: RequestResponse?) {
                        completableFuture.complete(Runnable {
                            onSyncSuccess(id, response, cacheExecMode)
                            successHandler?.invoke(response)
                        })
                    }

                    override fun onFailed(error: Throwable?) {
                        completableFuture.complete(Runnable {
                            onSyncFailure(id, error, cacheExecMode)
                        })
                    }

                }
            )
        }
        if (!completableFuture.isDone) {
            completableFuture.complete(Runnable { })
        }
        return completableFuture
    }

    private fun onSyncSuccess(
        crashId: String,
        response: RequestResponse?,
        cacheExecMode: ExecutionMode
    ) {
        cacheHandler.delete(crashId, cacheExecMode)
        crashSettings.setLastRequestStartedAt(0)
        "reporting EarlyCrash request Succeeded, Response code: ${response?.responseCode}".logVerbose()
        crashSettings.lastCrashTime = TimeUtils.currentTimeMillis()
    }

    private fun onSyncFailure(crashId: String, error: Throwable?, cacheExecMode: ExecutionMode) {
        if (error is RateLimitedException) {
            handleRateLimitResponse(crashId, cacheExecMode, error)
        } else {
            IBGDiagnostics.reportNonFatalAndLog(
                error,
                "Reporting early crash got an error ${error?.message}",
                Constants.LOG_TAG
            )
        }
    }

}

class ASynchronousSingleEarlyCrashUploader(
    private val synchronousUploader: SingleEarlyCrashUploader<Runnable?>,
    private val executor: ExecutorService
) : SingleEarlyCrashUploader<Future<Runnable?>> {
    override fun invoke(
        id: String,
        jsonObject: JSONObject,
        cacheExecMode: ExecutionMode,
        successHandler: ((RequestResponse?) -> Unit)?
    ): Future<Runnable?> = executor.submit(
        Callable {
            synchronousUploader(
                id,
                jsonObject,
                cacheExecMode,
                successHandler
            )
        }
    )

}