package com.instabug.crash.network;

import static com.instabug.commons.diagnostics.event.CalibrationDiagnosticEvent.Action;
import static com.instabug.library.internal.storage.cache.db.InstabugDbContract.AttachmentEntry.COLUMN_LOCALE_PATH;
import static com.instabug.library.internal.storage.cache.db.InstabugDbContract.CrashEntry.COLUMN_CRASH_STATE;
import static com.instabug.library.internal.storage.cache.db.InstabugDbContract.CrashEntry.COLUMN_TEMPORARY_SERVER_TOKEN;

import android.content.ContentValues;
import android.content.Context;
import android.net.Uri;

import androidx.annotation.NonNull;
import androidx.annotation.Nullable;
import androidx.annotation.VisibleForTesting;

import com.instabug.commons.di.CommonsLocator;
import com.instabug.commons.diagnostics.di.DiagnosticsLocator;
import com.instabug.commons.diagnostics.event.CalibrationDiagnosticEvent;
import com.instabug.commons.utils.RateLimiterUtilsKt;
import com.instabug.crash.Constants;
import com.instabug.crash.OnCrashSentCallback;
import com.instabug.crash.cache.CrashReportsDbHelper;
import com.instabug.crash.di.CrashesServiceLocator;
import com.instabug.crash.diagnostics.CrashIncidentType;
import com.instabug.crash.models.Crash;
import com.instabug.crash.models.CrashMetadata;
import com.instabug.crash.settings.CrashSettings;
import com.instabug.crash.utils.DeleteCrashUtilsKt;
import com.instabug.library.IBGNetworkWorker;
import com.instabug.library.Instabug;
import com.instabug.library.InstabugNetworkJob;
import com.instabug.library.internal.storage.AttachmentManager;
import com.instabug.library.internal.storage.AttachmentsUtility;
import com.instabug.library.internal.storage.cache.AttachmentsDbHelper;
import com.instabug.library.internal.video.InstabugVideoUtils;
import com.instabug.library.model.Attachment;
import com.instabug.library.networkv2.RateLimitedException;
import com.instabug.library.networkv2.request.Request;
import com.instabug.library.settings.SettingsManager;
import com.instabug.library.util.InstabugSDKLogger;

import org.json.JSONException;

import java.io.File;
import java.io.IOException;
import java.util.Calendar;
import java.util.List;
import java.util.Locale;


public class InstabugCrashesUploaderJob extends InstabugNetworkJob {
    private static final String TAG = "InstabugCrashesUploaderJob";
    @Nullable
    private static InstabugCrashesUploaderJob INSTANCE;

    private InstabugCrashesUploaderJob() {
    }

    public synchronized static InstabugCrashesUploaderJob getInstance() {
        if (INSTANCE == null) {
            INSTANCE = new InstabugCrashesUploaderJob();
        }

        return INSTANCE;
    }

    private static void trimScreenRecordsIfAny(@NonNull Context context) throws IOException {
        if (!SettingsManager.getInstance().isAutoScreenRecordingEnabled()) {
            return;
        }

        int maxDuration = SettingsManager.getInstance().autoScreenRecordingMaxDuration();
        List<String> crashesIdsList = CrashReportsDbHelper.retrieveIds();
        for (String crashId : crashesIdsList) {
            Crash crash = CrashReportsDbHelper.retrieveById(crashId, context);
            if (crash == null) {
                InstabugSDKLogger.e(Constants.LOG_TAG, "Something went wrong while retrieving crash " + crashId + " for screen records trimming");
                continue;
            }
            if (crash.getCrashState() == Crash.CrashState.WAITING_FOR_SCREEN_RECORDING_TO_BE_TRIMMED) {
                for (Attachment attachment : crash.getAttachments()) {
                    if (attachment.isEncrypted()) {
                        boolean isAttachmentDecrypted = AttachmentsUtility.decryptAttachmentAndUpdateDb(attachment);
                        attachment.setEncrypted(isAttachmentDecrypted);
                    }

                    if (attachment.getType() != null && attachment.getType().toString().equalsIgnoreCase(Attachment.Type.AUTO_SCREEN_RECORDING_VIDEO.toString())) {
                        if (attachment.getLocalPath() != null) {
                            File asrFile = new File(attachment.getLocalPath());
                            File trimmedFile = InstabugVideoUtils.startTrim(asrFile,
                                    AttachmentManager.getAutoScreenRecordingFile(context),
                                    maxDuration);

                            Uri uri = Uri.fromFile(trimmedFile);
                            if (uri.getLastPathSegment() != null) {
                                attachment.setName(uri.getLastPathSegment());
                            }
                            if (uri.getPath() != null) {
                                attachment.setLocalPath(uri.getPath());
                            }

                            crash.setCrashState(Crash.CrashState.READY_TO_BE_SENT);

                            //Update crash status record in the DB
                            ContentValues contentValues = new ContentValues();
                            contentValues.put(COLUMN_CRASH_STATE, Crash.CrashState.READY_TO_BE_SENT.name());
                            CrashReportsDbHelper.update(crashId, contentValues);

                            //Update the auto screen record attachment path with the trimmed medFile path in the DB
                            ContentValues values = new ContentValues();
                            values.put(COLUMN_LOCALE_PATH, trimmedFile.getPath());
                            AttachmentsDbHelper.update(attachment.getId(), values);
                            break;
                        }
                    }
                }
            }
        }
    }

    private static void uploadCrashes(@NonNull final Context context) throws JSONException {

        List<String> crashesIdsList = CrashReportsDbHelper.retrieveIds();
        InstabugSDKLogger.d(Constants.LOG_TAG, "Found " + crashesIdsList.size() +
                " crashes in cache");
        for (String crashId : crashesIdsList) {
            Crash crash = CrashReportsDbHelper.retrieveById(crashId, context);
            if (crash == null) {
                InstabugSDKLogger.e(Constants.LOG_TAG, "Something went wrong retrieving crash with id " + crashId);
                continue;
            }
            if (crash.getCrashState().equals(Crash.CrashState.READY_TO_BE_SENT)) {
                if (CrashSettings.getInstance().isRateLimited()) {
                    deleteCrash(context, crash);
                    logRateIsLimited();
                    continue;
                }
                if (isCrashTypeNotEnabled(crash)) {
                    deleteCrash(context, crash);
                    String handled = crash.isHandled() ? "handled" : "unhandled";
                    InstabugSDKLogger.d(Constants.LOG_TAG, "Cannot sync " + handled + " exception as the feature seems to be disabled");
                    continue;
                }
                CrashSettings.getInstance().setLastRequestStartedAt(System.currentTimeMillis());
                InstabugSDKLogger.d(Constants.LOG_TAG, "Uploading crash: " + crash.getId() + " is handled: " + crash.isHandled());
                CrashesService.getInstance().reportCrash(crash, new Request.Callbacks<String, Throwable>() {
                    @Override
                    public void onSucceeded(@Nullable String crashID) {
                        if (crashID == null) {
                            InstabugSDKLogger.v(Constants.LOG_TAG, "Crash uploading response was null, aborting...");
                            return;
                        }
                        if (RateLimiterUtilsKt.hasRateLimitedPrefix(crashID)) {
                            onCrashSentSetup(crash, crashID);
                            deleteCrash(context, crash);
                        } else {
                            onCrashSentSetup(crash, crashID);
                            //updating crash in db
                            ContentValues contentValues = new ContentValues();
                            contentValues.put(COLUMN_TEMPORARY_SERVER_TOKEN, crashID);
                            contentValues.put(COLUMN_CRASH_STATE, Crash.CrashState.LOGS_READY_TO_BE_UPLOADED.name());
                            String crashId = crash.getId();
                            CrashReportsDbHelper.update(crashId, contentValues);
                            uploadCrashLogs(crash, context);
                            updateLastCrashTime();
                        }
                    }

                    @Override
                    public void onFailed(Throwable throwable) {
                        if (throwable instanceof RateLimitedException)
                            handleRateLimitedException((RateLimitedException) throwable, crash, context);
                        else
                            InstabugSDKLogger.d(Constants.LOG_TAG, "Something went" + " wrong while uploading crash");

                    }
                });
            } else if (crash.getCrashState().equals(Crash.CrashState.LOGS_READY_TO_BE_UPLOADED)) {
                InstabugSDKLogger.v(Constants.LOG_TAG, "crash: " + crash.getId() + " already uploaded but has unsent logs, uploading now");
                uploadCrashLogs(crash, context);
            } else if (crash.getCrashState().equals(Crash.CrashState.ATTACHMENTS_READY_TO_BE_UPLOADED)) {
                InstabugSDKLogger.d(Constants.LOG_TAG, "crash: " + crash.getId() +
                        " already uploaded but has unsent attachments, uploading now");
                uploadAttachments(crash, context);
            }
        }
    }

    private static boolean isCrashTypeNotEnabled(Crash crash) {
        return crash.isHandled() &&
                !CrashesServiceLocator.getCrashConfigurationProvider()
                        .isNonFatalReportingEnabled();
    }

    private static void onCrashSent(Crash crash) {
        OnCrashSentCallback crashMetadataCallback = CommonsLocator.getCrashMetadataCallback();
        CrashMetadata crashMetaData = CommonsLocator.getCrashMetadataMapper().toMetadata(crash);
        crashMetadataCallback.onCrashSent(crashMetaData);
    }

    private static void handleRateLimitedException(RateLimitedException exception, @NonNull Crash crash, Context context) {
        CrashSettings.getInstance().setLimitedUntil(exception.getPeriod());
        logRateIsLimited();
        deleteCrash(context, crash);
    }

    @VisibleForTesting
    public static void deleteCrash(Context context, @NonNull Crash crash) {
        DeleteCrashUtilsKt.deleteCrash(context, crash);
    }

    private static void logRateIsLimited() {
        InstabugSDKLogger.d(Constants.LOG_TAG, String.format(RateLimitedException.RATE_LIMIT_REACHED, Constants.FEATURE_NAME));
    }

    private static void uploadCrashLogs(final Crash crash, @NonNull final Context context) {


        CrashesService.getInstance().uploadCrashLogs(crash, new Request.Callbacks<Boolean, Crash>() {

            @Override
            public void onSucceeded(@Nullable Boolean isSucceeded) {
                crash.setCrashState(Crash.CrashState.ATTACHMENTS_READY_TO_BE_UPLOADED);

                //updating crash in db
                ContentValues contentValues = new ContentValues();
                contentValues.put(COLUMN_CRASH_STATE, Crash.CrashState.ATTACHMENTS_READY_TO_BE_UPLOADED.name());
                String crashId = crash.getId();
                if (crashId != null)
                    CrashReportsDbHelper.update(crashId, contentValues);

                try {
                    uploadAttachments(crash, context);
                } catch (JSONException e) {
                }
            }

            @Override
            public void onFailed(Crash failedCrash) {

            }
        });
    }

    private static void uploadAttachments(final Crash crash, @NonNull final Context context) throws JSONException {

        InstabugSDKLogger.d(Constants.LOG_TAG, "Found " + crash.getAttachments().size() + " attachments related to crash");
        CrashesService.getInstance().uploadCrashAttachments(crash, new Request.Callbacks<Boolean, Crash>() {
            @Override
            public void onSucceeded(@Nullable Boolean isSucceeded) {
                InstabugSDKLogger.d(Constants.LOG_TAG, "Crash attachments uploaded successfully");
                Context context = Instabug.getApplicationContext();
                if (context != null) {
                    DeleteCrashUtilsKt.deleteCrash(context, crash);
                } else {
                    InstabugSDKLogger.v(Constants.LOG_TAG, "unable to delete state file for crash with id: " + crash.getId()
                            + "due to null context reference");
                }
                DiagnosticsLocator.getReporter().report(new CalibrationDiagnosticEvent(new CrashIncidentType(), Action.Synced));
                updateLastCrashTime();
            }

            @Override
            public void onFailed(Crash failedCrash) {
                InstabugSDKLogger.e(Constants.LOG_TAG, "Something went wrong while uploading crash attachments");
            }
        });
    }

    private static void updateLastCrashTime() {
        // initialize current calendar.
        Calendar calendar = Calendar.getInstance(Locale.ENGLISH);
        InstabugSDKLogger.v(Constants.LOG_TAG, "Updating last_crash_time to " + calendar.getTime());
        CrashSettings.getInstance().setLastCrashTime(calendar.getTime().getTime());
    }

    private static void onCrashSentSetup(Crash crash, String crashID) {
        CrashSettings.getInstance().setLastRequestStartedAt(0L);
        InstabugSDKLogger.d(com.instabug.crash.Constants.LOG_TAG, "crash uploaded successfully");
        crash.setTemporaryServerToken(crashID);
        crash.setCrashState(Crash.CrashState.LOGS_READY_TO_BE_UPLOADED);
        InstabugCrashesUploaderJob.onCrashSent(crash);
    }

    public void start() {
        enqueueJob(IBGNetworkWorker.CRASH, () -> {
            if (Instabug.getApplicationContext() != null) {
                try {
                    trimScreenRecordsIfAny(Instabug.getApplicationContext());
                    uploadCrashes(Instabug.getApplicationContext());
                } catch (Exception e) {
                    InstabugSDKLogger.e(Constants.LOG_TAG, "Error " + e.getMessage() + "occurred while uploading crashes", e);
                }
            } else {
                InstabugSDKLogger.d(Constants.LOG_TAG, "Context was null while uploading Crashes");
            }
        });
    }
}
