package com.instabug.crash.cache;

import static com.instabug.library.internal.storage.cache.db.InstabugDbContract.CrashEntry.COLUMN_CRASH_MESSAGE;
import static com.instabug.library.internal.storage.cache.db.InstabugDbContract.CrashEntry.COLUMN_CRASH_STATE;
import static com.instabug.library.internal.storage.cache.db.InstabugDbContract.CrashEntry.COLUMN_FINGERPRINT;
import static com.instabug.library.internal.storage.cache.db.InstabugDbContract.CrashEntry.COLUMN_HANDLED;
import static com.instabug.library.internal.storage.cache.db.InstabugDbContract.CrashEntry.COLUMN_ID;
import static com.instabug.library.internal.storage.cache.db.InstabugDbContract.CrashEntry.COLUMN_LEVEL;
import static com.instabug.library.internal.storage.cache.db.InstabugDbContract.CrashEntry.COLUMN_RETRY_COUNT;
import static com.instabug.library.internal.storage.cache.db.InstabugDbContract.CrashEntry.COLUMN_STATE;
import static com.instabug.library.internal.storage.cache.db.InstabugDbContract.CrashEntry.COLUMN_TEMPORARY_SERVER_TOKEN;
import static com.instabug.library.internal.storage.cache.db.InstabugDbContract.CrashEntry.COLUMN_THREADS_DETAILS;
import static com.instabug.library.internal.storage.cache.db.InstabugDbContract.CrashEntry.COLUMN_UUID;
import static com.instabug.library.internal.storage.cache.db.InstabugDbContract.CrashEntry.TABLE_NAME;

import android.content.ContentValues;
import android.content.Context;
import android.database.Cursor;
import android.net.Uri;

import androidx.annotation.NonNull;
import androidx.annotation.Nullable;
import androidx.annotation.WorkerThread;

import com.instabug.commons.models.IncidentMetadata;
import com.instabug.crash.Constants;
import com.instabug.crash.models.Crash;
import com.instabug.library.Instabug;
import com.instabug.library.core.InstabugCore;
import com.instabug.library.diagnostics.IBGDiagnostics;
import com.instabug.library.internal.storage.DiskUtils;
import com.instabug.library.internal.storage.cache.AttachmentsDbHelper;
import com.instabug.library.internal.storage.cache.db.DatabaseManager;
import com.instabug.library.internal.storage.cache.db.SQLiteDatabaseWrapper;
import com.instabug.library.internal.storage.operation.DeleteUriDiskOperation;
import com.instabug.library.model.Attachment;
import com.instabug.library.model.State;
import com.instabug.library.util.InstabugSDKLogger;

import java.io.File;
import java.io.IOException;
import java.util.ArrayList;
import java.util.List;

public class CrashReportsDbHelper {

    private static final int MAX_RETRY_ALLOWED = 3;
    private static final int MAX_CRASHES_ALLOWED = 100;

    public static synchronized int getCrashesCount() {
        InstabugSDKLogger.v(Constants.LOG_TAG, "getting Crashes Count");
        SQLiteDatabaseWrapper db = DatabaseManager.getInstance().openDatabase();
        try {
            return (int) db.queryNumEntries(TABLE_NAME);
        } catch (Exception e) {
            InstabugSDKLogger.e(Constants.LOG_TAG, "Error while getting crashes count: " + e.getMessage(), e);
            IBGDiagnostics.reportNonFatal(e, "Error while getting crashes count: " + e.getMessage());
        } finally {
            db.close();
        }
        return 0;
    }

    public static synchronized long trimAndInsert(Crash crash) {
        InstabugSDKLogger.d(Constants.LOG_TAG, "Inserting crash to DB");
        SQLiteDatabaseWrapper db = DatabaseManager.getInstance().openDatabase();
        try {
            trimByHandlingStatus(crash.isHandled(), MAX_CRASHES_ALLOWED - 1, db);
            // db will be closed inside insert method
            return insert(crash, db);
        } catch (Throwable throwable) {
            InstabugSDKLogger.e(Constants.LOG_TAG, "Error while inserting crash to DB ", throwable);
            IBGDiagnostics.reportNonFatal(throwable, "trimAndInsert crashes throwed an error: " + throwable.getMessage());
        }
        return -1;
    }

    /**
     * Inserting a crash report in the database
     *
     * @param crash instance of {@link Crash}
     * @return the row ID of the newly inserted row, or -1 if an error occurred
     */
    public static synchronized long insert(Crash crash, SQLiteDatabaseWrapper db) {
        // Gets the data repository in write mode
        db.beginTransaction();
        try {
            // Create a new map of values, where column names are the keys
            ContentValues values = new ContentValues();
            if (crash.getCrashMessage() != null) {
                values.put(COLUMN_CRASH_MESSAGE, crash.getCrashMessage());
            }
            values.put(COLUMN_CRASH_STATE, crash.getCrashState().name());
            values.put(COLUMN_HANDLED, crash.isHandled());
            if (crash.getState() != null && crash.getState().getUri() != null) {
                values.put(COLUMN_STATE, crash.getState().getUri().toString());
            }
            if (crash.getTemporaryServerToken() != null) {
                values.put(COLUMN_TEMPORARY_SERVER_TOKEN, crash.getTemporaryServerToken());
            }
            if (crash.getThreadsDetails() != null) {
                values.put(COLUMN_THREADS_DETAILS, crash.getThreadsDetails());
            }
            if (crash.getFingerprint() != null) {
                values.put(COLUMN_FINGERPRINT, crash.getFingerprint());
            }
            if (crash.getLevel() != null) {
                values.put(COLUMN_LEVEL, crash.getLevel().getSeverity());
            }
            if (crash.getId() != null) {
                values.put(COLUMN_ID, crash.getId());
                for (Attachment attachment : crash.getAttachments()) {
                    long rowId = AttachmentsDbHelper.insert(attachment, crash.getId());
                    attachment.setId(rowId);
                }
            }
            if (crash.getMetadata().getUuid() != null)
                values.put(COLUMN_UUID, crash.getMetadata().getUuid());

            // Insert the new row, returning the primary key value of the new row
            long rowId = db.insert(TABLE_NAME, null, values);
            db.setTransactionSuccessful();
            InstabugSDKLogger.d(Constants.LOG_TAG, "crash inserted to db successfully");
            return rowId;
        } catch (Exception e) {
            InstabugSDKLogger.e(Constants.LOG_TAG, "Error:" + e.getMessage() + "while inserting crash ");
            IBGDiagnostics.reportNonFatal(e, "Error while inserting crash" + e.getMessage());
            return -1;
        } finally {
            db.endTransaction();
            db.close();
        }
    }

    /**
     * Retrieves all crashes ids in crashes table ordered by ID ascending.
     *
     * @return A list of Strings representing crashes ids.
     */
    public synchronized static List<String> retrieveIds() {
        SQLiteDatabaseWrapper database = DatabaseManager.getInstance().openDatabase();
        Cursor cursor = null;
        ArrayList<String> idsList = new ArrayList<>();
        try {
            String[] columns = {COLUMN_ID};
            cursor = database.query(TABLE_NAME, columns, null, null, null, null, COLUMN_ID + " ASC");
            if (cursor == null) return idsList;
            while (cursor.moveToNext()) {
                idsList.add(cursor.getString(cursor.getColumnIndexOrThrow(COLUMN_ID)));
            }
        } catch (Throwable throwable) {
            IBGDiagnostics.reportNonFatalAndLog(throwable, "Error: " + throwable.getMessage() + " while retrieving crashes ids", Constants.LOG_TAG);
            return new ArrayList<>();
        } finally {
            if (cursor != null) cursor.close();
        }
        return idsList;
    }

    /**
     * Retrieves a crash by its id.
     *
     * @param id      the id of the crash to be retrieved
     * @param context A valid context to be used for retrieving crash attachments and reading state file
     * @return A {@link Crash} instance representing the crash with the given id or null if something went wrong while retrieving it.
     */
    @Nullable
    public synchronized static Crash retrieveById(@NonNull String id, @NonNull Context context) {
        SQLiteDatabaseWrapper database = DatabaseManager.getInstance().openDatabase();
        Crash crash;
        try {
            crash = retrieveCrashDetails(id, context, database);
            if (crash == null) return null;
            crash.setCrashMessage(new CachedCrashMessageLimiter().getCrashMessage(id, database));
            return crash;
        } catch (Throwable throwable) {
            IBGDiagnostics.reportNonFatalAndLog(throwable, "Error: " + throwable.getMessage() + " while retrieving latest crash", Constants.LOG_TAG);
            return null;
        } finally {
            database.close();
        }
    }

    @WorkerThread
    public static synchronized void trim() {
        InstabugSDKLogger.d(Constants.LOG_TAG, "Inserting crash to DB");
        SQLiteDatabaseWrapper db = DatabaseManager.getInstance().openDatabase();
        try {
            trimByHandlingStatus(true, MAX_CRASHES_ALLOWED, db);
            trimByHandlingStatus(false, MAX_CRASHES_ALLOWED, db);
        } catch (Throwable throwable) {
            InstabugSDKLogger.e(Constants.LOG_TAG, "Error " + throwable.getMessage() + " while trimming incidents.");
        } finally {
            db.close();
        }
    }

    private static void trimByHandlingStatus(boolean handled, int limit, SQLiteDatabaseWrapper db) throws Throwable {
        String[] columns = {COLUMN_ID, COLUMN_STATE};
        String[] selectionArgs = {String.valueOf(handled ? 1 : 0)};
        Cursor deletionCursor = null;
        // Database closure should be handled in caller.
        try {
            deletionCursor = db.query(
                    TABLE_NAME,
                    columns,
                    COLUMN_HANDLED + " = ?",
                    selectionArgs,
                    null,
                    null,
                    COLUMN_ID + " ASC",
                    null);
            if (deletionCursor == null) return;
            int incidentsCount = deletionCursor.getCount();
            if (incidentsCount < limit) return;
            int deletionCount = incidentsCount - limit;
            int deletionIndex = 0;
            while (deletionIndex < deletionCount && deletionCursor.moveToNext()) {
                String incidentId = deletionCursor.getString(deletionCursor.getColumnIndexOrThrow(COLUMN_ID));
                String stateUriString = deletionCursor.getString(deletionCursor.getColumnIndexOrThrow(COLUMN_STATE));
                List<Attachment> incidentAttachments = AttachmentsDbHelper.retrieve(incidentId, db);

                Uri stateUri = null;
                if (stateUriString != null)
                    stateUri = Uri.parse(stateUriString);
                if (stateUri != null)
                    deleteCrashStateUriFile(stateUri);
                deleteCrashAttachments(incidentAttachments, incidentId);
                delete(incidentId);
                deletionIndex++;
            }
        } finally {
            if (deletionCursor != null) deletionCursor.close();
        }
    }

    @Nullable
    private static Crash retrieveCrashDetails(@NonNull String id, @NonNull Context context, @NonNull SQLiteDatabaseWrapper database) throws Throwable {
        Cursor cursor = null;
        try {
            String[] columns = {COLUMN_ID, COLUMN_TEMPORARY_SERVER_TOKEN, COLUMN_CRASH_STATE, COLUMN_STATE, COLUMN_HANDLED, COLUMN_RETRY_COUNT, COLUMN_THREADS_DETAILS, COLUMN_FINGERPRINT, COLUMN_LEVEL, COLUMN_UUID};
            String[] selectionArgs = {id};
            cursor = database.query(TABLE_NAME, columns, COLUMN_ID + " = ?", selectionArgs, null, null, null, null);
            if (cursor == null || !cursor.moveToFirst()) return null;
            return createCrashFromCursor(cursor, database, context);
        } finally {
            if (cursor != null) cursor.close();
        }
    }

    @Nullable
    private static Crash createCrashFromCursor(@NonNull Cursor cursor, @NonNull SQLiteDatabaseWrapper database, @NonNull Context context) throws Throwable {
        String crashId = cursor.getString(cursor.getColumnIndexOrThrow(COLUMN_ID));
        if (crashId == null) {
            InstabugSDKLogger.e(Constants.LOG_TAG, "Crash id is null, couldn't create crash");
            return null;
        }
        String crashUUID = cursor.getString(cursor.getColumnIndexOrThrow(COLUMN_UUID));
        Crash crash = new Crash.Factory().create(crashId, IncidentMetadata.Factory.create(crashUUID));
        crash.setHandled(cursor.getInt(cursor.getColumnIndexOrThrow(COLUMN_HANDLED)) != 0);
        crash.setCrashState(Enum.valueOf(Crash.CrashState.class, cursor.getString(cursor.getColumnIndexOrThrow(COLUMN_CRASH_STATE))));
        crash.setTemporaryServerToken(cursor.getString(cursor.getColumnIndexOrThrow(COLUMN_TEMPORARY_SERVER_TOKEN)));
        crash.setThreadsDetails(cursor.getString(cursor.getColumnIndexOrThrow(COLUMN_THREADS_DETAILS)));
        crash.setFingerprint(cursor.getString(cursor.getColumnIndexOrThrow(COLUMN_FINGERPRINT)));
        if (!cursor.isNull(cursor.getColumnIndexOrThrow(COLUMN_LEVEL)))
            crash.setLevel(cursor.getInt(cursor.getColumnIndexOrThrow(COLUMN_LEVEL)));
        crash.setAttachments(AttachmentsDbHelper.retrieve(crashId, database));
        crash.setRetryCount(cursor.getInt(cursor.getColumnIndexOrThrow(COLUMN_RETRY_COUNT)));
        String uriString = cursor.getString(cursor.getColumnIndexOrThrow(COLUMN_STATE));
        Uri uri = null;
        if (uriString != null)
            uri = Uri.parse(uriString);
        return addStateToCrash(crash, uri, context);
    }

    @Nullable
    private static Crash addStateToCrash(@NonNull Crash crash, @Nullable Uri uri, @NonNull Context context) {
        int updatedTrialCount = crash.getRetryCount();
        try {
            updatedTrialCount++;
            crash.setState(State.getState(context, uri));
        } catch (OutOfMemoryError | Exception e) {
            InstabugCore.reportError(e, "retrieving crash state throwed an error");
            InstabugSDKLogger.e(Constants.LOG_TAG, "Retrieving crash state throws an exception: " + e.getMessage());
            if (updatedTrialCount >= MAX_RETRY_ALLOWED) {
                // Remove the State Uri file
                deleteCrashStateUriFile(uri);
                // Delete Crash Attachments
                if (crash.getId() != null) {
                    deleteCrashAttachments(crash);
                } else {
                    InstabugSDKLogger.e(Constants.LOG_TAG, "Couldn't delete crash attachments: crash id was null");
                    return null;
                }


                // Remove the crash from DB
                delete(crash.getId());
                return null;
            }
        }
        // Update the crash with the new trials number
        ContentValues contentValues = new ContentValues();
        contentValues.put(COLUMN_RETRY_COUNT, updatedTrialCount);
        if (crash.getId() != null) {
            CrashReportsDbHelper.update(crash.getId(), contentValues);
        }

        crash.setRetryCount(updatedTrialCount);
        return crash;
    }

    /**
     * Update a specific crash report that is already in the database
     *
     * @param id {@link String} the id of the crash report
     * @param cv {@link ContentValues} params of the values that needs to be updated
     */
    public static synchronized void update(String id, ContentValues cv) {
        InstabugSDKLogger.v(Constants.LOG_TAG, "Updating crash " + id);
        SQLiteDatabaseWrapper db = DatabaseManager.getInstance().openDatabase();
        String whereClause = COLUMN_ID + "=? ";
        String[] whereArgs = new String[]{id};
        db.beginTransaction();
        try {
            db.update(TABLE_NAME, cv, whereClause, whereArgs);
            db.setTransactionSuccessful();
        } finally {
            db.endTransaction();
            db.close();
        }
    }

    /**
     * Delete a specific crash report
     *
     * @param id {@link String} the id of the crash report
     */
    public static synchronized void delete(String id) {
        InstabugSDKLogger.v(Constants.LOG_TAG, "delete crash: " + id);
        SQLiteDatabaseWrapper db = DatabaseManager.getInstance().openDatabase();
        String whereClause = COLUMN_ID + "=? ";
        String[] whereArgs = new String[]{id};
        db.beginTransaction();
        try {
            db.delete(TABLE_NAME, whereClause, whereArgs);
            db.setTransactionSuccessful();
        } finally {
            db.endTransaction();
            db.close();
        }
    }

    @WorkerThread
    static void deleteCrashStateUriFile(Uri uri) {
        Context context = Instabug.getApplicationContext();
        if (context != null) {
            try {
                DiskUtils.with(context)
                        .deleteOperation(new DeleteUriDiskOperation(uri)).execute();
            } catch (IOException e) {
            }
        }
    }

    static synchronized void deleteCrashAttachments(Crash crash) {
        List<Attachment> attachments = crash.getAttachments();
        deleteCrashAttachments(attachments, crash.getId());
    }

    synchronized static void deleteCrashAttachments(List<Attachment> attachments, @Nullable String incidentId) {
        for (Attachment attachment : attachments) {
            if (attachment.getLocalPath() != null && attachment.getName() != null) {
                // Delete the attachment file
                File attachmentFile = new File(attachment.getLocalPath());
                attachmentFile.delete();
                // Delete the attachment instance from DB
                if (attachment.getId() != -1) {
                    AttachmentsDbHelper.delete(attachment.getId());
                } else {
                    if (incidentId != null) {
                        AttachmentsDbHelper.delete(attachment.getName(), incidentId);
                    } else {
                        InstabugSDKLogger.e(Constants.LOG_TAG, "Couldn't delete attachments: crash.getId() is null");
                    }
                }
            }
        }
    }

    /**
     * Delete all crash reports stored in the database
     * i.e. deleting the whole table
     */
    public static synchronized void deleteAll() {
        SQLiteDatabaseWrapper db = DatabaseManager.getInstance().openDatabase();
        db.beginTransaction();
        try {
            db.delete(TABLE_NAME, null, null);
            db.setTransactionSuccessful();
        } catch (Exception e) {
            IBGDiagnostics.reportNonFatalAndLog(e, "deleteAll crashes throwed an error: " + e.getMessage(), Constants.LOG_TAG);
        } finally {
            db.endTransaction();
            db.close();
        }
    }

    @NonNull
    public static List<String> getCrashesStateFiles() {
        ArrayList<String> crashesStateFiles = new ArrayList<>();
        if (Instabug.getApplicationContext() != null) {
            Cursor cursor = null;
            try {
                DatabaseManager databaseManager = DatabaseManager.getInstance();
                SQLiteDatabaseWrapper sqLiteDatabaseWrapper = databaseManager.openDatabase();
                cursor = sqLiteDatabaseWrapper.query(TABLE_NAME, new String[]{COLUMN_STATE},
                        null,
                        null,
                        null,
                        null,
                        null);
                if (cursor != null && cursor.moveToFirst()) {
                    do {
                        crashesStateFiles.add(cursor.getString(cursor.getColumnIndexOrThrow(COLUMN_STATE)));
                    } while (cursor.moveToNext());
                }
            } catch (Exception e) {
                InstabugSDKLogger.e(Constants.LOG_TAG, "Error while getting crash state files", e);
            } finally {
                if (cursor != null) {
                    cursor.close();
                }
            }
        }
        return crashesStateFiles;
    }
}
