package com.instabug.commons.threading

import android.os.Looper
import com.instabug.commons.di.CommonsLocator
import com.instabug.commons.logging.getOrReportError
import com.instabug.commons.logging.logVerbose
import org.json.JSONArray
import org.json.JSONException
import org.json.JSONObject

private const val STACK_TRACE_ELEMENT = "\t at %s\n"

val Thread.isMainThread
    get() = Looper.getMainLooper() != null && this === Looper.getMainLooper().thread

fun Thread.isCrashing(crashingThread: Thread?) = this === crashingThread

fun Thread.getFormattedStackTrace(
    limit: Int,
    shouldLog: Boolean = false,
    preElements: StringBuilder.() -> Unit = {}
): Pair<String, Int> = stackTrace.let { trace ->
    StringBuilder().apply {
        preElements()
        trace.run { if (limit >= 0) take(limit) else this.toList() }
            .forEach { element -> append(STACK_TRACE_ELEMENT.format(element)) }
    }.toString().let { formattedTrace ->
        val droppedFrames = limit.takeUnless { it < 0 }
            ?.let { trace.size.minus(it) }
            ?.takeUnless { it < 0 } ?: 0
        if (shouldLog) {
            "For thread $this: original frames' count = ${trace.size}, dropped frames' count = $droppedFrames".logVerbose()
            "For thread $this: latest original frame = ${trace.firstOrNull()}, oldest original frame = ${trace.lastOrNull()}".logVerbose()
        }
        Pair(formattedTrace, droppedFrames)
    }
}

val Thread.formattedData: JSONObject
    @Throws(JSONException::class) get() = JSONObject().apply {
        put("threadName", name)
        put("threadId", id)
        put("threadPriority", priority)
        put("threadState", state.toString())
        threadGroup?.formattedData?.let { groupData -> put("threadGroup", groupData) }
    }

fun Thread.formattedDataWithStackTrace(
    crashingThread: Thread?,
    framesLimit: Int = CommonsLocator.threadingLimitsProvider.provideFramesLimit(),
    shouldLog: Boolean = false
) = formattedData.apply {
    put("isMain", isMainThread)
    val isCrashing = isCrashing(crashingThread)
    getFormattedStackTrace(framesLimit, shouldLog || isCrashing).also { (trace, droppedFrames) ->
        put("stackTrace", trace)
        put("droppedFrames", droppedFrames)
    }
    put("isCrashing", isCrashing)
}


val ThreadGroup.formattedData
    @Throws(JSONException::class) get() = JSONObject().apply {
        put("name", name)
        put("maxPriority", maxPriority)
        put("activeCount", activeCount())
    }

fun Set<Thread>.toFormattedData(
    crashingThread: Thread?,
    framesLimit: Int = CommonsLocator.threadingLimitsProvider.provideFramesLimit()
): JSONArray = runCatching {
    asSequence()
        .mapIndexed { index, thread ->
            thread.formattedDataWithStackTrace(crashingThread, framesLimit, index == 0)
        }.map { threadData -> JSONObject().put("thread", threadData) }
        .fold(JSONArray()) { threadsList, threadObject -> threadsList.put(threadObject) }
}.getOrReportError(JSONArray(), "Failed parsing threads data")