package com.instabug.commons.snapshot

import com.instabug.commons.logging.getOrReportError
import org.json.JSONObject
import java.io.File
import java.io.ObjectInputStream
import java.io.ObjectOutputStream
import java.io.Serializable
import java.nio.charset.StandardCharsets

fun File.writeSerializable(savable: Serializable) {
    runCatching { ObjectOutputStream(outputStream()).use { stream -> stream.writeObject(savable) } }
}

fun File.writeTextSafely(text: String) {
    kotlin.runCatching {
        writeText(text, StandardCharsets.UTF_8)
    }
}

inline fun <reified T> File.readSerializable(): T? =
    runCatching { ObjectInputStream(inputStream()).use { stream -> stream.readObject() as? T } }
        .getOrReportError(null, "Error while reading serialized file.")

fun File.readSerializableAsAny(): Any? =
    runCatching { ObjectInputStream(inputStream()).use { stream -> stream.readObject() } }
        .getOrReportError(null, "Error while reading serialized file.")

fun File.readJsonObject(): JSONObject? =
    runCatching {
        readText(StandardCharsets.UTF_8).let(::JSONObject)
    }.getOrReportError(null, "Error while reading text file.", true)

fun File.writeString(string: String) =
    runCatching { writeText(string, StandardCharsets.UTF_8) }
        .getOrReportError(null, "Error while writing text file.")

inline fun File.ifNotExists(block: File.() -> Unit): File = run {
    takeIf { it.exists() } ?: block(); this
}

fun File.rename(newName: String) =
    parentFile?.let { parent -> renameTo(File("${parent.absolutePath}${File.separator}$newName")) }
