package com.instabug.bganr

import androidx.annotation.VisibleForTesting
import com.instabug.anr.configuration.AnrConfigurationProvider
import com.instabug.commons.configurations.ConfigurationsHandler
import com.instabug.commons.preferences.crashPref
import com.instabug.crash.Constants
import com.instabug.crash.configurations.CrashConfigurationProvider
import com.instabug.library.Instabug
import com.instabug.library.util.extenstions.runOrLogError
import org.json.JSONObject

private const val KEY_CRASHES = "crashes"
private const val KEY_BG_ANR = "bg_anr"

@VisibleForTesting
const val PARSING_ERROR_MESSAGE =
    "Something went wrong while parsing BG ANRs configurations from features response"

internal const val DEFAULT_BG_ANR_AVAILABILITY = true

class BackgroundAnrConfigurationsHandler(
    private val provider: IBackgroundAnrConfigurationsProvider
) : ConfigurationsHandler {
    override fun handleConfiguration(configuration: String?) {
        runOrLogError(tag = Constants.LOG_TAG, errorMessage = PARSING_ERROR_MESSAGE) {
            configuration?.let(::JSONObject)
                ?.optJSONObject(KEY_CRASHES)
                ?.takeIf { crashesObject -> crashesObject.has(KEY_BG_ANR) }
                ?.optBoolean(KEY_BG_ANR)
                ?.let { availability -> provider.isAvailable = availability }
        }
    }

    override fun migrateCurrentConfiguration() {
        // No-Op
    }
}

interface IBackgroundAnrConfigurationsProvider {
    /**
     * Whether background ANRs feature is enabled or not
     */
    val isEnabled: Boolean

    /**
     * Whether background ANRs feature is available BE side or not
     */
    var isAvailable: Boolean
    val isAnrV2Enabled: Boolean
}

@VisibleForTesting
val PREF_BG_ANR_AVAILABILITY = "bg_anr_availability" to DEFAULT_BG_ANR_AVAILABILITY

class BackgroundAnrConfigurationsProvider(
    private val crashesConfigurationsProvider: CrashConfigurationProvider,
    private val anrConfigurationsProvider: AnrConfigurationProvider
) : IBackgroundAnrConfigurationsProvider {
    override var isAvailable by crashPref(PREF_BG_ANR_AVAILABILITY)

    // Background ANRs state depends on the state of Instabug, CrashReporting and ANRs
    override val isEnabled: Boolean
        get() = Instabug.isEnabled() &&
                crashesConfigurationsProvider.isCrashReportingEnabled() &&
                anrConfigurationsProvider.isAnrEnabled() &&
                isAvailable
    override val isAnrV2Enabled: Boolean
        get() = anrConfigurationsProvider.isAnrV2Enabled
}
