package com.instabug.fatalhangs.sync

import android.annotation.SuppressLint
import androidx.annotation.VisibleForTesting
import com.instabug.commons.di.CommonsLocator.configurationsProvider
import com.instabug.fatalhangs.model.FatalHang
import com.instabug.library.diagnostics.IBGDiagnostics
import com.instabug.library.featuresflags.EnhancementRequestBodyParams
import com.instabug.library.featuresflags.di.FeaturesFlagServiceLocator.featuresFlagsConfigsProvider
import com.instabug.library.model.Attachment
import com.instabug.library.networkv2.request.Endpoints
import com.instabug.library.networkv2.request.FileToUpload
import com.instabug.library.networkv2.request.Header
import com.instabug.library.networkv2.request.Request
import com.instabug.library.networkv2.request.RequestMethod
import com.instabug.library.networkv2.request.RequestParameter
import com.instabug.library.networkv2.request.RequestType
import com.instabug.library.networkv2.request.getTokenFromState

object FatalHangsRequestsBuilder {

    private const val PARAM_TITLE = "title"
    private const val PARAM_THREADS_DETAILS = "threads_details"
    private const val PARAM_KEY_ID = "id"

    @VisibleForTesting
    const val PARAM_ACTIVITY_NAME = "activity_name"
    private const val PARAM_ATTACHMENTS_COUNT = "attachments_count"
    private const val PARAM_FILE_TYPE = "metadata[file_type]"
    private const val PARAM_DURATION = "metadata[duration]"
    private const val PARAM_REPORTED_AT = "reported_at"

    @SuppressLint("WrongConstant")
    fun buildFatalHangRequest(fatalHang: FatalHang): Request {
        val requestBuilder = Request.Builder()
            .endpoint(Endpoints.REPORT_FATAL_HANG)
            .method(RequestMethod.POST)
            .getTokenFromState(fatalHang.state)

        fatalHang.metadata.uuid?.let { uuid ->
            requestBuilder.addHeader(RequestParameter(Header.ID, uuid))
        }
        requestBuilder.addStateItems(fatalHang)
        updateReportedAtIfNeeded(requestBuilder, fatalHang)
        requestBuilder.addParameter(RequestParameter(PARAM_TITLE, fatalHang.mainThreadData))
        requestBuilder
            .addParameter(RequestParameter(PARAM_THREADS_DETAILS, fatalHang.restOfThreadsData))
        fatalHang.metadata.uuid?.let { uuid ->
            requestBuilder.addParameter(RequestParameter(PARAM_KEY_ID, uuid))
        }
        if (fatalHang.attachments.size > 0) {
            requestBuilder
                .addParameter(RequestParameter(PARAM_ATTACHMENTS_COUNT, fatalHang.attachments.size))
        }
        return requestBuilder.build()
    }

    private fun Request.Builder.addStateItems(fatalHang: FatalHang) {
        addParameter(RequestParameter(PARAM_ACTIVITY_NAME, fatalHang.lastActivity))
        val userIdentificationEnabled = configurationsProvider.userIdentificationEnabled
        val enhancementRequestBodyParams = EnhancementRequestBodyParams().getModifiedStateItemsList(
            fatalHang.state?.getStateItems(userIdentificationEnabled),
            featuresFlagsConfigsProvider.mode
        )
        enhancementRequestBodyParams
            .takeIf { it.isNotEmpty() }
            ?.asSequence()
            ?.filter { it.key != PARAM_ACTIVITY_NAME }
            ?.forEach { (key, value) -> addParameter(RequestParameter(key, value)) }
    }

    private fun updateReportedAtIfNeeded(builder: Request.Builder, fatalHang: FatalHang) {
        fatalHang.state?.run { if (!isMinimalState && reportedAt != 0L) return }
        runCatching {
            //FH id is basically a UTC time in milliseconds. So, it can be used as the report time.
            fatalHang.id?.let { it.toLong() }
                ?.let { reportedAt -> RequestParameter(PARAM_REPORTED_AT, reportedAt) }
                ?.let { requestParam -> builder.addParameter(requestParam) }
        }.getOrElse { throwable ->
            IBGDiagnostics.reportNonFatal(
                throwable,
                "Failed to update reported_at in fatal hang reporting request."
            )
        }
    }

    fun buildFatalHangLogsRequest(fatalHang: FatalHang): Request {
        val requestBuilder = Request.Builder()
            .endpoint(fatalHang.temporaryServerToken?.let {
                Endpoints.CRASH_LOGS.replace(
                    ":crash_token".toRegex(),
                    it
                )
            })
            .method(RequestMethod.POST)
            .getTokenFromState(fatalHang.state)

        fatalHang.state?.let {
            val logsItems = it.logsItems
            if (logsItems != null && logsItems.size > 0) {
                for (logItem in logsItems) {
                    if (logItem.key != null) {
                        requestBuilder.addParameter(
                            RequestParameter(
                                logItem.key,
                                if (logItem.value != null) logItem.value else ""
                            )
                        )
                    }
                }
            }
        }
        return requestBuilder.build()
    }

    fun buildSingleAttachmentRequest(fatalHang: FatalHang, attachment: Attachment): Request? {
        fatalHang.temporaryServerToken?.let { temporaryServerToken ->
            val requestBuilder = Request.Builder()
                .endpoint(
                    Endpoints.ADD_CRASH_ATTACHMENT.replace(
                        ":crash_token".toRegex(),
                        temporaryServerToken
                    )
                )
                .method(RequestMethod.POST)
                .type(RequestType.MULTI_PART)
                .getTokenFromState(fatalHang.state)

            if (attachment.type != null)
                requestBuilder.addParameter(RequestParameter(PARAM_FILE_TYPE, attachment.type))
            if (attachment.type == Attachment.Type.AUDIO && attachment.duration != null) {
                requestBuilder.addParameter(RequestParameter(PARAM_DURATION, attachment.duration))
            }
            val name = attachment.name
            val localPath = attachment.localPath
            if (name != null && localPath != null) {
                requestBuilder
                    .fileToUpload(FileToUpload("file", name, localPath, attachment.fileType))
            }
            return requestBuilder.build()
        }
        return null
    }
}