package com.instabug.anr.cache;

import static com.instabug.library.internal.storage.cache.db.InstabugDbContract.AnrEntry.COLUMN_ANR_LONG_MESSAGE;
import static com.instabug.library.internal.storage.cache.db.InstabugDbContract.AnrEntry.COLUMN_ANR_MAIN_THREAD_DATA;
import static com.instabug.library.internal.storage.cache.db.InstabugDbContract.AnrEntry.COLUMN_ANR_REST_OF_THREADS_DATA;
import static com.instabug.library.internal.storage.cache.db.InstabugDbContract.AnrEntry.COLUMN_ANR_TEMPORARY_SERVER_TOKEN;
import static com.instabug.library.internal.storage.cache.db.InstabugDbContract.AnrEntry.COLUMN_ANR_UPLOAD_STATE;
import static com.instabug.library.internal.storage.cache.db.InstabugDbContract.AnrEntry.COLUMN_ANR_VERSION;
import static com.instabug.library.internal.storage.cache.db.InstabugDbContract.AnrEntry.COLUMN_EARLY_ANR;
import static com.instabug.library.internal.storage.cache.db.InstabugDbContract.AnrEntry.COLUMN_ID;
import static com.instabug.library.internal.storage.cache.db.InstabugDbContract.AnrEntry.COLUMN_STATE;
import static com.instabug.library.internal.storage.cache.db.InstabugDbContract.AnrEntry.COLUMN_UUID;
import static com.instabug.library.internal.storage.cache.db.InstabugDbContract.AnrEntry.TABLE_NAME;

import android.content.ContentValues;
import android.content.Context;
import android.database.Cursor;
import android.net.Uri;

import androidx.annotation.NonNull;

import com.instabug.anr.model.Anr;
import com.instabug.commons.models.IncidentMetadata;
import com.instabug.crash.Constants;
import com.instabug.library.Instabug;
import com.instabug.library.core.InstabugCore;
import com.instabug.library.diagnostics.IBGDiagnostics;
import com.instabug.library.internal.storage.cache.AttachmentsDbHelper;
import com.instabug.library.internal.storage.cache.db.DatabaseManager;
import com.instabug.library.internal.storage.cache.db.InstabugDbContract;
import com.instabug.library.internal.storage.cache.db.SQLiteDatabaseWrapper;
import com.instabug.library.model.Attachment;
import com.instabug.library.model.State;
import com.instabug.library.util.InstabugSDKLogger;

import org.json.JSONException;

import java.util.ArrayList;
import java.util.List;

public class AnrReportsDbHelper {


    /**
     * Inserting an ANR report in the database
     *
     * @param anr instance of {@link Anr}
     * @return the row ID of the newly inserted row, or -1 if an error occurred
     */
    public static synchronized long insert(Anr anr) {
        SQLiteDatabaseWrapper db = DatabaseManager.getInstance().openDatabase();
        // Gets the data repository in write mode
        db.beginTransaction();
        try {
            // Create a new map of values, where column names are the keys
            ContentValues values = new ContentValues();
            values.put(COLUMN_ANR_MAIN_THREAD_DATA, anr.getMainThreadData());
            values.put(COLUMN_ANR_REST_OF_THREADS_DATA, anr.getRestOfThreadsData());
            values.put(COLUMN_ANR_UPLOAD_STATE, anr.getAnrState());
            if (anr.getState() != null && anr.getState().getUri() != null) {
                values.put(COLUMN_STATE, anr.getState().getUri().toString());
            }
            values.put(COLUMN_ID, anr.getId());
            values.put(COLUMN_ANR_LONG_MESSAGE, anr.getLongMessage());
            values.put(COLUMN_ANR_VERSION, anr.getAnrVersion());
            values.put(COLUMN_EARLY_ANR, anr.isEarlyAnr());
            if (anr.getMetadata().getUuid() != null)
                values.put(COLUMN_UUID, anr.getMetadata().getUuid());

            // Insert the attachments
            for (Attachment attachment : anr.getAttachments()) {
                long rowId = AttachmentsDbHelper.insert(attachment, anr.getId());
                if (rowId != -1) {
                    attachment.setId(rowId);
                }
            }

            // Insert the new row, returning the primary key value of the new row
            long rowId = db.insert(TABLE_NAME, null, values);
            db.setTransactionSuccessful();
            return rowId;
        } finally {
            db.endTransaction();
            db.close();
        }
    }

    /**
     * Retrieve all ANR reports stored in the database ordered by ID (Ascending)
     * which represents the time the ANR was created.
     *
     * @return a list of {@link Anr}
     * @throws JSONException
     */
    public synchronized static List<Anr> retrieve(Context context) {
        SQLiteDatabaseWrapper db = DatabaseManager.getInstance().openDatabase();
        List<Anr> anrs = new ArrayList<>();
        Cursor c = null;
        try {
            c = db.query(
                    InstabugDbContract.AnrEntry.TABLE_NAME,
                    null,
                    null,
                    null,
                    null,
                    null,
                    InstabugDbContract.AnrEntry.COLUMN_ID + " ASC",
                    null
            );

            if (c != null && c.moveToFirst()) {
                do {
                    int idColIndex = c.getColumnIndex(COLUMN_ID);
                    if (c.getString(idColIndex) == null) continue;
                    int uuidColIndex = c.getColumnIndex(COLUMN_UUID);
                    Anr anr = new Anr(
                            c.getString(idColIndex),
                            IncidentMetadata.Factory.create(c.getString(uuidColIndex))
                    );

                    int anrMainThreadData = c.getColumnIndex(COLUMN_ANR_MAIN_THREAD_DATA);
                    anr.setMainThreadData(c.getString(anrMainThreadData));

                    int anrRestOfThreadsData = c.getColumnIndex(COLUMN_ANR_REST_OF_THREADS_DATA);
                    anr.setRestOfThreadsData(c.getString(anrRestOfThreadsData));

                    int anrUploadStateData = c.getColumnIndex(COLUMN_ANR_UPLOAD_STATE);
                    anr.setAnrState(c.getInt(anrUploadStateData));

                    int anrTempServerTokenData = c.getColumnIndex(COLUMN_ANR_TEMPORARY_SERVER_TOKEN);
                    anr.setTemporaryServerToken(c.getString(anrTempServerTokenData));

                    int anrLongMessage = c.getColumnIndex(COLUMN_ANR_LONG_MESSAGE);
                    anr.setLongMessage(c.getString(anrLongMessage));

                    int anrVersionIndex = c.getColumnIndex(COLUMN_ANR_VERSION);
                    anr.setAnrVersion(c.getString(anrVersionIndex));

                    int isEarlyAnrIndex = c.getColumnIndex(COLUMN_EARLY_ANR);
                    anr.setEarlyAnr(c.getInt(isEarlyAnrIndex) == 1);

                    anr.setAttachments(AttachmentsDbHelper.retrieve(anr.getId(), db));

                    int stateColIndex = c.getColumnIndex(InstabugDbContract.AnrEntry.COLUMN_STATE);
                    String uriString = c.getString(stateColIndex);
                    Uri uri = null;
                    if (uriString != null)
                        uri = Uri.parse(uriString);
                    try {
                        anr.setState(State.getState(context, uri));
                    } catch (OutOfMemoryError e) {
                        InstabugCore.reportError(e, "Retrieving ANR state throws OOM");
                        InstabugSDKLogger.e(Constants.LOG_TAG, "Retrieving ANR state throws OOM", e);
                    }

                    anrs.add(anr);
                } while (c.moveToNext());
            }
        } catch (Exception e) {
            InstabugSDKLogger.e(Constants.LOG_TAG, "Retrieve ANRs failed: ", e);
            IBGDiagnostics.reportNonFatal(e, " retrieve ANRs failed: " + e.getMessage());
        } finally {
            if (c != null) {
                c.close();
            }
            db.close();
        }
        return anrs;
    }

    public static int getAnrsCount() {
        SQLiteDatabaseWrapper db = DatabaseManager.getInstance().openDatabase();
        int noOfANRs = (int) db.queryNumEntries(TABLE_NAME);
        db.close();
        return noOfANRs;
    }

    public static void update(String anrId, ContentValues contentValues) {
        SQLiteDatabaseWrapper db = DatabaseManager.getInstance().openDatabase();
        String whereClause = InstabugDbContract.AnrEntry.COLUMN_ID + "=? ";
        String[] whereArgs = new String[]{anrId};
        db.beginTransaction();
        try {
            db.update(InstabugDbContract.AnrEntry.TABLE_NAME, contentValues, whereClause, whereArgs);
            db.setTransactionSuccessful();
        } finally {
            db.endTransaction();
            db.close();
        }
    }

    public static void delete(String anrId) {
        SQLiteDatabaseWrapper db = DatabaseManager.getInstance().openDatabase();
        String whereClause = InstabugDbContract.AnrEntry.COLUMN_ID + "=? ";
        String[] whereArgs = new String[]{anrId};
        db.beginTransaction();
        try {
            db.delete(InstabugDbContract.AnrEntry.TABLE_NAME, whereClause, whereArgs);
            db.setTransactionSuccessful();
        } finally {
            db.endTransaction();
            db.close();
        }
    }

    @NonNull
    public static List<String> getANRsStateFiles() {
        ArrayList<String> anrsStateFiles = new ArrayList<>();
        if (Instabug.getApplicationContext() != null) {
            Cursor cursor = null;
            try {
                DatabaseManager databaseManager = DatabaseManager.getInstance();
                SQLiteDatabaseWrapper sqLiteDatabaseWrapper = databaseManager.openDatabase();
                cursor = sqLiteDatabaseWrapper.query(InstabugDbContract.AnrEntry.TABLE_NAME, new String[]{InstabugDbContract.AnrEntry.COLUMN_STATE},
                        null,
                        null,
                        null,
                        null,
                        null);
                if (cursor != null && cursor.moveToFirst()) {
                    do {
                        anrsStateFiles.add(cursor.getString(cursor.getColumnIndexOrThrow(InstabugDbContract.CrashEntry.COLUMN_STATE)));
                    } while (cursor.moveToNext());
                }
            } catch (Exception e) {
                InstabugSDKLogger.e(Constants.LOG_TAG, "Error while getting ANRs state files");
            } finally {
                if (cursor != null) {
                    cursor.close();
                }
            }
        }
        return anrsStateFiles;
    }
}
