package com.instabug.early_crash

import android.content.Context
import com.instabug.commons.IUncaughtExceptionHandlingDelegate
import com.instabug.commons.logging.logError
import com.instabug.commons.threading.CrashDetailsParser
import com.instabug.crash.Constants
import com.instabug.early_crash.caching.IEarlyCrashCacheHandler
import com.instabug.early_crash.configurations.IEarlyCrashesConfigProvider
import com.instabug.early_crash.di.EarlyCrashesServiceLocator
import com.instabug.early_crash.model.EarlyCrash
import com.instabug.early_crash.network.SingleEarlyCrashUploader
import com.instabug.early_crash.threading.ExecutionMode
import com.instabug.library.diagnostics.IBGDiagnostics
import com.instabug.library.map.Mapper
import com.instabug.library.model.State
import com.instabug.library.util.TimeUtils
import org.json.JSONObject
import java.util.concurrent.ExecutionException
import java.util.concurrent.Future
import java.util.concurrent.TimeUnit

class EarlyUncaughtExceptionHandlingDelegate(
    private val cacheHandler: IEarlyCrashCacheHandler,
    private val mapper: Mapper<EarlyCrash, JSONObject>,
    private val configurationProvider: IEarlyCrashesConfigProvider,
    private val uploader: SingleEarlyCrashUploader<Future<Runnable?>>,
    private val uploadTimeOutSeconds: Long = 3
) : IUncaughtExceptionHandlingDelegate {
    override fun invoke(parser: CrashDetailsParser, ctx: Context?) {
        val (id, crashJson) = EarlyCrash(
            id = TimeUtils.currentTimeMillis(),
            message = parser.crashDetails.toString(),
            threadsDetails = parser.threadsDetails.toString(),
            state = State.Builder(ctx).buildEarlyState()
        ).let { it.id to mapper.map(it) }
        val crashJsonString = crashJson.toString()
        val uploadFuture = uploadCrashIfPossible(id, crashJson)
        cacheHandler.save(id, JSONObject(crashJsonString))
        runCatching { uploadFuture?.get(uploadTimeOutSeconds, TimeUnit.SECONDS)?.run() }
            .onFailure {
                when (it) {
                    is ExecutionException ->
                        IBGDiagnostics.reportNonFatalAndLog(
                            it,
                            "Failed to sync most recent early crash",
                            Constants.LOG_TAG
                        )

                    else -> "Failed to sync most recent early crash".logError(it)
                }
            }
    }

    private fun uploadCrashIfPossible(
        id: Long,
        crashJson: JSONObject
    ) =
        takeIf { configurationProvider.immediateSyncEnabled }
            ?.let {
                uploader.invoke(
                    id.toString(),
                    crashJson,
                    ExecutionMode.Immediate,
                    null
                )
            }

    class Factory {
        fun buildWithDefaults(): EarlyUncaughtExceptionHandlingDelegate =
            EarlyUncaughtExceptionHandlingDelegate(
                cacheHandler = EarlyCrashesServiceLocator.cacheHandler,
                mapper = EarlyCrashesServiceLocator.earlyCrashToJsonMapper,
                configurationProvider = EarlyCrashesServiceLocator.configurationsProvider,
                uploader = EarlyCrashesServiceLocator.asynchronousSingleCrashUploader
            )
    }
}