package com.instabug.anr.network;

import static com.instabug.crash.utils.DeleteCrashUtilsKt.deleteAttachment;
import static com.instabug.library.networkv2.request.RequestExtKt.getTokenFromState;

import androidx.annotation.NonNull;
import androidx.annotation.VisibleForTesting;

import com.instabug.anr.model.Anr;
import com.instabug.commons.di.CommonsLocator;
import com.instabug.crash.Constants;
import com.instabug.library.Instabug;
import com.instabug.library.diagnostics.IBGDiagnostics;
import com.instabug.library.featuresflags.EnhancementRequestBodyParams;
import com.instabug.library.featuresflags.di.FeaturesFlagServiceLocator;
import com.instabug.library.internal.storage.AttachmentsUtility;
import com.instabug.library.model.Attachment;
import com.instabug.library.model.State;
import com.instabug.library.networkv2.NetworkManager;
import com.instabug.library.networkv2.RateLimitedException;
import com.instabug.library.networkv2.RequestResponse;
import com.instabug.library.networkv2.request.Endpoints;
import com.instabug.library.networkv2.request.FileToUpload;
import com.instabug.library.networkv2.request.Header;
import com.instabug.library.networkv2.request.Request;
import com.instabug.library.networkv2.request.RequestMethod;
import com.instabug.library.networkv2.request.RequestParameter;
import com.instabug.library.networkv2.request.RequestType;
import com.instabug.library.util.InstabugSDKLogger;

import org.json.JSONException;
import org.json.JSONObject;

import java.io.File;
import java.util.ArrayList;
import java.util.List;
import java.util.Map;

public class AnrsService {

    @VisibleForTesting
    public static final String PARAM_TITLE = "title";
    @VisibleForTesting
    public static final String PARAM_THREADS_DETAILS = "threads_details";
    private static final String PARAM_ATTACHMENTS_COUNT = "attachments_count";
    private static final String PARAM_MESSAGE = "ANR_message";
    private static final String PARAM_ID = "id";
    private static final String PARAM_FILE_TYPE = "metadata[file_type]";
    private static final String PARAM_DURATION = "metadata[duration]";
    private static final String PARAM_REPORTED_AT = "reported_at";
    private static final String PARAM_ANR_VERSION = "anr_version";
    private static final String PARAM_IS_EARLY_ANR = "early_anr";

    private static final String TAG = "AnrsService";
    private static AnrsService INSTANCE;
    private NetworkManager networkManager;

    public AnrsService() {
        networkManager = new NetworkManager();
    }

    public static synchronized AnrsService getInstance() {
        if (INSTANCE == null) {
            INSTANCE = new AnrsService();
        }
        return INSTANCE;
    }

    public void reportAnr(final Anr anr,
                          final Request.Callbacks<String, Throwable> reportingAnrCallbacks) throws JSONException {
        // Build ANR request
        Request reportingAnrRequest = buildAnrReportingRequest(anr);
        // do request with NORMAL request type.
        networkManager.doRequestOnSameThread(RequestType.NORMAL, reportingAnrRequest, new Request.Callbacks<RequestResponse, Throwable>() {
            @Override
            public void onSucceeded(RequestResponse requestResponse) {
                InstabugSDKLogger.d(Constants.LOG_TAG, "ReportingAnrRequest Succeeded, Response code: " + requestResponse.getResponseCode());
                InstabugSDKLogger.v(Constants.LOG_TAG, "ReportingAnrRequest Succeeded, Response body: " + requestResponse.getResponseBody());
                try {
                    if (requestResponse.getResponseBody() != null) {
                        JSONObject response = new JSONObject((String) requestResponse.getResponseBody());
                        reportingAnrCallbacks.onSucceeded(response.getString("id"));
                    } else {
                        reportingAnrCallbacks.onFailed(
                                new JSONException("requestResponse.getResponseBody() returned null"));
                    }
                } catch (JSONException e) {
                    InstabugSDKLogger.e(Constants.LOG_TAG, "Couldn't parse Anr request response.", e);
                }
            }

            @Override
            public void onFailed(Throwable error) {
                if (error instanceof RateLimitedException) {
                    reportingAnrCallbacks.onFailed(error);
                    return;
                }
                InstabugSDKLogger.e(Constants.LOG_TAG, "ReportingAnrRequest got error: ", error);
                InstabugSDKLogger.e(TAG, "ReportingAnrRequest got error: ", error);
                AttachmentsUtility.encryptAttachmentsAndUpdateDb(anr.getAttachments());
                reportingAnrCallbacks.onFailed(error);
            }
        });
    }

    @NonNull
    @VisibleForTesting
    public Request buildAnrReportingRequest(Anr anr) {

        String appToken = Instabug.getAppToken();
        Request.Builder requestBuilder = new Request.Builder()
                .endpoint(Endpoints.REPORT_ANR)
                .method(RequestMethod.POST)
                .addHeader(new RequestParameter<>(Header.APP_TOKEN, appToken != null ? appToken : ""));

        getTokenFromState(requestBuilder, anr.getState());


        if (anr.getMetadata().getUuid() != null) {
            requestBuilder.addHeader(new RequestParameter<>(Header.ID, anr.getMetadata().getUuid()));
        }

        if (anr.getState() != null) {
            boolean userIdentificationEnabled = CommonsLocator.getConfigurationsProvider().getUserIdentificationEnabled();
            EnhancementRequestBodyParams enhancementRequestBodyParams = new EnhancementRequestBodyParams();
            Map<String, Object> result = enhancementRequestBodyParams
                    .getModifiedStateItemsList(anr.getState().getStateItems(userIdentificationEnabled), FeaturesFlagServiceLocator.getFeaturesFlagsConfigsProvider().getMode());
            for (Map.Entry<String, Object> entry : result.entrySet()) {
                requestBuilder.addParameter(new RequestParameter(entry.getKey(), entry.getValue()));
            }
        }
        updateReportedAtIfNeeded(requestBuilder, anr);
        requestBuilder.addParameter(new RequestParameter<>(PARAM_TITLE, anr.getMainThreadData()));
        requestBuilder.addParameter(new RequestParameter<>(PARAM_THREADS_DETAILS, anr.getRestOfThreadsData()));
        requestBuilder.addParameter(new RequestParameter<>(PARAM_MESSAGE, anr.getLongMessage()));
        requestBuilder.addParameter(new RequestParameter<>(PARAM_ANR_VERSION, anr.getAnrVersion()));
        requestBuilder.addParameter(new RequestParameter<>(PARAM_IS_EARLY_ANR, anr.isEarlyAnr()));
        if (anr.getMetadata().getUuid() != null) {
            requestBuilder.addParameter(new RequestParameter<>(PARAM_ID, anr.getMetadata().getUuid()));
        }
        if (anr.getAttachments() != null && anr.getAttachments().size() > 0) {
            requestBuilder.addParameter(new RequestParameter<>(PARAM_ATTACHMENTS_COUNT, anr.getAttachments().size()));
        }
        return requestBuilder.build();
    }

    public void uploadAnrLogs(final Anr anr, final Request.Callbacks<Boolean, Anr> reportingANRLogsCallbacks) {
        Request anrLogsRequest = buildAnrLogsRequest(anr);
        networkManager.doRequestOnSameThread(RequestType.NORMAL, anrLogsRequest, new Request.Callbacks<RequestResponse, Throwable>() {
            @Override
            public void onSucceeded(RequestResponse response) {
                InstabugSDKLogger.d(Constants.LOG_TAG, "Uploading ANR logs succeeded, Response code: " + response.getResponseCode());
                InstabugSDKLogger.v(Constants.LOG_TAG, "Uploading ANR logs succeeded,, Response body: " + response.getResponseBody());

                reportingANRLogsCallbacks.onSucceeded(true);
            }

            @Override
            public void onFailed(Throwable error) {
                InstabugSDKLogger.e(Constants.LOG_TAG, "Uploading ANR logs got error: " + error.getMessage());
                reportingANRLogsCallbacks.onFailed(anr);
            }
        });
    }

    @NonNull
    @VisibleForTesting
    public Request buildAnrLogsRequest(Anr anr) {
        // As agreed with BE, reporting ANR state logs will use the same endpoint as crashes
        Request.Builder requestBuilder = new Request.Builder()
                .endpoint(Endpoints.CRASH_LOGS.replaceAll(":crash_token",
                        anr.getTemporaryServerToken()))
                .method(RequestMethod.POST);

        getTokenFromState(requestBuilder, anr.getState());

        if (anr.getState() != null) {
            ArrayList<State.StateItem> logsItems = anr.getState().getLogsItems();
            if (logsItems != null && logsItems.size() > 0) {
                for (State.StateItem logItem : logsItems) {
                    if (logItem.getKey() != null) {
                        requestBuilder.addParameter(new RequestParameter<>(logItem.getKey(),
                                logItem.getValue() != null ? logItem.getValue() : ""));
                    }
                }
            }
        }
        return requestBuilder.build();
    }

    private void updateReportedAtIfNeeded(Request.Builder builder, Anr anr) {
        State state = anr.getState();
        if (state != null && !state.isMinimalState() && state.getReportedAt() != 0) return;
        try {
            //ANR id is basically a UTC time in milliseconds. So, it can be used as the report time.
            long reportedAt = anr.getId() != null ? Long.parseLong(anr.getId()) : 0L;
            if (reportedAt != 0L) {
                builder.addParameter(new RequestParameter<>(PARAM_REPORTED_AT, reportedAt));
            }
        } catch (Exception e) {
            IBGDiagnostics.reportNonFatal(e, "Failed to update reported_at in anr reporting request.");
        }
    }


    public void uploadAnrAttachments(final Anr anr, final Request.Callbacks<Boolean, Anr> anrAttachmentsCallbacks) throws JSONException {
        InstabugSDKLogger.d(Constants.LOG_TAG, "Uploading Anr attachments, size: " + anr.getAttachments().size());
        if (anr.getAttachments().size() == 0) {
            anrAttachmentsCallbacks.onSucceeded(true);
            return;
        }
        final List<Attachment> synced = new ArrayList<>();
        for (int i = 0; i < anr.getAttachments().size(); i++) {
            // create attachment request.
            final Attachment attachment = anr.getAttachments().get(i);
            boolean isAttachmentDecrypted = AttachmentsUtility.decryptAttachmentAndUpdateDb(attachment);
            if (isAttachmentDecrypted) {
                Request uploadingAnrAttachmentRequest = buildAnrSingleAttachmentRequest(anr, attachment);

                if (attachment.getLocalPath() != null) {
                    File file = new File(attachment.getLocalPath());
                    if (file.exists() && file.length() > 0) {
                        attachment.setAttachmentState(Attachment.AttachmentState.SYNCED);
                        networkManager.doRequestOnSameThread(RequestType.MULTI_PART, uploadingAnrAttachmentRequest, new Request.Callbacks<RequestResponse, Throwable>() {
                            @Override
                            public void onSucceeded(RequestResponse requestResponse) {
                                InstabugSDKLogger.d(Constants.LOG_TAG, "uploadingAnrAttachmentRequest Succeeded, Response code:" + requestResponse.getResponseCode());
                                InstabugSDKLogger.v(Constants.LOG_TAG, "uploadingAnrAttachmentRequest Succeeded, Response body: " + requestResponse.getResponseBody());

                                if (attachment.getLocalPath() != null) {
                                    deleteAttachment(attachment, anr.getId());
                                    synced.add(attachment);

                                }

                                if (synced.size() == anr.getAttachments().size()) {
                                    anrAttachmentsCallbacks.onSucceeded(true);
                                }
                            }

                            @Override
                            public void onFailed(Throwable error) {
                                InstabugSDKLogger.d(Constants.LOG_TAG, "uploading AnrAttachment Request got error: " + error.getMessage());
                                anrAttachmentsCallbacks.onFailed(anr);
                            }
                        });
                    } else {
                        InstabugSDKLogger.e(Constants.LOG_TAG, "Skipping attachment file of type "
                                + attachment.getType() + " because it's either not found or empty file");
                    }
                } else {
                    InstabugSDKLogger.e(Constants.LOG_TAG, "Skipping attachment file of type "
                            + attachment.getType() + " because it's either not found or empty file");
                }
            } else {
                InstabugSDKLogger.e(Constants.LOG_TAG, "Skipping attachment file of type "
                        + attachment.getType() + " because it was not decrypted successfully");
            }
        }
    }

    @NonNull
    @VisibleForTesting
    public Request buildAnrSingleAttachmentRequest(Anr anr, Attachment attachment) throws
            JSONException {
        // As agreed with BE, Uploading ANR Attachments will use the same endpoint as crashes
        Request.Builder requestBuilder = new Request.Builder()
                .endpoint(Endpoints.ADD_CRASH_ATTACHMENT.replaceAll(":crash_token", anr.getTemporaryServerToken()))
                .method(RequestMethod.POST)
                .type(RequestType.MULTI_PART);

        getTokenFromState(requestBuilder, anr.getState());

        if (attachment.getType() != null)
            requestBuilder.addParameter(new RequestParameter<>(PARAM_FILE_TYPE, attachment.getType()));
        if (attachment.getType() == Attachment.Type.AUDIO && attachment.getDuration() != null) {
            requestBuilder.addParameter(new RequestParameter<>(PARAM_DURATION, attachment
                    .getDuration()));
        }
        if (attachment.getName() != null && attachment.getLocalPath() != null) {
            requestBuilder.fileToUpload(new FileToUpload("file",
                    attachment.getName(), attachment.getLocalPath(), attachment.getFileType()));
        }
        return requestBuilder.build();
    }

}
