package com.instabug.anr.model;

import static com.instabug.library.core.InstabugCore.getFeatureState;
import static com.instabug.library.util.ReportHelper.getReport;

import android.annotation.SuppressLint;
import android.content.Context;
import android.net.Uri;

import androidx.annotation.NonNull;
import androidx.annotation.Nullable;
import androidx.annotation.VisibleForTesting;
import androidx.annotation.WorkerThread;

import com.instabug.anr.di.AnrServiceLocator;
import com.instabug.bganr.BackgroundAnrTraceFileParser;
import com.instabug.commons.AttachmentsHolder;
import com.instabug.commons.BasicAttachmentsHolder;
import com.instabug.commons.caching.DiskHelper;
import com.instabug.commons.models.Incident;
import com.instabug.commons.models.IncidentMetadata;
import com.instabug.commons.threading.CrashDetailsParser;
import com.instabug.commons.threading.CrashDetailsParser.ErrorParsingStrategy;
import com.instabug.commons.threading.CrashDetailsParser.ThreadParsingStrategy;
import com.instabug.commons.utils.StateExtKt;
import com.instabug.crash.Constants;
import com.instabug.library.Feature;
import com.instabug.library.IBGFeature;
import com.instabug.library.Instabug;
import com.instabug.library.core.InstabugCore;
import com.instabug.library.internal.storage.AttachmentsUtility;
import com.instabug.library.internal.storage.DiskUtils;
import com.instabug.library.internal.storage.cache.db.userAttribute.UserAttributesDbHelper;
import com.instabug.library.internal.storage.operation.WriteStateToFileDiskOperation;
import com.instabug.library.logging.InstabugLog;
import com.instabug.library.model.Attachment;
import com.instabug.library.model.Report;
import com.instabug.library.model.State;
import com.instabug.library.settings.SettingsManager;
import com.instabug.library.util.InstabugSDKLogger;
import com.instabug.library.util.ReportHelper;
import com.instabug.library.util.memory.MemoryUtils;

import org.json.JSONArray;
import org.json.JSONException;
import org.json.JSONObject;

import java.io.File;
import java.io.IOException;
import java.io.InputStream;
import java.util.List;
import java.util.Map;

import kotlin.Pair;

public class Anr implements Incident, AttachmentsHolder {

    public static final String ANR_STATE = "anr_state";
    private final String id;
    private String mainThreadData;
    private String restOfThreadsData;
    private final AttachmentsHolder attachmentsHolder;
    @AnrState
    private int anrState;
    private String temporaryServerToken;
    private State state;
    private String longMessage;

    @NonNull
    private IncidentMetadata metadata;

    @Nullable
    private String sessionId;

    @NonNull
    private Incident.Type type = Type.ANR;
    @NonNull
    @AnrVersion
    private String anrVersion = AnrVersion.VERSION_1;
    private boolean isEarlyAnr = false;

    // Creates an Empty ANR object to be filled with data while retrieving ANRs from DB
    @SuppressLint("ERADICATE_FIELD_NOT_INITIALIZED")
    public Anr(@NonNull String id, @NonNull IncidentMetadata metadata) {
        this.id = id;
        this.metadata = metadata;
        attachmentsHolder = new BasicAttachmentsHolder();
    }

    public Anr(Context context, String mainThreadData, String restOfThreadsData, String longMessage, @NonNull IncidentMetadata metadata) {
        this(String.valueOf(System.currentTimeMillis()), mainThreadData, restOfThreadsData, longMessage, State.getState(context), metadata);
    }

    @SuppressLint("ERADICATE_FIELD_NOT_INITIALIZED")
    private Anr(String id, String mainThreadData, String restOfThreadsData, String longMessage, State state, @NonNull IncidentMetadata metadata) {
        this(id, metadata);
        this.state = state;
        this.mainThreadData = mainThreadData;
        this.restOfThreadsData = restOfThreadsData;
        this.longMessage = longMessage;
    }

    @NonNull
    public String getId() {
        return id;
    }

    public void setMainThreadData(String mainThreadData) {
        this.mainThreadData = mainThreadData;
    }

    public String getMainThreadData() {
        return mainThreadData;
    }

    public String getRestOfThreadsData() {
        return restOfThreadsData;
    }

    public void setRestOfThreadsData(String restOfThreadsData) {
        this.restOfThreadsData = restOfThreadsData;
    }

    public Anr addAttachment(Uri attachmentUri) {
        addAttachment(attachmentUri, Attachment.Type.ATTACHMENT_FILE, false);
        return this;
    }

    public void setState(State state) {
        this.state = state;
    }

    public State getState() {
        return state;
    }

    @AnrState
    public int getAnrState() {
        return anrState;
    }

    public void setAnrState(@AnrState int anrState) {
        this.anrState = anrState;
    }

    public String getTemporaryServerToken() {
        return temporaryServerToken;
    }

    public void setTemporaryServerToken(String temporaryServerToken) {
        this.temporaryServerToken = temporaryServerToken;
    }

    public String getLongMessage() {
        return longMessage;
    }

    public void setLongMessage(String longMessage) {
        this.longMessage = longMessage;
    }

    @NonNull
    @Override
    public IncidentMetadata getMetadata() {
        return metadata;
    }

    @NonNull
    @Override
    public Type getType() {
        return type;
    }

    @VisibleForTesting
    public void setType(@NonNull Incident.Type type) {
        this.type = type;
    }

    @Nullable
    public String getSessionId() {
        return sessionId;
    }

    @VisibleForTesting
    public void setSessionId(@Nullable String sessionId) {
        this.sessionId = sessionId;
    }

    @NonNull
    @Override
    public File getSavingDirOnDisk(@NonNull Context ctx) {
        return DiskHelper.getIncidentSavingDirectory(ctx, Type.ANR.name(), id);
    }

    @NonNull
    @Override
    public List<Attachment> getAttachments() {
        return attachmentsHolder.getAttachments();
    }

    @Override
    public void addAttachment(@Nullable Uri uri, @NonNull Attachment.Type type, boolean isEncrypted) {
        attachmentsHolder.addAttachment(uri, type, isEncrypted);
    }

    @Override
    public void setAttachments(@NonNull List<? extends Attachment> attachments) {
        attachmentsHolder.setAttachments(attachments);
    }

    @NonNull
    @AnrVersion
    public String getAnrVersion() {
        return anrVersion;
    }

    public void setAnrVersion(@NonNull @AnrVersion String anrVersion) {
        this.anrVersion = anrVersion;
    }

    public boolean isEarlyAnr() {
        return isEarlyAnr;
    }

    public void setEarlyAnr(boolean earlyAnr) {
        isEarlyAnr = earlyAnr;
    }

    public static class Factory {

        public static final String ANR_ERROR_NAME = "ANRError: Application Not Responding for at least 5000 ms.";
        public static final String BG_ANR_EXCEPTION = "An ANR is detected while the app is in the background.";

        @Nullable
        public Anr createAnr(String anrException, String longMessage, @NonNull IncidentMetadata metadata) throws JSONException, IOException {
            Context context = Instabug.getApplicationContext();
            if (context == null) {
                InstabugSDKLogger.e(Constants.LOG_TAG, "Couldn't create a new instance of ANR due to a null context.");
                return null;
            }
            CrashDetailsParser parser = new CrashDetailsParser(
                    ThreadParsingStrategy.Main.INSTANCE,
                    new ErrorParsingStrategy.Main(ANR_ERROR_NAME, anrException)
            );
            Anr anr = new Anr(context, parser.getCrashDetails().toString(), parser.getThreadsDetails().toString(), longMessage, metadata);
            // Create the State file
            if (anr.getState() != null) {
                anr.getState().setAppStatusToForeground();
                updateAnrState(anr);
                addReproScreenshotsAttachmentIfApplicable(anr, context);
                updateStateWithReport(anr.getState());
                persistAnrState(context, anr.getState(), anr.getSavingDirOnDisk(context));
            }
            // Add File attachments
            updateAttachmentFiles(context, anr);
            return anr;
        }

        public Anr createOSAnr(@NonNull Context ctx, InputStream traceStream, State state, @NonNull IncidentMetadata metadata, @NonNull String sessionId, @Nullable File reproScreenshotsDir, boolean isBackground) throws JSONException, IOException {
            String message = isBackground ? BG_ANR_EXCEPTION : ANR_ERROR_NAME;
            Type type = isBackground ? Type.BG_ANR : Type.ANR;
            Pair<JSONObject, JSONArray> detailsAndThreads = new BackgroundAnrTraceFileParser()
                    .invoke(traceStream, ANR_ERROR_NAME, message);
            Anr anr = new Anr(
                    String.valueOf(System.currentTimeMillis()),
                    detailsAndThreads.getFirst().toString(),
                    detailsAndThreads.getSecond().toString(),
                    message, state, metadata
            );
            anr.sessionId = sessionId;
            anr.type = type;
            if (anr.getState() != null) {
                if (type == Type.BG_ANR)
                    anr.getState().setAppStatusToBackground();
                else
                    anr.getState().setAppStatusToForeground();
                addReproScreenshotsAttachment(anr, ctx, reproScreenshotsDir);
                persistAnrState(ctx, anr.getState(), anr.getSavingDirOnDisk(ctx));
            }
            // Add File attachments
            updateAttachmentFiles(ctx, anr);
            return anr;
        }

        @Nullable
        public Anr createEarlyAnr(@NonNull Context ctx, InputStream traceStream, State state, @NonNull IncidentMetadata metadata) throws JSONException, IOException {
            Pair<JSONObject, JSONArray> detailsAndThreads = new BackgroundAnrTraceFileParser()
                    .invoke(traceStream, ANR_ERROR_NAME, ANR_ERROR_NAME);
            Anr anr = new Anr(
                    String.valueOf(System.currentTimeMillis()),
                    detailsAndThreads.getFirst().toString(),
                    detailsAndThreads.getSecond().toString(),
                    ANR_ERROR_NAME, state, metadata
            );
            if (anr.getState() != null) {
                anr.getState().setAppStatusToForeground();
                persistAnrState(ctx, anr.getState(), anr.getSavingDirOnDisk(ctx));
            }
            // Add File attachments
            updateAttachmentFiles(ctx, anr);
            anr.setAnrVersion(AnrVersion.VERSION_2);
            anr.setEarlyAnr(true);
            anr.setAnrState(AnrState.READY_TO_BE_SENT);
            return anr;
        }

        private void updateAnrState(Anr anr) {
            Context context = Instabug.getApplicationContext();
            if (context != null && !MemoryUtils.isLowMemory(context) && getFeatureState(IBGFeature.USER_EVENTS) == Feature.State.ENABLED) {
                try {
                    anr.getState().updateUserEvents();
                } catch (JSONException e) {
                    InstabugSDKLogger.e(Constants.LOG_TAG, "Got error while parsing " +
                            "user events logs", e);
                }
            }
            if (SettingsManager.getInstance().getOnReportCreatedListener() == null) {
                anr.getState().setTags(InstabugCore.getTagsAsString());

                anr.getState().updateConsoleLog();
                if (getFeatureState(IBGFeature.USER_DATA)
                        == Feature.State.ENABLED) {
                    anr.getState().setUserData(InstabugCore.getUserData());
                }
                if (getFeatureState(IBGFeature.INSTABUG_LOGS)
                        == Feature.State.ENABLED) {
                    anr.getState().setInstabugLog(InstabugLog.getLogs());
                }
            }
            anr.getState().setUserAttributes(UserAttributesDbHelper.getSDKUserAttributes());
            if (AnrServiceLocator.getReproConfigurationsProvider().isReproStepsEnabled()) {
                anr.getState().updateVisualUserSteps();
            }
            StateExtKt.updateScreenShotAnalytics(anr.getState());
        }

        private void persistAnrState(@NonNull Context ctx, @NonNull State state, @NonNull File savingDir) throws JSONException, IOException {
            File file = DiskHelper.getIncidentStateFile(savingDir, ANR_STATE);
            Uri uri = DiskUtils.with(ctx)
                    .writeOperation(new WriteStateToFileDiskOperation(file, state.toJson()))
                    .execute();
            state.setUri(uri);
        }

        private void updateStateWithReport(@NonNull State state) {
            Report report = getReport(InstabugCore.getOnReportCreatedListener());
            ReportHelper.update(state, report);
        }

        private void updateAttachmentFiles(@NonNull Context ctx, @NonNull Anr anr) {
            if (InstabugCore.getExtraAttachmentFiles() != null &&
                    InstabugCore.getExtraAttachmentFiles().size() >= 1) {
                for (Map.Entry<Uri, String> entry : InstabugCore.getExtraAttachmentFiles()
                        .entrySet()) {
                    Uri attachmentUri = AttachmentsUtility.getNewFileAttachmentUri(ctx, entry.getKey(),
                            entry.getValue());
                    if (attachmentUri != null) {
                        anr.addAttachment(attachmentUri);
                    }
                }
            }
        }

        @WorkerThread
        private void addReproScreenshotsAttachmentIfApplicable(@NonNull Anr anr, @NonNull Context context) {
            if (!AnrServiceLocator.getReproConfigurationsProvider().isReproScreenshotsEnabled())
                return;
            File reproScreenshotsDir = AnrServiceLocator.getReproScreenshotsCacheDir().getCurrentSpanDirectory();
            addReproScreenshotsAttachment(anr, context, reproScreenshotsDir);
        }

        @WorkerThread
        private void addReproScreenshotsAttachment(@NonNull Anr anr, @NonNull Context context, @Nullable File reproScreenshotsDir) {
            if (reproScreenshotsDir == null) return;
            Pair<String, Boolean> zippingResult = DiskHelper.getReproScreenshotsZipPath(context, anr.getId(), anr.getSavingDirOnDisk(context), reproScreenshotsDir);
            if (zippingResult.getFirst() == null) return;
            anr.addAttachment(Uri.parse(zippingResult.getFirst()), Attachment.Type.VISUAL_USER_STEPS, zippingResult.getSecond());
        }
    }

}
