package com.instabug.early_crash

import android.content.Context
import com.instabug.commons.PluginDelegate
import com.instabug.crash.BuildConfig
import com.instabug.early_crash.di.EarlyCrashesServiceLocator.cacheHandler
import com.instabug.early_crash.di.EarlyCrashesServiceLocator.configurationsHandler
import com.instabug.early_crash.di.EarlyCrashesServiceLocator.configurationsProvider
import com.instabug.early_crash.di.EarlyCrashesServiceLocator.normalEarlyCrashUploaderJob
import com.instabug.early_crash.threading.ExecQs
import com.instabug.early_crash.threading.ExecutionMode
import com.instabug.library.core.eventbus.coreeventbus.IBGSdkCoreEvent
import com.instabug.library.core.eventbus.coreeventbus.IBGSdkCoreEvent.Features
import com.instabug.library.core.eventbus.coreeventbus.IBGSdkCoreEvent.FeaturesFetched
import com.instabug.library.core.eventbus.coreeventbus.IBGSdkCoreEvent.NetworkActivated
import com.instabug.library.util.threading.PoolProvider

class EarlyCrashesPluginDelegate : PluginDelegate {

    private var currentFeatureState = BuildConfig.CAPTURE_EARLY_CRASHES
    override fun init(context: Context) {
        // No-Op
    }

    override fun start(context: Context) {
        operateOnExec {
            currentFeatureState = configurationsProvider.isEarlyCrashesEnabled
            internalStart()
        }
    }

    private fun internalStart() {
        if (currentFeatureState) {
            cacheHandler.trim(configurationsProvider.maxCachedReports, ExecutionMode.OrderedCaching)
            normalEarlyCrashUploaderJob()
            return
        }
        cacheHandler.clearAll(ExecutionMode.OrderedCaching)
    }

    override fun wake() {
        // No-Op
    }

    override fun sleep() {
        // No-Op
    }

    override fun stop() {
        // No-Op
    }

    override fun handleSDKCoreEvent(sdkCoreEvent: IBGSdkCoreEvent) {
        when (sdkCoreEvent) {
            NetworkActivated -> operateOnExec { doIfFeatureEnabled { normalEarlyCrashUploaderJob() } }
            is Features.Updated -> operateOnExec { handleConfigurationChanges() }
            is FeaturesFetched -> operateOnExec {
                configurationsHandler.handleConfiguration(sdkCoreEvent.response)
                handleConfigurationChanges()
            }

            else -> Unit
        }
    }

    private fun operateOnExec(operation: () -> Unit) {
        PoolProvider.postOrderedIOTask(ExecQs.Lifecycle, operation)
    }

    private fun handleConfigurationChanges() {
        val newState = configurationsProvider.isEarlyCrashesEnabled
        if (newState == currentFeatureState) return
        currentFeatureState = newState
        doIfFeatureDisabled {
            cacheHandler.clearAll(ExecutionMode.OrderedCaching)
        }
    }

    private inline fun doIfFeatureEnabled(operation: () -> Unit) {
        if (!currentFeatureState) return
        operation()
    }

    private inline fun doIfFeatureDisabled(operation: () -> Unit) {
        if (currentFeatureState) return
        operation()
    }
}