package com.instabug.crash.configurations

import com.instabug.commons.configurations.ConfigurationsHandler
import com.instabug.commons.preferences.crashesPreferences
import com.instabug.commons.utils.SharedPreferencesUtils
import com.instabug.crash.Constants
import com.instabug.crash.Constants.Preferences.CRASH_METADATA_CALLBACK_ENABLED
import com.instabug.crash.Constants.Preferences.IS_CRASH_REPORTING_MIGRATED
import com.instabug.crash.Constants.Preferences.METADATA_IMMEDIATE_SYNC_AVAILABILITY
import com.instabug.crash.Constants.Preferences.NON_FATAL_AVAILABILITY
import com.instabug.crash.di.CrashesServiceLocator
import com.instabug.library.Instabug
import com.instabug.library.internal.sharedpreferences.corePreferences
import com.instabug.library.util.InstabugSDKLogger
import com.instabug.library.util.extenstions.runOrLogError
import org.json.JSONObject

class CrashConfigurationHandlerImpl : ConfigurationsHandler {


    companion object {
        const val CRASH_REPORTING_KEY = "crash_reporting"
        const val CORE_CRASH_REPORTING_KEY = "CRASH_REPORTINGAVAIL"
        const val NON_FATAL_AVAILABLE_KEY = "non_fatal"
        const val METADATA_IMMEDIATE_SYNC_KEY = "realtime_metadata"
        private const val CRASHES_KEY = "crashes"
        private const val CRASHES_METADATA_CALLBACK_KEY = "metadata_callback"

    }

    private val JSONObject.crashes
        get() = this.optJSONObject(CRASHES_KEY)

    private val JSONObject.metadataCallbackEnabled
        get() = optBoolean(
            CRASHES_METADATA_CALLBACK_KEY,
            CRASH_METADATA_CALLBACK_ENABLED.second
        )

    private val JSONObject?.nonFatalEnabled
        get() = this?.optBoolean(
            NON_FATAL_AVAILABLE_KEY,
            NON_FATAL_AVAILABILITY.second
        ) ?: NON_FATAL_AVAILABILITY.second

    private val JSONObject?.metadataImmediateSync
        get() = this?.optBoolean(
            METADATA_IMMEDIATE_SYNC_KEY,
            METADATA_IMMEDIATE_SYNC_AVAILABILITY.second
        ) ?: METADATA_IMMEDIATE_SYNC_AVAILABILITY.second

    override fun handleConfiguration(configuration: String?) {
        configuration?.let {
            runOrLogError(
                tag = Constants.LOG_TAG,
                errorMessage = "Something went wrong while parsing crash_reporting from features response"
            ) {
                val responseObject = JSONObject(configuration)
                val isCrashReportingEnabled = responseObject.optBoolean(CRASH_REPORTING_KEY)
                val crashesJsonObject = responseObject.crashes
                val crashMetaDataCallbackEnabled =
                    crashesJsonObject?.metadataCallbackEnabled ?: false
                val isNonFatalEnabled =
                    crashesJsonObject.nonFatalEnabled
                val metadataImmediateSyncEnabled =
                    crashesJsonObject.metadataImmediateSync
                CrashesServiceLocator.crashConfigurationProvider.apply {
                    isCrashReportingAvailable = isCrashReportingEnabled
                    isCrashMetaDataCallbackEnabled = crashMetaDataCallbackEnabled
                    isNonFatalReportingAvailable = isNonFatalEnabled
                    isMetadataImmediateSyncAvailable = metadataImmediateSyncEnabled
                }
                InstabugSDKLogger.i(
                    Constants.LOG_TAG,
                    "Crash reporting enabled = $isCrashReportingEnabled"
                )
            }
        }
    }

    override fun migrateCurrentConfiguration() {
        if (!isCrashReportingMigrated())
            Instabug.getApplicationContext()?.let {
                SharedPreferencesUtils.readBoolean(
                    CORE_CRASH_REPORTING_KEY,
                    Constants.Preferences.CRASH_REPORTING_AVAILABILITY.second,
                    corePreferences
                ).also { migratedValue ->
                    CrashesServiceLocator.crashConfigurationProvider.isCrashReportingAvailable =
                        migratedValue
                    setCrashReportingMigrated()
                }
            }
    }

    private fun isCrashReportingMigrated(): Boolean {
        val (key, defaultValue) = IS_CRASH_REPORTING_MIGRATED
        return SharedPreferencesUtils.readBoolean(key, defaultValue, crashesPreferences)
    }

    private fun setCrashReportingMigrated() {
        SharedPreferencesUtils.writeBoolean(
            IS_CRASH_REPORTING_MIGRATED.first,
            true,
            crashesPreferences
        )
    }
}