package com.instabug.commons.configurations

import androidx.annotation.WorkerThread
import com.instabug.commons.logging.runOrReportError
import com.instabug.commons.preferences.PrefSpecs.ReproAttachmentsAvailability
import com.instabug.commons.preferences.crashPref
import com.instabug.crash.utils.CrashReportingUtility
import com.instabug.library.IBGFeature
import com.instabug.library.core.InstabugCore
import com.instabug.library.visualusersteps.ReproConfigurationsProvider
import com.instabug.library.visualusersteps.ReproProxyAuthID
import com.instabug.library.visualusersteps.ReproRuntimeConfigurationsHandler
import org.json.JSONObject

interface ConfigurationsHandler: ReproRuntimeConfigurationsHandler {

    /**
     * Handles feature request response
     *
     * @param configuration response of feature request
     */
    @WorkerThread
    fun handleConfiguration(configuration: String?)

    /**
     * Migrates old configuration from core module
     */
    @WorkerThread
    fun migrateCurrentConfiguration()
}

/**
 * A composite configurations handler combining [ConfigurationsHandler] & [ReproRuntimeConfigurationsHandler]
 */
interface CompositeConfigurationsHandler : ConfigurationsHandler, ReproRuntimeConfigurationsHandler

interface ConfigurationsProvider : ReproConfigurationsProvider {
    /**
     * Whether repro screenshots feature is available.
     */
    var userIdentificationEnabled: Boolean
}

private const val CONFIG_KEY_CRASHES = "crashes"
private const val CONFIG_KEY_REPRO_SCREENSHOTS = "rsa"

class CommonCrashesConfigurationsHandler(
    private val configurationsProvider: ConfigurationsProvider,
    private val reproRuntimeStateHandlerDelegate: ReproRuntimeConfigurationsHandler
) : CompositeConfigurationsHandler,
    ReproRuntimeConfigurationsHandler by reproRuntimeStateHandlerDelegate {
    override fun handleConfiguration(configuration: String?) {
        runCatching {
            configuration?.let(::JSONObject)?.optJSONObject(CONFIG_KEY_CRASHES)
                ?.also(this::extractAndUpdateRSA)
        }.runOrReportError("Error while parsing configurations")
    }

    override fun migrateCurrentConfiguration() {
        // No-Op
    }

    private fun extractAndUpdateRSA(crashesObj: JSONObject) {
        crashesObj.takeIf { obj -> obj.has(CONFIG_KEY_REPRO_SCREENSHOTS) }
            ?.optBoolean(CONFIG_KEY_REPRO_SCREENSHOTS, ReproAttachmentsAvailability.second)
            ?.let { rsa -> configurationsProvider.isReproScreenshotsAvailable = rsa }
    }
}

class CommonCrashesConfigurationsProvider : ConfigurationsProvider {
    override val reproProxyAuthId: Int = ReproProxyAuthID.Fatals
    override var isReproScreenshotsAvailable: Boolean by crashPref(ReproAttachmentsAvailability)
    override var userIdentificationEnabled: Boolean = true

    override var isReproStepsEnabledSDK: Boolean = true
    override val isReproStepsEnabled: Boolean
        get() = isReproStepsEnabledSDK && isReproFeaturesAvailable && isCrashReportingEnabled


    override var isReproScreenShotsEnabledSDK: Boolean = false

    override val isReproScreenshotsEnabled: Boolean
        get() = isReproScreenShotsEnabledSDK && isReproScreenshotsAvailable &&
                isReproFeaturesAvailable && isCrashReportingEnabled

    private val isReproFeaturesAvailable: Boolean
        get() = InstabugCore.isFeatureAvailable(IBGFeature.REPRO_STEPS)
    private val isCrashReportingEnabled: Boolean
        get() = CrashReportingUtility.isCrashReportingEnabled()


}