package com.instabug.bug.view;

import static androidx.core.view.accessibility.AccessibilityNodeInfoCompat.ACTION_CLICK;

import android.annotation.SuppressLint;
import android.app.Activity;
import android.content.Context;
import android.graphics.Bitmap;
import android.graphics.ColorFilter;
import android.graphics.PorterDuff;
import android.graphics.PorterDuffColorFilter;
import android.graphics.drawable.AnimationDrawable;
import android.graphics.drawable.Drawable;
import android.view.LayoutInflater;
import android.view.View;
import android.view.ViewGroup;
import android.widget.ImageView;
import android.widget.ProgressBar;
import android.widget.RelativeLayout;

import androidx.annotation.NonNull;
import androidx.annotation.Nullable;
import androidx.annotation.StringRes;
import androidx.annotation.VisibleForTesting;
import androidx.appcompat.content.res.AppCompatResources;
import androidx.core.view.AccessibilityDelegateCompat;
import androidx.core.view.ViewCompat;
import androidx.core.view.accessibility.AccessibilityNodeInfoCompat;
import androidx.recyclerview.widget.RecyclerView;

import com.instabug.bug.Constants;
import com.instabug.bug.R;
import com.instabug.bug.settings.BugSettings;
import com.instabug.library.core.InstabugCore;
import com.instabug.library.model.Attachment;
import com.instabug.library.settings.SettingsManager;
import com.instabug.library.util.AccessibilityUtils;
import com.instabug.library.util.AttrResolver;
import com.instabug.library.util.BitmapUtils;
import com.instabug.library.util.InstabugSDKLogger;
import com.instabug.library.util.KeyboardUtils;
import com.instabug.library.util.LocaleUtils;
import com.instabug.library.util.OnVideoFrameReady;
import com.instabug.library.util.VideoManipulationUtils;
import com.instabug.library.view.IconView;

import org.jetbrains.annotations.NotNull;

import java.util.ArrayList;
import java.util.List;
import java.util.Locale;

/**
 * Created by mohamedzakaria on 5/14/17.
 */

public class AttachmentsAdapter extends RecyclerView.Adapter<RecyclerView.ViewHolder> {

    @VisibleForTesting
    static final int TYPE_IMG = 0;
    @VisibleForTesting
    static final int TYPE_VID = 1;
    int[] helperImages = {
            R.drawable.ibg_bug_ic_edit,
            R.drawable.ibg_bug_ic_magnify,
            R.drawable.ibg_bug_ic_blur,
            R.drawable.ibg_bug_ic_edit,
            R.drawable.ibg_bug_ic_magnify,
            R.drawable.ibg_bug_ic_blur,
            R.drawable.ibg_bug_ic_edit};
    private List<Attachment> attachmentList;
    @Nullable
    private ColorFilter colorFilter;
    private AttachmentOnClickListener onClickListener;
    @Nullable
    private ProgressBar videoProgressBar;
    @Nullable
    private ImageView videoPlayImageView;
    @Nullable
    private Context context;
    private int lastImgIndex = -1;

    public AttachmentsAdapter(@Nullable Context context, @Nullable ColorFilter colorFilter, AttachmentOnClickListener onClickListener) {
        this.context = context;
        this.colorFilter = colorFilter;
        this.onClickListener = onClickListener;
        setHasStableIds(true);
        attachmentList = new ArrayList<>();
    }

    @NonNull
    public Attachment getItem(int position) {
        return attachmentList.get(position);
    }

    @Override
    public long getItemId(int position) {
        return getItem(position).getId();
    }

    @Override
    public int getItemCount() {
        return attachmentList != null ? attachmentList.size() : 0;
    }

    @NotNull
    @Override
    public RecyclerView.ViewHolder onCreateViewHolder(@NotNull ViewGroup parent, int viewType) {
        switch (viewType) {
            default:
            case TYPE_IMG:
                return new ImgViewHolder(LayoutInflater.from(parent.getContext()).inflate(R.layout.ibg_bug_lyt_attachment_image, parent, false));
            case TYPE_VID:
                return new VideoViewHolder(LayoutInflater.from(parent.getContext()).inflate(R.layout.ibg_bug_lyt_attachment_video, parent, false));
        }
    }

    @Override
    @SuppressLint("STARVATION")
    public void onBindViewHolder(@NotNull RecyclerView.ViewHolder holder, int position) {
        switch (getItemViewType(position)) {
            default:
            case TYPE_IMG:
                bindImageAttachmentView((ImgViewHolder) holder, getItem(position));
                if (lastImgIndex != -1 && position == lastImgIndex && getItem(position).shouldAnimate()) {
                    animate((ImgViewHolder) holder);
                    getItem(position).setShouldAnimate(false);
                }
                break;
            case TYPE_VID:
                bindVideoAttachmentView((VideoViewHolder) holder, getItem(position));
                break;
        }
    }

    @Override
    public int getItemViewType(int position) {
        if (attachmentList == null || attachmentList.size() == 0 || attachmentList.get(position).getType() == null)
            return super.getItemViewType(position);
        switch (attachmentList.get(position).getType()) {
            default:
            case EXTRA_IMAGE:
            case GALLERY_IMAGE:
            case MAIN_SCREENSHOT:
                return TYPE_IMG;
            case EXTRA_VIDEO:
            case GALLERY_VIDEO:
            case AUTO_SCREEN_RECORDING_VIDEO:
                return TYPE_VID;
        }
    }


    private void bindImageAttachmentView(final ImgViewHolder holder,@NonNull Attachment attachment) {
        if (attachment.getLocalPath() != null && holder.imageView != null) {
            BitmapUtils.loadBitmap(attachment.getLocalPath(), holder.imageView);
        }
        if (holder.imageView != null) {
            holder.imageView.setTag(attachment);
            if (holder.gridItem != null) {
                holder.imageView.setOnClickListener(registerOnClickListener(holder.gridItem, attachment));
            }
        }
        if (holder.editIcon != null && holder.gridItem != null) {
            holder.editIcon.setOnClickListener(registerOnClickListener(holder.gridItem, attachment));
        }
        if (holder.gridItem != null) {
            holder.gridItem.setOnClickListener(registerOnClickListener(holder.gridItem, attachment));
        }
        if (holder.removeAttachment != null) {
            holder.removeAttachment.setTag(attachment);
            holder.removeAttachment.setOnClickListener(registerOnClickListener(holder.removeAttachment, attachment));
            holder.removeAttachment.setTextColor(SettingsManager.getInstance().getPrimaryColor());
        }
        if (attachment.getName() != null && holder.imageView != null) {
            ViewCompat.setTransitionName(holder.imageView, attachment.getName());
        }
        if (holder.squareLayout != null) {
            setBorder(holder.squareLayout);
        }
        if (holder.removeAttachment != null && holder.removeAttachmentCircleView != null) {
            if (attachment.getType() == Attachment.Type.MAIN_SCREENSHOT &&
                    BugSettings.getInstance().isScreenshotRequired()) {
                holder.removeAttachment.setVisibility(View.GONE);
                holder.removeAttachmentCircleView.setVisibility(View.GONE);
            } else {
                holder.removeAttachment.setVisibility(View.VISIBLE);
                holder.removeAttachmentCircleView.setVisibility(View.VISIBLE);
            }
        }
        final String attachmentDescription = getImageContentDescription(holder.getAdapterPosition());
        if (holder.imageView != null)
            holder.imageView.setContentDescription(attachmentDescription);

        // Handle Accessibility
        if (AccessibilityUtils.isTalkbackEnabled()) {
            if (holder.editIcon != null)
                ViewCompat.setImportantForAccessibility(holder.editIcon, ViewCompat.IMPORTANT_FOR_ACCESSIBILITY_NO);
            if (holder.gridItem != null) {
                ViewCompat.setImportantForAccessibility(holder.gridItem, ViewCompat.IMPORTANT_FOR_ACCESSIBILITY_NO);
                holder.gridItem.setFocusable(false);
            }

            if (holder.imageView != null) {
                ViewCompat.setAccessibilityDelegate(holder.imageView, new AccessibilityDelegateCompat() {

                    @Override
                    public void onInitializeAccessibilityNodeInfo(View host, AccessibilityNodeInfoCompat info) {
                        super.onInitializeAccessibilityNodeInfo(host, info);
                        info.setContentDescription(attachmentDescription);
                        info.setRoleDescription("");
                        host.setOnFocusChangeListener((view, isFocused) -> {
                            if (isFocused) KeyboardUtils.hide((Activity) context);
                        });
                        info.addAction(new AccessibilityNodeInfoCompat.AccessibilityActionCompat(
                                ACTION_CLICK,
                                getLocalizedString(R.string.ibg_bug_report_attachment_edit_content_description,
                                        holder.itemView.getContext())));
                    }
                });
            }

            if (holder.removeAttachment != null) {
                final String removeAttachmentDescription = getLocalizedString(R.string.ibg_bug_report_attachment_remove_content_description,
                        holder.itemView.getContext()) + " " + attachmentDescription;
                holder.removeAttachment.setContentDescription(removeAttachmentDescription);
                ViewCompat.setAccessibilityDelegate(holder.removeAttachment, new AccessibilityDelegateCompat() {
                    @Override
                    public void onInitializeAccessibilityNodeInfo(View host, AccessibilityNodeInfoCompat info) {
                        super.onInitializeAccessibilityNodeInfo(host, info);
                        info.setRoleDescription("Button");
                    }
                });
            }

        }
    }

    private String getImageContentDescription(int adapterPosition) {
        int noOfImagesInList = 0;
        for (int index = 0; index <= adapterPosition; index++) {
            if (getItemViewType(index) == TYPE_IMG) {
                noOfImagesInList++;
            }
        }
        return String.format(Locale.ENGLISH, "Image attachment number %d", noOfImagesInList);
    }

    public void animate(final ImgViewHolder holder) {
        final AnimationDrawable animationDrawable = new AnimationDrawable();
        for (int helperImageRes : helperImages) {
            if (context != null) {
                Drawable drawable = AppCompatResources.getDrawable(context, helperImageRes);
                if (drawable != null) {
                    animationDrawable.addFrame(drawable, 1500);
                } else {
                    animationDrawable.stop();
                }
            }
        }
        animationDrawable.setEnterFadeDuration(200);
        animationDrawable.setOneShot(true);
        if (holder.editIcon != null) {
            holder.editIcon.setImageDrawable(animationDrawable);
            holder.editIcon.post(() -> animationDrawable.start());
        }
    }

    private void setBorder(RelativeLayout squareLayout) {
        if (context != null) {
            Drawable corneredBackground = context.getResources().getDrawable(R.drawable.ibg_bug_shape_attachment_border);
            corneredBackground.setColorFilter(new
                    PorterDuffColorFilter(AttrResolver.resolveAttributeColor(context, R.attr.ibg_bug_attachment_border_color), PorterDuff.Mode.SRC_IN));
            squareLayout.setBackgroundDrawable(corneredBackground);
        }
    }

    private void bindVideoAttachmentView(VideoViewHolder holder, final Attachment attachment) {
        if (holder.removeAttachment != null) {
            View removeAttachmentButton = holder.removeAttachment.findViewById(R.id.instabug_btn_remove_attachment);
            if (removeAttachmentButton != null) {
                removeAttachmentButton.setTag(attachment);
                removeAttachmentButton.setOnClickListener(registerOnClickListener(holder.removeAttachment, attachment));
            }
            holder.removeAttachment.setTextColor(SettingsManager.getInstance().getPrimaryColor());
        }
        if (holder.videoPlayImageView != null && colorFilter != null) {
            holder.videoPlayImageView.setColorFilter(colorFilter);
        }
        if (holder.imageView != null) {
            holder.imageView.setTag(attachment);
            if (holder.gridItem != null) {
                holder.imageView.setOnClickListener(registerOnClickListener(holder.gridItem, attachment));
            }
        }
        if (holder.videoPlayImageView != null && holder.gridItem != null) {
            holder.videoPlayImageView.setOnClickListener(registerOnClickListener(holder.gridItem, attachment));
        }
        if (holder.gridItem != null) {
            holder.gridItem.setOnClickListener(registerOnClickListener(holder.gridItem, attachment));
        }
        videoPlayImageView = holder.videoPlayImageView;
        videoProgressBar = holder.videoProgressBar;

        if (attachment.getLocalPath() != null) {
            InstabugSDKLogger.v(Constants.LOG_TAG, "Video path found, extracting it's first frame " + attachment.getLocalPath());
            VideoManipulationUtils.extractFirstVideoFrame(attachment.getLocalPath(), new OnVideoFrameReady() {
                @Override
                public void onReady(@Nullable Bitmap frame) {
                    if (frame != null && holder.imageView != null) {
                        holder.imageView.setImageBitmap(frame);
                    }
                }
            });
        } else {
            InstabugSDKLogger.v(Constants.LOG_TAG, "Neither video path nor main screenshot found, using white background");
            if (holder.imageView != null) {
                holder.imageView.setImageResource(R.drawable.ibg_core_bg_card);
            }
            if (videoProgressBar != null && videoProgressBar.getVisibility() == View.GONE) {
                videoProgressBar.setVisibility(View.VISIBLE);
            }

            if (videoPlayImageView != null && videoPlayImageView.getVisibility() == View.VISIBLE) {
                videoPlayImageView.setVisibility(View.GONE);
            }
        }

        if (holder.squareLayout != null) {
            setBorder(holder.squareLayout);
        }

        // Handle Accessibility
        if (AccessibilityUtils.isTalkbackEnabled()) {
            final String attachmentDescription = getVideoContentDescription(holder.getAdapterPosition());
            if (holder.videoPlayImageView != null)
                ViewCompat.setImportantForAccessibility(holder.videoPlayImageView, ViewCompat.IMPORTANT_FOR_ACCESSIBILITY_NO);

            if (holder.imageView != null) {
                ViewCompat.setAccessibilityDelegate(holder.imageView, new AccessibilityDelegateCompat() {

                    @Override
                    public void onInitializeAccessibilityNodeInfo(View host, AccessibilityNodeInfoCompat info) {
                        super.onInitializeAccessibilityNodeInfo(host, info);
                        info.setContentDescription(attachmentDescription);
                        info.setRoleDescription("");
                        host.setOnFocusChangeListener((view, isFocused) -> {
                            if (isFocused) KeyboardUtils.hide((Activity) context);
                        });
                        info.addAction(new AccessibilityNodeInfoCompat.AccessibilityActionCompat(
                                ACTION_CLICK,
                                holder.itemView.getContext().getString(R.string.ibg_bug_report_video_play_content_description)));
                    }
                });
            }
            if (holder.removeAttachment != null) {
                final String removeAttachmentDescription = getLocalizedString(R.string.ibg_bug_report_attachment_remove_content_description,
                        holder.itemView.getContext()) + " " + attachmentDescription;
                holder.removeAttachment.setContentDescription(removeAttachmentDescription);
                ViewCompat.setAccessibilityDelegate(holder.removeAttachment, new AccessibilityDelegateCompat() {
                    @Override
                    public void onInitializeAccessibilityNodeInfo(View host, AccessibilityNodeInfoCompat info) {
                        super.onInitializeAccessibilityNodeInfo(host, info);
                        info.setRoleDescription("Button");
                    }
                });
            }

        }
    }

    private String getVideoContentDescription(int adapterPosition) {
        int noOfVideosInList = 0;
        for (int index = 0; index <= adapterPosition; index++) {
            if (getItemViewType(index) == TYPE_VID) {
                noOfVideosInList++;
            }
        }
        return String.format(Locale.ENGLISH, "Video attachment number %d", noOfVideosInList);
    }

    private View.OnClickListener registerOnClickListener(final View itemView,@NonNull final Attachment attachment) {
        return v -> onClickListener.onAttachmentClicked(itemView, attachment);
    }

    @Nullable
    public ProgressBar getVideoProgressBar() {
        return videoProgressBar;
    }

    @Nullable
    public ImageView getVideoPlayImageView() {
        return videoPlayImageView;
    }

    public void addAttachment(Attachment attachment) {
        attachmentList.add(attachment);
    }

    public void clearAttachments() {
        attachmentList.clear();
    }

    public void removeAttachment(Attachment attachmentToRemove) {
        attachmentList.remove(attachmentToRemove);
    }

    public void setLastImageIndex(int lastImgIndex) {
        this.lastImgIndex = lastImgIndex;
    }

    public List<Attachment> getDataset() {
        return attachmentList;
    }

    public interface AttachmentOnClickListener {
        void onAttachmentClicked(View v, Attachment attachment);
    }

    public static class ImgViewHolder extends RecyclerView.ViewHolder {
        @Nullable
        RelativeLayout gridItem;
        @Nullable
        RelativeLayout squareLayout;
        @Nullable
        ImageView imageView;
        @Nullable
        ImageView editIcon;
        @Nullable
        IconView removeAttachment;
        @Nullable
        View removeAttachmentCircleView;

        public ImgViewHolder(View itemView) {
            super(itemView);
            imageView = itemView.findViewById(R.id.instabug_img_attachment);
            editIcon = itemView.findViewById(R.id.instabug_btn_image_edit_attachment);
            gridItem = itemView.findViewById(R.id.instabug_attachment_img_item);
            removeAttachment = itemView.findViewById(R.id.instabug_btn_remove_attachment);
            squareLayout = itemView.findViewById(R.id.instabug_attachemnt_thumb_background);
            removeAttachmentCircleView = itemView.findViewById(R.id.instabug_btn_remove_attachment_circle);
        }
    }

    public static class VideoViewHolder extends RecyclerView.ViewHolder {
        @Nullable
        RelativeLayout gridItem;
        @Nullable
        RelativeLayout squareLayout;
        @Nullable
        ProgressBar videoProgressBar;
        @Nullable
        IconView removeAttachment;
        @Nullable
        ImageView videoPlayImageView;
        @Nullable
        ImageView imageView;

        public VideoViewHolder(View itemView) {
            super(itemView);
            gridItem = itemView.findViewById(R.id.instabug_attachment_video_item);
            imageView = itemView.findViewById(R.id.instabug_img_video_attachment);
            removeAttachment = itemView.findViewById(R.id.instabug_btn_remove_attachment);
            videoProgressBar = itemView.findViewById(R.id.instabug_attachment_progress_bar);
            videoPlayImageView = itemView.findViewById(R.id.instabug_btn_video_play_attachment);
            squareLayout = itemView.findViewById(R.id.instabug_attachemnt_thumb_background);

            if (videoProgressBar != null) {
                videoProgressBar.getIndeterminateDrawable()
                        .setColorFilter(InstabugCore.getPrimaryColor(),
                                android.graphics.PorterDuff.Mode.MULTIPLY);
            }
        }
    }

    public String getLocalizedString(@StringRes int resourceId, Context context) {
        return LocaleUtils.getLocaleStringResource(InstabugCore.getLocale(context), resourceId,
                context);
    }
}
