package com.instabug.bug.utils

import android.content.Context
import com.instabug.bug.Constants
import com.instabug.bug.di.ServiceLocator
import com.instabug.bug.model.Bug
import com.instabug.bug.testingreport.ReportUploadingStateEventBus
import com.instabug.library.Instabug
import com.instabug.library.core.InstabugCore
import com.instabug.library.internal.storage.DiskUtils
import com.instabug.library.internal.storage.cache.AttachmentsDbHelper
import com.instabug.library.internal.storage.operation.DeleteUriDiskOperation
import com.instabug.library.internal.storage.operation.DiskOperationCallback
import com.instabug.library.model.Attachment
import com.instabug.library.util.InstabugSDKLogger
import java.io.File

fun deleteBug(bug: Bug, context: Context) {
    try {
        bug
            .deleteAttachments()
            .deleteBugAndStateFile(context)
    } catch (exception: Exception) {
        InstabugCore.reportError(exception, "couldn't delete Bug ${bug.id}")
    }

}

fun Bug.deleteBugAndStateFile(context: Context) {
    state
        ?.uri
        ?.let { deleteStateFile(context) }
        ?: apply {
            InstabugSDKLogger.i(Constants.LOG_TAG, "No state file found. deleting the bug")
            delete()
            ReportUploadingStateEventBus.post(ReportUploadingStateEventBus.SENT)
        }
}

fun Bug.deleteStateFile(context: Context) {
    InstabugSDKLogger.v(
        Constants.LOG_TAG,
        "attempting to delete state file for bug with id: $id"
    )
    DiskUtils.with(context)
        .deleteOperation(DeleteUriDiskOperation(state!!.uri))
        .executeAsync(object : DiskOperationCallback<Boolean?> {
            override fun onSuccess(args: Boolean?) {
                delete()
                ReportUploadingStateEventBus.post(ReportUploadingStateEventBus.SENT)
            }

            override fun onFailure(t: Throwable) {
                InstabugSDKLogger.e(
                    Constants.LOG_TAG,
                    "Deleting attachment file failed due to: " + t.message,
                    t
                )
                ReportUploadingStateEventBus.post(ReportUploadingStateEventBus.SENT)
            }
        })
}

private fun Bug.delete() {
    id?.let {
        deleteBugFolder()
        ServiceLocator.getBugReportsDbHelper().delete(it)
    }

}

private fun Bug.deleteBugFolder() {
    BugUtils.getBugAttachmentFolder(Instabug.getApplicationContext(), id)?.let {
        File(it).delete()
    }

}

private fun Bug.deleteAttachments() = apply {
    attachments
        ?.filter { it.localPath != null }
        ?.forEach { deleteAttachment(it, id) }
}

fun deleteAttachment(attachment: Attachment, bugId: String?) {
    attachment.localPath?.let(::File)
        ?.delete()
        ?.let(::logAttachmentDeleted)
        .also { deleteAttachmentFromDb(attachment, bugId) }
}

private fun deleteAttachmentFromDb(
    attachment: Attachment,
    bugId: String?
) {
    if (attachment.id != -1L) {
        AttachmentsDbHelper.delete(attachment.id)
    } else {
        if (attachment.name != null && bugId != null) {
            AttachmentsDbHelper.delete(attachment.name, bugId)
        }
    }
}

private fun logAttachmentDeleted(isDeleted: Boolean) {
    if (isDeleted) {
        InstabugSDKLogger.v(
            Constants.LOG_TAG,
            "uploadingBugAttachmentRequest succeeded, attachment file deleted successfully"
        )
    }
}