package com.instabug.common.photopicker

import android.content.Intent
import android.net.Uri
import android.os.Bundle
import androidx.activity.ComponentActivity
import androidx.activity.result.ActivityResultLauncher
import androidx.activity.result.PickVisualMediaRequest
import androidx.activity.result.contract.ActivityResultContracts.GetContent
import androidx.activity.result.contract.ActivityResultContracts.PickVisualMedia
import androidx.activity.result.contract.ActivityResultContracts.PickVisualMedia.ImageAndVideo
import androidx.activity.result.contract.ActivityResultContracts.StartActivityForResult
import com.instabug.library._InstabugActivity
import com.instabug.library.util.StatusBarUtils

class PhotoPickerActivity : ComponentActivity(), _InstabugActivity {
    private lateinit var pickMedia: ActivityResultLauncher<PickVisualMediaRequest>
    private lateinit var legacyPickMedia: ActivityResultLauncher<Intent>
    private lateinit var fallbackPickMedia: ActivityResultLauncher<String>

    override fun onCreate(savedInstanceState: Bundle?) {
        super.onCreate(savedInstanceState)
        StatusBarUtils.setStatusBarForDialog(this)

        pickMedia = initPicker()
        legacyPickMedia = initLegacyPicker()
        fallbackPickMedia = initFallbackPicker()

        launchPhotoPicker()
    }

    private fun launchPhotoPicker() {
        try {
            pickMedia.launch(PickVisualMediaRequest(ImageAndVideo))

        } catch (ignored: Exception) {
            launchLegacyPicker()
        }
    }

    private fun launchLegacyPicker() {
        try {
            val intent = Intent(Intent.ACTION_PICK).apply {
                type = "*/*"
                putExtra(Intent.EXTRA_MIME_TYPES, arrayOf("image/*", "video/*"))
                putExtra(Intent.EXTRA_ALLOW_MULTIPLE, false)
            }

            legacyPickMedia.launch(intent)
        } catch (ignored: Exception) {
            launchFallbackPicker()
        }
    }

    private fun launchFallbackPicker() {
        try {
            fallbackPickMedia.launch("image/*")
        } catch (ignored: Exception) {
            setResult(RESULT_CANCELED)
            finish()
        }
    }

    private fun initPicker() = registerForActivityResult(PickVisualMedia(), ::onUriReady)

    private fun initLegacyPicker() = registerForActivityResult(StartActivityForResult()) { result ->
        if (result.resultCode == RESULT_OK) {
            onUriReady(result.data?.data)
        } else {
            setResult(RESULT_CANCELED)
            finish()
        }
    }

    private fun initFallbackPicker() = registerForActivityResult(GetContent(), ::onUriReady)

    private fun onUriReady(uri: Uri?) {
        setResult(RESULT_OK, Intent().setData(uri))
        finish()
    }
}
