package com.instabug.chat.cache;

import android.content.Context;
import android.net.Uri;

import androidx.annotation.NonNull;
import androidx.annotation.Nullable;

import com.instabug.chat.Constants;
import com.instabug.chat.model.Attachment;
import com.instabug.chat.model.Chat;
import com.instabug.chat.model.Message;
import com.instabug.library.apichecker.ReturnableRunnable;
import com.instabug.library.internal.storage.DiskUtils;
import com.instabug.library.internal.storage.cache.AssetsCacheManager;
import com.instabug.library.internal.storage.cache.Cache;
import com.instabug.library.internal.storage.cache.CacheManager;
import com.instabug.library.internal.storage.cache.InMemoryCache;
import com.instabug.library.model.AssetEntity;
import com.instabug.library.util.InstabugSDKLogger;
import com.instabug.library.util.threading.PoolProvider;

import java.io.File;
import java.io.IOException;
import java.util.ArrayList;
import java.util.Collections;
import java.util.List;

/**
 * @author mesbah
 */
public class ChatsCacheManager {

    public static final String CHATS_DISK_CACHE_KEY = "chats_disk_cache";
    public static final String CHATS_MEMORY_CACHE_KEY = "chats_memory_cache";
    public static final String CHATS_DISK_CACHE_FILE_NAME = "/chats.cache";

    /**
     * Loads chat cache from disk if it's not in memory
     *
     * @return in-memory cache for chatsReadQueueCacheManager.java
     * @throws IllegalArgumentException if the from cache is not found
     */
    @Nullable
    public synchronized static InMemoryCache<String, Chat> getCache() throws IllegalArgumentException {
        return PoolProvider.getChatsCacheExecutor().executeAndGet(new ReturnableRunnable<InMemoryCache<String, Chat>>() {
            @Nullable
            @Override
            public InMemoryCache<String, Chat> run() {
                if (!CacheManager.getInstance().cacheExists(CHATS_MEMORY_CACHE_KEY)) {
                    CacheManager.getInstance().migrateCache(CHATS_DISK_CACHE_KEY, CHATS_MEMORY_CACHE_KEY,
                            new CacheManager.KeyExtractor<String, Chat>() {
                                @Override
                                public String extractKey(Chat value) {
                                    return value.getId();
                                }
                            });
                }
                return (InMemoryCache<String, Chat>) CacheManager.getInstance().getCache
                        (CHATS_MEMORY_CACHE_KEY);
            }
        });
    }

    /**
     * Saves all cached chat from {@code CHATS_MEMORY_CACHE_KEY} to disk cache
     *
     * @throws IllegalArgumentException if the from cache isn't found
     */
    public static void saveCacheToDisk() throws IllegalArgumentException {
        InstabugSDKLogger.d(Constants.LOG_TAG, "Persisting chats to disk");
        PoolProvider.getChatsCacheExecutor().execute(new Runnable() {
            @Override
            public void run() {
                final Cache chatsMemoryCache = CacheManager.getInstance().getCache(CHATS_MEMORY_CACHE_KEY);
                final Cache chatsDiskCache = CacheManager.getInstance().getCache(CHATS_DISK_CACHE_KEY);

                if (chatsMemoryCache != null && chatsDiskCache != null) {
                    CacheManager.getInstance()
                            .migrateCache(chatsMemoryCache, chatsDiskCache,
                                    new CacheManager.KeyExtractor<String, Chat>() {
                                        @Override
                                        public String extractKey(Chat value) {
                                            return value.getId();
                                        }
                                    });
                    InstabugSDKLogger.d(Constants.LOG_TAG, "Chats memory cache had been persisted on-disk");
                } else {
                    InstabugSDKLogger.d(Constants.LOG_TAG, "Chats memory cache was null");
                }
            }
        });
    }

    /**
     * @return offline chat id
     */
    public static Chat addOfflineChat(Context context) {
        Chat offlineChat = new Chat.Factory().create(context);
        InMemoryCache<String, Chat> cache = getCache();
        if (cache != null) {
            cache.put(offlineChat.getId(), offlineChat);
        }
        return offlineChat;
    }

    /**
     * @param chatId chat id to find
     * @return Chat with ID matching the provided {@code chatId}
     * @throws IllegalArgumentException if no chat with this ID is found
     */
    @Nullable
    public static Chat getChat(@Nullable String chatId) {
        if (chatId == null) {
            return null;
        }
        InMemoryCache<String, Chat> cache = getCache();
        if (cache != null) {
            List<Chat> values = cache.getValues();
            for (Chat chat : values) {
                if (chat.getId() == null) {
                    continue;
                }
                if (chat.getId().equals(chatId)) {
                    return chat;
                }
            }
        }
        InstabugSDKLogger.e(Constants.LOG_TAG, "No chat with id: " + chatId + " found, returning null");
        return null;
    }

    /**
     * @return valid chats that contains messages.
     */
    public static List<Chat> getValidChats() {
        List<Chat> validChats = new ArrayList<>();

        InMemoryCache<String, Chat> cache = getCache();
        if (cache != null) {
            List<Chat> values = cache.getValues();
            for (Chat chat : values) {
                if (chat.getMessages().size() > 0) {
                    validChats.add(chat);
                }
            }
        }

        return validChats;
    }

    /**
     * @return offline chats.
     */
    public synchronized static List<Chat> getOfflineChats() {
        List<Chat> offlineChats = new ArrayList<>();
        InMemoryCache<String, Chat> cache = getCache();
        if (cache != null) {
            List<Chat> values = cache.getValues();
            for (Chat chat : values) {
                if (chat.getChatState() != null && (chat.getChatState().equals(Chat.ChatState.READY_TO_BE_SENT) ||
                        chat.getChatState().equals(Chat.ChatState.LOGS_READY_TO_BE_UPLOADED))
                        && chat.getMessages().size() > 0) {
                    offlineChats.add(chat);
                }
            }
        }

        return offlineChats;
    }

    public static void updateLocalMessageWithSyncedMessage(@NonNull Context context, @NonNull final
    Message syncedMessage) throws IOException {
        InstabugSDKLogger.v(Constants.LOG_TAG, "Updating local messages after sync");
        InMemoryCache<String, Chat> cache = getCache();
        if (cache != null && syncedMessage.getChatId() != null) {
            Chat chat = cache.get(syncedMessage.getChatId());
            if (chat != null) {
                List<Message> messages = chat.getMessages();
                for (int i = 0; i < messages.size(); i++) {
                    if (messages.get(i).getId().equals(syncedMessage.getId())
                            && messages.get(i).getMessageState().equals(Message.MessageState
                            .READY_TO_BE_SYNCED)
                            && messages.get(i).getAttachments().size() == syncedMessage
                            .getAttachments().size()) {
                        for (int j = 0; j < messages.get(i).getAttachments().size(); j++) {
                            AssetEntity assetEntity;
                            Attachment attachment = syncedMessage.getAttachments().get(j);
                            if (attachment != null && attachment.getType() != null && attachment.getUrl() != null) {
                                switch (attachment.getType()) {
                                    case Attachment.AttachmentType.TYPE_AUDIO:
                                        assetEntity = AssetsCacheManager.createEmptyEntity(context,
                                                attachment.getUrl(), AssetEntity.AssetType.AUDIO);
                                        break;
                                    case Attachment.AttachmentType.TYPE_VIDEO_GALLERY:
                                    case Attachment.AttachmentType.TYPE_VIDEO_RECORD:
                                        assetEntity = AssetsCacheManager.createEmptyEntity(context,
                                                attachment.getUrl(), AssetEntity.AssetType.VIDEO);
                                        break;
                                    default:
                                        assetEntity = AssetsCacheManager.createEmptyEntity(context,
                                                attachment.getUrl(), AssetEntity.AssetType.IMAGE);
                                        break;
                                }
                                String attachmentPath = messages.get(i).getAttachments().get(j)
                                        .getLocalPath();
                                if (attachmentPath != null) {
                                    File attachmentFile = new File(attachmentPath);
                                    DiskUtils.copyFromUriIntoFile(context, Uri.fromFile(attachmentFile),
                                            assetEntity.getFile());
                                    AssetsCacheManager.addAssetEntity(assetEntity);
                                    attachmentFile.delete();
                                }
                            }
                        }
                        chat.getMessages().set(i, syncedMessage);
                        cache.put(chat.getId(), chat);
                        break;
                    }
                }
            }
        }
    }

    /**
     * @return total number of messages received from server
     */

    public static int getTotalMessagesCount() {
        int totalMessagesCount = 0;
        InMemoryCache<String, Chat> cache = getCache();
        if (cache != null) {
            List<Chat> chats = cache.getValues();
            for (Chat chat : chats) {
                for (Message message : chat.getMessages()) {
                    if (message.getMessageState().equals(Message.MessageState.SYNCED)) {
                        totalMessagesCount++;
                    }
                }
            }
        }
        return totalMessagesCount;
    }

    /**
     * @return Offline messages.
     */
    public static List<Message> getOfflineMessages() {
        List<Message> offlineMessages = new ArrayList<>();

        InMemoryCache<String, Chat> cache = getCache();
        if (cache != null) {
            List<Chat> chats = cache.getValues();
            for (Chat chat : chats) {
                if (chat.getChatState() != null && chat.getChatState().equals(Chat.ChatState.SENT)) {
                    for (Message message : chat.getMessages()) {
                        if (message.getMessageState().equals(Message.MessageState.READY_TO_BE_SENT)
                                || message.getMessageState().equals(Message.MessageState.SENT)) {
                            offlineMessages.add(message);
                        }
                    }
                }
            }
        }

        return offlineMessages;
    }

    /**
     * @return last message received timestamp
     */
    public static long getLastMessageMessagedAt() {
        List<Message> messages = new ArrayList<>();
        InMemoryCache<String, Chat> cache = getCache();
        if (cache != null) {
            List<Chat> chats = cache.getValues();
            for (Chat chat : chats) {
                for (Message message : chat.getMessages()) {
                    if (message.getMessageState() == Message.MessageState.SYNCED) {
                        messages.add(message);
                    }
                }
            }
        }
        Collections.sort(messages, new Message.Comparator());
        for (int i = messages.size() - 1; i >= 0; i--) {
            Message message = messages.get(i);
            if (!message.getId().equals("0")) {
                return message.getMessagedAt();
            }
        }

        return 0L;
    }

    /**
     * @return current unread messages count
     */
    public static int getUnreadCount() {
        int unreadCount = 0;
        InMemoryCache<String, Chat> cache = getCache();
        if (cache != null) {
            List<Chat> chats = cache.getValues();
            for (Chat chat : chats) {
                unreadCount += chat.getUnreadCount();
            }
        }
        return unreadCount;
    }

    public static List<Message> getNotSentMessages() {
        final List<Message> notSentMessage = new ArrayList<>();

        InMemoryCache<String, Chat> cache = getCache();
        if (cache != null) {
            List<Chat> chats = cache.getValues();
            for (Chat chat : chats) {
                for (final Message message : chat.getMessages()) {
                    if (message.getMessageState() == Message.MessageState.SENT
                            || message.getMessageState() == Message.MessageState.READY_TO_BE_SENT) {
                        notSentMessage.add(message);
                    }
                }
            }
        }
        InstabugSDKLogger.v(Constants.LOG_TAG, "not sent messages count: " + notSentMessage.size());
        return notSentMessage;
    }

    public static void cleanupChats() {
        InstabugSDKLogger.v(Constants.LOG_TAG, "cleanupChats");
        InMemoryCache<String, Chat> cache = getCache();
        if (cache != null) {
            List<Chat> chats = cache.getValues();
            List<Chat> hangingChats = new ArrayList<>();
            for (Chat chat : chats) {
                if (chat.getChatState() == Chat.ChatState.WAITING_ATTACHMENT_MESSAGE)
                    hangingChats.add(chat);
            }
            deleteChatsFromCache(hangingChats, cache);
        }
        saveCacheToDisk();
    }

    public static void clearChats() {
        InMemoryCache<String, Chat> cache = getCache();
        if(cache != null) {
            deleteChatsFromCache(cache.getValues(), cache);
        }
        saveCacheToDisk();
    }

    private static void deleteChatsFromCache(List<Chat> chats ,InMemoryCache<String, Chat> cache) {
        if (cache != null) {
            for (Chat chat : chats) {
                cache.delete(chat.getId());
            }
        }
    }

    public static void migrateOnEncryptionStateChange() {
        PoolProvider.getChatsCacheExecutor().execute(new Runnable() {
            @Override
            public void run() {
                Cache chatsDiskCache = CacheManager.getInstance().getCache(CHATS_DISK_CACHE_KEY);
                if (chatsDiskCache != null) {
                    List<Chat> chats = chatsDiskCache.getValues();
                    CacheManager.KeyExtractor<String, Chat> extractor = new CacheManager.KeyExtractor<String, Chat>() {
                        @Override
                        public String extractKey(Chat value) {
                            return value.getId();
                        }
                    };

                    chatsDiskCache.invalidate();

                    for (Chat chat : chats) {
                        chatsDiskCache.put(extractor.extractKey(chat), chat);
                    }
                }
            }
        });
    }
}
