package com.instabug.chat.ui.chat;

import static android.content.Context.MEDIA_PROJECTION_SERVICE;
import static com.instabug.chat.ui.chat.ChatPresenter.MAX_FILE_SIZE_IN_MB;
import static com.instabug.library.InstabugCustomTextPlaceHolder.Key;

import android.Manifest;
import android.annotation.SuppressLint;
import android.content.Context;
import android.content.Intent;
import android.content.pm.PackageManager;
import android.graphics.drawable.Drawable;
import android.media.projection.MediaProjectionConfig;
import android.media.projection.MediaProjectionManager;
import android.net.Uri;
import android.os.Build;
import android.os.Bundle;
import android.text.InputType;
import android.text.TextUtils;
import android.view.View;
import android.widget.EditText;
import android.widget.ImageButton;
import android.widget.ImageView;
import android.widget.ListView;
import android.widget.Toast;

import androidx.annotation.NonNull;
import androidx.annotation.Nullable;
import androidx.core.content.ContextCompat;
import androidx.core.view.AccessibilityDelegateCompat;
import androidx.core.view.ViewCompat;
import androidx.core.view.accessibility.AccessibilityNodeInfoCompat;
import androidx.fragment.app.FragmentActivity;
import androidx.fragment.app.FragmentTransaction;

import com.instabug.bug.R;
import com.instabug.bug.internal.video.VideoPlayerFragment;
import com.instabug.chat.Constants;
import com.instabug.chat.cache.ChatsCacheManager;
import com.instabug.chat.model.Attachment;
import com.instabug.chat.model.Chat;
import com.instabug.chat.model.Message;
import com.instabug.chat.screenrecording.ExternalScreenRecordHelper;
import com.instabug.chat.ui.annotation.ChatAnnotationFragment;
import com.instabug.common.photopicker.PhotoPickerActivity;
import com.instabug.chat.ui.base.ToolbarFragment;
import com.instabug.library.internal.InstabugMediaProjectionIntent;
import com.instabug.library.internal.video.RequestPermissionActivityLauncher;
import com.instabug.library.ui.custom.InstabugAlertDialog;
import com.instabug.library.util.Colorizer;
import com.instabug.library.util.InstabugSDKLogger;
import com.instabug.library.util.PlaceHolderUtils;
import com.instabug.library.util.SystemServiceUtils;
import com.instabug.library.util.ThemeApplier;

import org.jetbrains.annotations.NotNull;

import java.util.ArrayList;
import java.util.List;

@SuppressLint({"ERADICATE_FIELD_NOT_INITIALIZED", "ERADICATE_FIELD_NOT_NULLABLE"})
public class ChatFragment extends ToolbarFragment<ChatContract.Presenter> implements ChatContract.View,
        View.OnClickListener, MessagesListAdapter.OnMessageItemClickListener,
        AttachmentsBottomSheetFragment.Callbacks, ChatAnnotationFragment.Callbacks {

    public static final String TAG = "chat_fragment";
    public static final int PICK_IMAGE_REQUEST_CODE = 0xA1;
    private static final int REQUEST_SCREEN_RECORDING_MIC_PERMISSION = 0xA3;
    private static final String EXTRA_CHAT = "chat_number";
    private static final String EXTRA_ATTACHMENT = "attachment";

    private String chatNumber;
    private MessagesListAdapter messagesListAdapter;
    private EditText messageEditText;
    private String chatTitle;
    @Nullable
    ImageButton backToConversationsToolbarLeftButton;
    @Nullable
    ImageView attachImageView;

    public static ChatFragment newInstance(String chatId) {
        ChatFragment fragment = new ChatFragment();
        Bundle bundle = new Bundle();
        bundle.putString(EXTRA_CHAT, chatId);
        fragment.setArguments(bundle);
        return fragment;
    }

    public static ChatFragment newInstance(String chatId, Attachment attachment) {
        ChatFragment fragment = new ChatFragment();
        Bundle bundle = new Bundle();
        bundle.putString(EXTRA_CHAT, chatId);
        bundle.putSerializable(EXTRA_ATTACHMENT, attachment);
        fragment.setArguments(bundle);
        return fragment;
    }

    @Override
    public void onCreate(@Nullable Bundle savedInstanceState) {
        super.onCreate(savedInstanceState);
        setRetainInstance(true);
        if (getArguments() != null) {
            chatNumber = getArguments().getString(EXTRA_CHAT);
        }
        presenter = new ChatPresenter(this);
    }

    @Override
    public void onViewCreated(@NotNull View view, @Nullable Bundle savedInstanceState) {
        super.onViewCreated(view, savedInstanceState);
        if (presenter != null) {
            presenter.init(chatNumber);
        }

        ViewCompat.setAccessibilityDelegate(view, new AccessibilityDelegateCompat() {

            @Override
            public void onInitializeAccessibilityNodeInfo(View host, AccessibilityNodeInfoCompat info) {
                super.onInitializeAccessibilityNodeInfo(host, info);
                if (chatTitle != null) {
                    info.setContentDescription(getLocalizedString(R.string.ibg_chat_conversation_with_name_content_description, chatTitle));
                } else {
                    info.setContentDescription(getLocalizedString(R.string.ibg_chat_conversation_content_description));
                }
            }
        });
    }

    @Override
    public void onStart() {
        super.onStart();
        if (presenter != null) {
            presenter.listen();
        }
        // check if there is sent attachment
        Attachment sentAttachment = null;
        if (getArguments() != null) {
            sentAttachment = (Attachment) getArguments().getSerializable(EXTRA_ATTACHMENT);
        }
        if (sentAttachment != null && presenter != null) {
            presenter.handleSentAttachment(sentAttachment);
        }
        // avoid duplicate sent attachment
        if (getArguments() != null) {
            getArguments().clear();
        }
    }

    @Override
    public void onStop() {
        super.onStop();
        if (presenter != null) {
            presenter.sleep();
        }
    }

    @Override
    public void onDestroyView() {
        super.onDestroyView();
        if (presenter != null) {
            presenter.cleanOfflineChats();
        }

        messageEditText = null;
        attachImageView = null;
        backToConversationsToolbarLeftButton = null;
    }

    @Override
    protected int getContentLayout() {
        return R.layout.instabug_fragment_chat;
    }

    @Override
    @SuppressLint("ERADICATE_PARAMETER_NOT_NULLABLE")
    protected void initContentViews(View rootView, @Nullable Bundle savedInstanceState) {
        ImageButton toolbarRight = rootView.findViewById(R.id.instabug_btn_toolbar_right);
        if (toolbarRight != null) {
            toolbarRight.setVisibility(View.GONE);
        }
        ListView messagesListView = rootView.findViewById(R.id.instabug_lst_messages);
        messageEditText = rootView.findViewById(R.id.instabug_edit_text_new_message);
        if (messageEditText != null) {
            messageEditText.setHint(PlaceHolderUtils.getPlaceHolder(Key.CONVERSATION_TEXT_FIELD_HINT,
                    getLocalizedString(R.string.instabug_str_sending_message_hint)));
            messageEditText.setInputType(InputType.TYPE_CLASS_TEXT | InputType.TYPE_TEXT_FLAG_CAP_SENTENCES);
            ThemeApplier.applyEditTextStyle(messageEditText, getIBGTheme());
        }

        ImageView sendImageButton = rootView.findViewById(R.id.instabug_btn_send);
        if (getContext() != null) {
            Drawable sendImageButtonDrawable = Colorizer.getPrimaryColorTintedDrawable(ContextCompat
                    .getDrawable(getContext(), R.drawable.ibg_chat_ic_send));
            if (sendImageButton != null) {
                sendImageButton.setImageDrawable(sendImageButtonDrawable);
                sendImageButton.setContentDescription(getLocalizedString(R.string.ibg_chat_send_message_btn_content_description));
            }
        }
        if (sendImageButton != null) {
            sendImageButton.setOnClickListener(this);
        }
        messagesListAdapter = new MessagesListAdapter(
                new ArrayList<>(),
                getActivity(),
                messagesListView,
                this,
                getIBGTheme()
        );
        if (messagesListView != null) {
            messagesListView.setAdapter(messagesListAdapter);
        }
        // set a tag for the close button for testing purposes
        backToConversationsToolbarLeftButton =
                rootView.findViewById(R.id.instabug_btn_toolbar_left);
        if (backToConversationsToolbarLeftButton != null) {
            backToConversationsToolbarLeftButton.setTag(R.id.TAG_BTN_BACK, "instabug_btn_back");
            backToConversationsToolbarLeftButton.setContentDescription(getLocalizedString(R.string.ibg_chat_back_to_conversations_btn_content_description));
        }

        attachImageView = rootView.findViewById(R.id.instabug_btn_attach);
        if (attachImageView != null) {
            attachImageView.setContentDescription(getLocalizedString(R.string.ibg_chat_add_attachment_btn_content_description));
        }
    }

    @Override
    protected String getTitle() {
        Chat chat = ChatsCacheManager.getChat(chatNumber);
        if (chat != null) {
            chatTitle = chat.getTitle();
            return chatTitle;
        } else {
            return getLocalizedString(R.string.instabug_str_empty);
        }
    }

    @Override
    protected void onDoneButtonClicked() {
    }

    @Override
    public void onClick(View view) {
        if (view.getId() == R.id.instabug_btn_send) {
            String messageBody = messageEditText.getText().toString();
            if (TextUtils.isEmpty(messageBody.trim())) {
                return;
            }
            if (presenter != null) {
                presenter.sendMessage(presenter.createMessage(presenter.getChat().getId(), messageBody));
            }
            messageEditText.setText("");
        } else if (view.getId() == R.id.instabug_btn_attach && getActivity() != null) {
            SystemServiceUtils.hideInputMethod(this.getActivity());
            openAttachmentBottomSheetFragment();
        }
    }

    @Override
    public void onVideoAttachmentClicked(String videoPath) {
        if (getActivity() != null) {
            SystemServiceUtils.hideInputMethod(getActivity());
            FragmentTransaction transaction = getActivity().getSupportFragmentManager().beginTransaction();
            transaction.add(R.id.instabug_fragment_container, VideoPlayerFragment.newInstance
                            (videoPath), VideoPlayerFragment.TAG)
                    .addToBackStack(VideoPlayerFragment.TAG)
                    .commit();
        }
    }

    @Override
    public void onImageAttachmentClicked(String imagePath) {
        if (getActivity() != null) {
            SystemServiceUtils.hideInputMethod(getActivity());
            FragmentTransaction transaction = getActivity().getSupportFragmentManager().beginTransaction();
            transaction.add(R.id.instabug_fragment_container, ImageAttachmentViewerFragment
                            .newInstance(imagePath), ImageAttachmentViewerFragment.TAG)
                    .addToBackStack(ImageAttachmentViewerFragment.TAG)
                    .commit();
        }
    }

    @Override
    public void onMessageActionClicked(String url) {
        try {
            Intent i = new Intent(Intent.ACTION_VIEW);
            i.setData(Uri.parse(url));
            startActivity(i);
        } catch (Exception e) {
            InstabugSDKLogger.e(Constants.LOG_TAG, "Unable to view this url " + url + "\n"
                    + "Error message: " + e.getMessage());
        }
    }

    @Override
    public void attachScreenshot() {
        if (presenter != null) {
            presenter.attachScreenshot();
        }
    }

    @Override
    public void attachImageFromGallery() {
        pickPhotoFromGallery();
    }

    @Override
    public void attachVideo() {
        recordScreen();
    }

    public void recordScreen() {
        //checks if the internal screen recorder is busy or not
        if (!ExternalScreenRecordHelper.getInstance().isRecording()) {
            requestAudioPermission();
        } else {
            Context currentContext = getContext();
            if (currentContext != null) {
                Toast.makeText(currentContext, R.string.instabug_str_video_encoder_busy, Toast
                        .LENGTH_SHORT).show();
            }
        }
    }

    private void requestAudioPermission() {
        if (getActivity() != null) {
            if (ContextCompat.checkSelfPermission(getActivity(),
                    Manifest.permission.RECORD_AUDIO)
                    != PackageManager.PERMISSION_GRANTED) {

                requestPermissions(
                        new String[]{Manifest.permission.RECORD_AUDIO,},
                        REQUEST_SCREEN_RECORDING_MIC_PERMISSION);
            } else {
                requestVideoPermissions();
            }
        }
    }

    private void requestVideoPermissions() {
        if (getActivity() != null && Build.VERSION.SDK_INT >= Build.VERSION_CODES.LOLLIPOP) {
            if (InstabugMediaProjectionIntent.isMediaProjectionIntentReadyToUse()) {
                Intent mediaProjectionIntent = InstabugMediaProjectionIntent.getMediaProjectionIntent();
                if (presenter != null && mediaProjectionIntent != null) {
                    presenter.openVideoRecorder(mediaProjectionIntent);
                }
            } else if (InstabugMediaProjectionIntent.canStartMediaProjectionFGService(getActivity())) {
                MediaProjectionManager manager = (MediaProjectionManager)
                        getActivity().getSystemService(MEDIA_PROJECTION_SERVICE);
                if (manager != null) {
                    Intent intent;

                    if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.UPSIDE_DOWN_CAKE)
                        intent = manager.createScreenCaptureIntent(MediaProjectionConfig.createConfigForDefaultDisplay());
                    else
                        intent = manager.createScreenCaptureIntent();
                    startActivityForResult(intent, ChatPresenter
                            .REQUEST_SCREEN_RECORDING_CAPTURE_PERMISSIONS);
                }
            }
        }
    }

    @Override
    public void onImageEditingDone(String chatId, Uri annotatedImageUri, @Attachment.AttachmentType String attachmentType) {
        if (presenter != null) {
            if (chatId != null && chatId.equals(presenter.getChat().getId())) {
                presenter.sendMessage(presenter.createAttachmentMessage(presenter.getChat().getId(),
                        presenter.createImageAttachment(annotatedImageUri, attachmentType)));
            }
        }
    }

    @Override
    public void onImageEditingCancelled(String chatId, Uri baseImageUri) {
        FragmentActivity activity = getActivity();
        if (activity != null) {
            activity.onBackPressed();
        }
    }

    @Override
    public void onActivityResult(int requestCode, int resultCode, Intent data) {
        super.onActivityResult(requestCode, resultCode, data);
        if (presenter != null) {
            presenter.onActivityResult(requestCode, resultCode, data);
        }
    }

    @Override
    public void onRequestPermissionsResult(int requestCode, @NonNull String[] permissions, @NonNull int[] grantResults) {
        if (grantResults.length > 0 && grantResults[0] == PackageManager.PERMISSION_GRANTED) {
            if (requestCode == REQUEST_SCREEN_RECORDING_MIC_PERMISSION) {
                requestVideoPermissions();
            } else {
                super.onRequestPermissionsResult(requestCode, permissions, grantResults);
            }
        } else {
            if (requestCode == REQUEST_SCREEN_RECORDING_MIC_PERMISSION) {
                requestVideoPermissions();
            }
        }
    }

    @Override
    public void makeLeftTopButtonCloseButton() {
        if (backToConversationsToolbarLeftButton != null) {
            backToConversationsToolbarLeftButton.setImageResource(R
                    .drawable.ibg_core_ic_close);
        }
    }

    @Override
    public void makeLeftTopButtonBackButton() {
        if (backToConversationsToolbarLeftButton != null) {
            backToConversationsToolbarLeftButton.setImageResource(R
                    .drawable.ibg_core_ic_back);
            backToConversationsToolbarLeftButton.setRotation(getResources()
                    .getInteger(R.integer.instabug_icon_lang_rotation));
        }
    }

    @Override
    public void showAttachmentButton() {
        if (rootView != null) {
            if (attachImageView != null) {
                Colorizer.applyPrimaryColorTint(attachImageView);
                attachImageView.setOnClickListener(this);
            }
        }
    }

    @Override
    public void hideAttachmentButton() {
        if (attachImageView != null) {
            attachImageView.setVisibility(View.GONE);
        }
    }

    @Override
    public void updateChatMessages(List<Message> messages) {
        if (presenter != null) {
            messagesListAdapter.setFlatMessageItems(presenter.flattenMessages(messages));
        }
    }

    @Override
    public void notifyUIForUpdates() {
        messagesListAdapter.notifyDataSetChanged();
    }

    @Override
    public void annotateAttachmentImage(Uri imageUri, String attachmentType) {
        ChatContract.Presenter presenter = this.presenter;
        if (getActivity() != null && presenter != null) {
            FragmentTransaction transaction = getActivity().getSupportFragmentManager()
                    .beginTransaction();
            transaction.add(R.id.instabug_fragment_container,
                            ChatAnnotationFragment.newInstance(presenter.getChat().getTitle(),
                                    presenter.getChat().getId(), imageUri, attachmentType), ChatAnnotationFragment.TAG)
                    .addToBackStack(ChatAnnotationFragment.TAG)
                    .commit();
        }
        this.presenter = presenter;
    }

    @Override
    public void pickImageFromGallery() {
        Context context = getContext();
        if (context != null) {
            startActivityForResult(getPhotoPickerIntent(context), PICK_IMAGE_REQUEST_CODE);
        }
    }

    private void openAttachmentBottomSheetFragment() {
        if (getActivity() != null && presenter != null) {
            FragmentTransaction transaction = getActivity().getSupportFragmentManager().beginTransaction();
            transaction.add(R.id.instabug_fragment_container,
                            AttachmentsBottomSheetFragment.newInstance(this), AttachmentsBottomSheetFragment.TAG)
                    .addToBackStack(AttachmentsBottomSheetFragment.TAG)
                    .commit();
        }
    }

    private void pickPhotoFromGallery() {
        if (presenter != null) {
            presenter.startImageFromGalleryPicker();
        }
    }


    private Intent getPhotoPickerIntent(@NonNull Context context) {
        return new Intent(context, PhotoPickerActivity.class);
    }

    @Override
    public void showMediaFileSizeAlert() {
        if (getActivity() != null) {
            new InstabugAlertDialog.Builder(getActivity())
                    .setTitle(getLocalizedString(R.string.instabug_str_bugreport_file_size_limit_warning_title))
                    .setMessage(getLocalizedString(R.string.instabug_str_bugreport_file_size_limit_warning_message,
                            MAX_FILE_SIZE_IN_MB))
                    .setPositiveButton(getLocalizedString(R.string.instabug_str_ok), (dialogInterface, i) -> {
                        dialogInterface.dismiss();
                    }).show();
        }
    }

    @Override
    public void showVideoLengthAlert() {
        if (getActivity() != null) {
            new InstabugAlertDialog.Builder(getActivity())
                    .setTitle(getLocalizedString(com.instabug.library.R.string.instabug_str_video_length_limit_warning_title))
                    .setMessage(getLocalizedString(com.instabug.library.R.string.instabug_str_video_length_limit_warning_message))
                    .setPositiveButton(getLocalizedString(R.string.instabug_str_ok), (dialogInterface, i) -> {
                        dialogInterface.dismiss();
                    }).show();
        }
    }

    @Override
    public void requestMediaProjectionPermission() {
        if (getActivity() != null) {
            RequestPermissionActivityLauncher.start(getActivity(), false, false, null);
        }
    }
}