package com.instabug.bug;

import static com.instabug.bug.BugReportingWrapper.setReportTypes;

import android.content.Context;

import androidx.annotation.VisibleForTesting;

import com.instabug.bug.Constants.ReportType;
import com.instabug.bug.di.ServiceLocator;
import com.instabug.bug.reportingpromptitems.AskQuestionPromptItem;
import com.instabug.bug.reportingpromptitems.BugReportingPromptItem;
import com.instabug.bug.reportingpromptitems.FeedbackPromptItem;
import com.instabug.bug.settings.BugSettings;
import com.instabug.library.Feature;
import com.instabug.library.IBGFeature;
import com.instabug.library.OnSdkDismissCallback;
import com.instabug.library.core.InstabugCore;
import com.instabug.library.core.plugin.PluginPromptOption;
import com.instabug.library.internal.orchestrator.ActionsOrchestrator;

import java.util.ArrayList;

/**
 * Created by mohamedzakaria on 6/5/17.
 */

public class ReportingPluginWrapper {


    public static void init(final Context context) {
        BugSettings.init(context);
        checkEncryptorVersion();
    }

    static ArrayList<PluginPromptOption> getPromptOptions(Context context) {
        ArrayList<PluginPromptOption> bugPluginPromptOptions = new ArrayList<>();

        if (InstabugCore.isFeatureEnabled(IBGFeature.BUG_REPORTING)) {
            if (isReportTypeEnabled(ReportType.BUG))
                bugPluginPromptOptions.add(new BugReportingPromptItem().getItemOption(context));
            if (isReportTypeEnabled(ReportType.FEEDBACK))
                bugPluginPromptOptions.add(new FeedbackPromptItem().getItemOption(context));
            if (isReportTypeEnabled(ReportType.ASK_QUESTION)) {
                addAskQuestionOption(bugPluginPromptOptions, context);
            }
        }
        return bugPluginPromptOptions;
    }


    /**
     * A method to add ask question option only if
     * 1- Messaging feature is enabled and bug reporting is disabled but chats is enabled
     * 2- Messaging feature is enabled and bug reporting is enabled with report type question
     *
     * @param bugPluginPromptOptions array list to be added
     * @param context                context.
     */
    private static void addAskQuestionOption(ArrayList<PluginPromptOption> bugPluginPromptOptions,
                                             Context context) {
        if (!isMessagingEnabled()) return;
        if (isAskQuestionEnabled()) {
            bugPluginPromptOptions.add(new AskQuestionPromptItem().getItemOption(context));
        }

    }


    private static boolean isReportTypeEnabled(@ReportType String reportType) {
        return BugSettings.getInstance().getReportTypeStatus(reportType);
    }

    /**
     * This method to return all available options in the product
     *
     * @param ignoreBaseFeature if true then ignore state of bug reporting else take it into consideration.
     * @param context
     * @return
     */
    static ArrayList<PluginPromptOption> getReportingOptions(boolean ignoreBaseFeature, Context context) {
        ArrayList<PluginPromptOption> bugPluginPromptOptions = new ArrayList<>();

        if (!ignoreBaseFeature && InstabugCore.isFeatureEnabled(IBGFeature.BUG_REPORTING)) {
            bugPluginPromptOptions.add(new BugReportingPromptItem().getItemOption(context));
            bugPluginPromptOptions.add(new FeedbackPromptItem().getItemOption(context));
            bugPluginPromptOptions.add(new AskQuestionPromptItem().getItemOption(context));

        } else if (ignoreBaseFeature) {
            bugPluginPromptOptions.add(new BugReportingPromptItem().getItemOption(context));
            bugPluginPromptOptions.add(new FeedbackPromptItem().getItemOption(context));
            bugPluginPromptOptions.add(new AskQuestionPromptItem().getItemOption(context));
        }
        return bugPluginPromptOptions;
    }


    private static boolean isMessagingEnabled() {
        return InstabugCore.getFeatureState(IBGFeature.IN_APP_MESSAGING) == Feature.State.ENABLED;
    }


    /**
     * return user customization for now it will check only status of chat's feature
     * The current implementation that whenever the user change the report type chat state get changed as well,
     * this will be replaced with only report type enabled condition, after removing deprecated chats api
     *
     * @return user customization
     */
    private static boolean isAskQuestionEnabled() {
        return BugSettings.getInstance().getReportTypeStatus(ReportType.ASK_QUESTION);
    }


    public static void release() {

    }


    @VisibleForTesting
    static boolean isInstabugFeatureEnabled() {
        return InstabugCore.getFeatureState(IBGFeature.INSTABUG) == Feature.State.ENABLED;
    }


    public static OnSdkDismissCallback.DismissType getDismissType(OnSdkDismissedCallback.DismissType dismissType) {
        switch (dismissType) {
            default:
            case CANCEL:
                return OnSdkDismissCallback.DismissType.CANCEL;
            case SUBMIT:
                return OnSdkDismissCallback.DismissType.SUBMIT;
            case ADD_ATTACHMENT:
                return OnSdkDismissCallback.DismissType.ADD_ATTACHMENT;
        }
    }

    public static OnSdkDismissCallback.ReportType getReportType(@ReportType String dismissType) {
        switch (dismissType) {
            default:
            case ReportType.BUG:
                return OnSdkDismissCallback.ReportType.BUG;
            case ReportType.FEEDBACK:
                return OnSdkDismissCallback.ReportType.FEEDBACK;
            case ReportType.ASK_QUESTION:
                return OnSdkDismissCallback.ReportType.QUESTION;
            case ReportType.FRUSTRATING_EXPERIENCE:
                return OnSdkDismissCallback.ReportType.FRUSTRATING_EXPERIENCE;
            case ReportType.NOT_AVAILABLE:
                return OnSdkDismissCallback.ReportType.OTHER;
        }
    }

    public static void initDefaultPromptOptionAvailabilityState() {
        setReportTypes(BugReporting.ReportType.BUG, BugReporting.ReportType.FEEDBACK,
                BugReporting.ReportType.QUESTION);
    }

    /*
     * Due to the latest changes in Encryption/Decryption processing (Previously the process was done for the whole
     * file vs now the Process is done for the first 256 bytes only no matter how big the file is).
     * It was decided to drop Bugs BD table */
    private static void checkEncryptorVersion() {
        if (BugSettings.getInstance().isFirstRunAfterEncryptorUpdate()) {
            dropBugsDatabaseTable();
        }
    }

    private static void dropBugsDatabaseTable() {
        ActionsOrchestrator.obtainOrchestrator()
                .addWorkerThreadAction(
                        () -> {
                            ServiceLocator.getBugReportsDbHelper().deleteAll();
                            BugSettings.getInstance().setFirstRunAfterEncryptorUpdate(false);
                        }
                ).orchestrate();
    }
}