package com.instabug.bug.view.extrafields;

import android.annotation.SuppressLint;

import androidx.annotation.Nullable;
import androidx.annotation.VisibleForTesting;

import com.instabug.bug.LiveBugManager;
import com.instabug.bug.extendedbugreport.ExtendedBugReport;
import com.instabug.bug.model.ExtraReportField;
import com.instabug.bug.settings.BugSettings;
import com.instabug.library.core.ui.BasePresenter;

import org.json.JSONArray;
import org.json.JSONException;
import org.json.JSONObject;

import java.util.List;

import static com.instabug.bug.extendedbugreport.ExtendedBugReport.State.ENABLED_WITH_OPTIONAL_FIELDS;
import static com.instabug.bug.extendedbugreport.ExtendedBugReport.State.ENABLED_WITH_REQUIRED_FIELDS;


class ExtraFieldsPresenter extends BasePresenter<ExtraFieldsContract.View> implements
        ExtraFieldsContract.Presenter {


    @VisibleForTesting
    static final String KEY_ID = "id";
    @VisibleForTesting
    static final String KEY_NAME = "name";
    @VisibleForTesting
    static final String KEY_VALUE = "value";
    @VisibleForTesting
    static final String FIRST_COMMENT_ID = "description";
    @VisibleForTesting
    static final String FIRST_COMMENT_NAME = "Description";

    ExtraFieldsPresenter(ExtraFieldsContract.View view) {
        super(view);
    }

    @Override
    @Nullable
    public List<ExtraReportField> getExtraReportFields() {
        if (LiveBugManager.getInstance().getBug() == null) {
            return null;
        }
        List<ExtraReportField> extraReportFields = LiveBugManager.getInstance().getBug()
                .getExtraReportFields();

        if (extraReportFields != null) {
            return extraReportFields;
        }

        ExtendedBugReport.State state = BugSettings.getInstance().getExtendedBugReportState();

        switch (state) {
            case ENABLED_WITH_OPTIONAL_FIELDS:
            case ENABLED_WITH_REQUIRED_FIELDS:
                ExtraFieldsContract.View viewRef = view.get();
                if (viewRef != null && viewRef.getViewContext() != null &&
                        viewRef.getViewContext().getContext() != null) {
                    extraReportFields = ExtendedBugReport.getFields(viewRef.getViewContext()
                            .getContext(), state);
                }
                break;
            case DISABLED:
            default:
                extraReportFields = BugSettings.getInstance().getExtraReportFields();
                break;
        }

        LiveBugManager.getInstance().getBug().setExtraReportFields(extraReportFields);

        return extraReportFields;

    }

    @Override
    public void appendExtraReportFieldsToMessage(List<ExtraReportField> extraReportFields) {

        ExtendedBugReport.State state = BugSettings.getInstance().getExtendedBugReportState();

        if (state == ENABLED_WITH_OPTIONAL_FIELDS || state == ENABLED_WITH_REQUIRED_FIELDS) {
            appendExtraReportFieldsToMessageAsJson(extraReportFields);
        } else {
            appendExtraReportFieldsToMessageAsString(extraReportFields);
        }
    }

    @VisibleForTesting
    void appendExtraReportFieldsToMessageAsJson(List<ExtraReportField> extraReportFields) {
        if (LiveBugManager.getInstance().getBug() != null) {
            String message = LiveBugManager.getInstance().getBug().getMessage();
            JSONArray extraFieldsJSONArray = getExtraReportFieldsJsonArray(message, extraReportFields);
            LiveBugManager.getInstance().getBug().setMessage(extraFieldsJSONArray.toString());
            invalidateValuesOfExtraReports();
        }
    }

    @VisibleForTesting
    @SuppressLint("ERADICATE_PARAMETER_NOT_NULLABLE")
    JSONArray getExtraReportFieldsJsonArray(@Nullable String message, List<ExtraReportField> extraReportFields) {
        JSONArray extraFieldsJSONArray = new JSONArray();
        try {
            JSONObject descriptionJSONObject = new JSONObject();
            descriptionJSONObject.put(KEY_ID, FIRST_COMMENT_ID);
            descriptionJSONObject.put(KEY_NAME, FIRST_COMMENT_NAME);
            descriptionJSONObject.put(KEY_VALUE, message != null ? message : "");
            extraFieldsJSONArray.put(descriptionJSONObject);

            for (ExtraReportField extraReportField : extraReportFields) {
                JSONObject extraFieldJSONObject = new JSONObject();
                extraFieldJSONObject.put(KEY_ID, extraReportField.getId());
                extraFieldJSONObject.put(KEY_NAME, extraReportField.getUnlocalizedHint());
                extraFieldJSONObject.put(KEY_VALUE,
                        extraReportField.getValue() != null ? extraReportField.getValue() : "");
                extraFieldsJSONArray.put(extraFieldJSONObject);
            }

        } catch (JSONException e) {
            e.printStackTrace();
        }
        return extraFieldsJSONArray;
    }

    private void invalidateValuesOfExtraReports() {
        List<ExtraReportField> extraReportFields = BugSettings.getInstance().getExtraReportFields();
        for (ExtraReportField field : extraReportFields) {
            field.setValue(null);
        }
    }

    @VisibleForTesting
    void appendExtraReportFieldsToMessageAsString(List<ExtraReportField>
                                                          extraReportFields) {
        if (LiveBugManager.getInstance().getBug() != null) {
            String message = LiveBugManager.getInstance().getBug().getMessage();
            StringBuilder messageStringBuilder = new StringBuilder();
            if (message != null) {
                messageStringBuilder.append(message);
            }

            for (ExtraReportField extraReportField : extraReportFields) {
                if (messageStringBuilder.length() > 0) {
                    messageStringBuilder.append("\n");
                }
                messageStringBuilder.append(extraReportField.getHint());
                messageStringBuilder.append(":");
                messageStringBuilder.append("\n");
                messageStringBuilder.append(extraReportField.getValue());
            }
            LiveBugManager.getInstance().getBug().setMessage(messageStringBuilder.toString());
            invalidateValuesOfExtraReports();
        }
    }

    @Override
    public boolean validateExtraReportFields() {
        if (LiveBugManager.getInstance().getBug() == null) {
            return false;
        }
        List<ExtraReportField> extraReportFields = LiveBugManager.getInstance().getBug()
                .getExtraReportFields();
        if (extraReportFields != null && !extraReportFields.isEmpty()) {
            clearErrors(extraReportFields);
        }
        ExtraFieldsContract.View viewRef = view.get();
        if (viewRef != null) {
            for (int i = 0; (extraReportFields != null && i < extraReportFields.size()); i++) {
                ExtraReportField extraReportField = extraReportFields.get(i);
                if (extraReportField.isRequired()) {
                    if (extraReportField.getValue() == null) {
                        viewRef.focusAndSetError(i);
                        return false;
                    } else {
                        String trimmedValue = extraReportField.getValue().trim();
                        if (trimmedValue.isEmpty()) {
                            viewRef.focusAndSetError(i);
                            return false;
                        }
                    }
                }
            }
        }
        return true;
    }

    @VisibleForTesting
    void clearErrors(List<ExtraReportField> extraReportFields) {
        ExtraFieldsContract.View viewRef = view.get();
        if (viewRef != null) {
            for (int i = 0; i < extraReportFields.size(); i++) {
                viewRef.clearError(i);
            }
        }
    }

}
