package com.instabug.bug;

import static com.instabug.bug.utils.DeleteBugsUtilKt.deleteAttachment;
import static com.instabug.library.core.InstabugCore.getFeatureState;
import static com.instabug.library.internal.storage.DiskUtils.deleteFile;
import static com.instabug.library.util.StateKtxKt.dropLogs;

import android.content.Context;
import android.content.Intent;
import android.net.Uri;

import androidx.annotation.NonNull;
import androidx.annotation.Nullable;
import androidx.annotation.VisibleForTesting;
import androidx.annotation.WorkerThread;
import androidx.localbroadcastmanager.content.LocalBroadcastManager;

import com.instabug.bug.di.ServiceLocator;
import com.instabug.bug.model.Bug;
import com.instabug.bug.network.NormalBugsUploaderJob;
import com.instabug.bug.network.ProactiveReportsBugsUploaderJob;
import com.instabug.bug.settings.BugSettings;
import com.instabug.bug.testingreport.ReportUploadingStateEventBus;
import com.instabug.bug.userConsent.ActionType;
import com.instabug.bug.utils.BugUtils;
import com.instabug.bug.utils.DeleteBugsUtilKt;
import com.instabug.bug.view.reporting.BaseReportingPresenter;
import com.instabug.chat.synchronization.SynchronizationManager;
import com.instabug.library.Feature;
import com.instabug.library.IBGFeature;
import com.instabug.library.Instabug;
import com.instabug.library.core.InstabugCore;
import com.instabug.library.frustratingexperience.FrustratingExperienceEvent;
import com.instabug.library.frustratingexperience.FrustratingExperienceEventBus;
import com.instabug.library.internal.orchestrator.Action;
import com.instabug.library.internal.orchestrator.ActionsOrchestrator;
import com.instabug.library.internal.servicelocator.CoreServiceLocator;
import com.instabug.library.internal.storage.AttachmentsUtility;
import com.instabug.library.internal.storage.DiskUtils;
import com.instabug.library.internal.storage.cache.db.userAttribute.UserAttributeCacheManager;
import com.instabug.library.internal.storage.operation.WriteStateToFileDiskOperation;
import com.instabug.library.logging.InstabugLog;
import com.instabug.library.logging.InstabugUserEventLogger;
import com.instabug.library.model.Attachment;
import com.instabug.library.model.Report;
import com.instabug.library.model.State;
import com.instabug.library.settings.SettingsManager;
import com.instabug.library.user.UserEvent;
import com.instabug.library.util.BitmapUtils;
import com.instabug.library.util.FileUtils;
import com.instabug.library.util.InstabugSDKLogger;
import com.instabug.library.util.ReportHelper;
import com.instabug.library.util.memory.MemoryUtils;
import com.instabug.library.util.threading.PoolProvider;
import com.instabug.library.visualusersteps.VisualUserStepsHelper;

import org.json.JSONArray;
import org.json.JSONException;

import java.io.File;
import java.io.IOException;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;

import io.reactivexport.android.schedulers.AndroidSchedulers;
import io.reactivexport.disposables.Disposable;

/**
 * Created by mNagy on 4/27/16.
 */
public class LiveBugManager {

    public static final String REFRESH_ATTACHMENTS = "refresh.attachments";
    public static final String BUG_STATE = "bug_state";
    private static LiveBugManager instance;
    public static final int DURATION_NOT_EXISTS = -1;
    @Nullable
    private volatile Bug bug;
    private boolean isInBackground;
    private OnSdkDismissedCallback.DismissType dismissType = OnSdkDismissedCallback.DismissType.CANCEL;

    private int recordingDuration = DURATION_NOT_EXISTS;

    private LiveBugManager() {
    }

    public synchronized static LiveBugManager getInstance() {
        if (instance == null) {
            instance = new LiveBugManager();
        }
        return instance;
    }

    public void releaseBug() {
        this.bug = null;
    }

    public void removeBug() {
        if (bug != null && bug.getAttachments() != null) {
            for (Attachment attachment : bug.getAttachments()) {
                if (attachment.getLocalPath() != null) {
                    deleteFile(attachment.getLocalPath());
                }
            }
        }
        releaseBug();
    }

    @Nullable
    public Bug getBug() {
        return bug;
    }

    public void setBug(Bug bug) {
        this.bug = bug;
        this.isInBackground = false;
        this.dismissType = OnSdkDismissedCallback.DismissType.CANCEL;
    }

    public void createBug(Context context) {
        Bug currentBug = bug;
        if (currentBug == null) {
            currentBug = new Bug.Factory().create(context);
            currentBug.setAttachmentsPath(BugUtils.getBugAttachmentFolder(context, currentBug.getId()));
            setBug(currentBug);
            startStateBuilder(context);
        }
    }

    private void startStateBuilder(final Context context) {
        ActionsOrchestrator
                .obtainOrchestrator(PoolProvider.getBugsExecutor())
                .addWorkerThreadAction(new StateAction(context))
                .orchestrate();

    }

    public void addImageAttachment(Context context, Uri imageUri, Attachment.Type type) {
        addExternalAttachment(context, imageUri, null, type);
    }

    public void addVideoAttachment(Context context, Uri videoUri, Attachment.Type type) {
        addExternalAttachment(context, videoUri, null, type);
    }

    @WorkerThread
    public void addExternalAttachment(Context context, Uri attachmentUri, @Nullable String
            fileNameWithExtension, Attachment.Type type) {
        Bug currentBug = bug;
        if (currentBug != null) {
            Uri newUri;
            if (type == Attachment.Type.GALLERY_VIDEO) {
                newUri = AttachmentsUtility.getNewFileAttachmentUri(context, attachmentUri,
                        fileNameWithExtension, BaseReportingPresenter.MAX_FILE_SIZE_IN_MB);
            } else {
                newUri = AttachmentsUtility.getNewFileAttachmentUri(context, attachmentUri, fileNameWithExtension);
            }
            if (newUri != null) {
                currentBug.addAttachment(newUri, type);
                refreshAttachments(context);
            }
        }
    }

    public void addExternalAttachmentWithoutCopying(Context context, File file, Attachment.Type type) {
        if (getBug() == null) return;

        getBug().addAttachment(Uri.fromFile(file), type);
        refreshAttachments(context);
    }

    public void refreshAttachments(Context context) {
        LocalBroadcastManager.getInstance(context).sendBroadcast(new Intent(REFRESH_ATTACHMENTS));
    }

    @Nullable
    public OnSdkDismissedCallback.DismissType getDismissType() {
        return dismissType;
    }

    public void setDismissType(OnSdkDismissedCallback.DismissType dismissType) {
        this.dismissType = dismissType;
    }

    public boolean isInBackground() {
        return isInBackground;
    }

    public void setIsInBackground(boolean isInBackground) {
        this.isInBackground = isInBackground;
    }

    public void commit() {
        if (Instabug.getApplicationContext() != null) {
            if (ServiceLocator.getConfigurationsProvider().isBugReportingUsageExceeded())
                removeBug();
            else
                commit(Instabug.getApplicationContext());
        }
    }

    public void commit(final Context context) {
        ActionsOrchestrator
                .obtainOrchestrator(PoolProvider.getBugsExecutor())
                .addWorkerThreadAction(new Action() {
                    @Override
                    public void run() throws Exception {
                        final Bug bug = LiveBugManager.this.bug;
                        if (bug != null) {
                            //TODO: change setting manager to the bug's setting manager
                            final SettingsManager settingsManager = SettingsManager.getInstance();
                            if (settingsManager.getOnReportCreatedListener() != null) {
                                Report report = new Report();
                                try {
                                    settingsManager.getOnReportCreatedListener().onReportCreated(report);
                                } catch (Exception e) {
                                    InstabugSDKLogger.e(Constants.LOG_TAG, "Exception occurred in report Submit Handler ", e);
                                }
                                if (LiveBugManager.getInstance().getBug() != null) {
                                    ReportHelper.update(LiveBugManager.getInstance().getBug().getState(), report);
                                }
                            }

                            addAttachmentFiles(context);
                            copyAndDeleteOriginalAttachments(context);
                            compressTheAttachedImages(context);

                            addReprostepsAttachments(context);

                            AttachmentsUtility.encryptAttachments(bug.getAttachments());

                            updateBugState();

                            setDismissType(OnSdkDismissedCallback.DismissType.SUBMIT);
                            ServiceLocator.getConsentsActionHandler().handle(bug);
                            handleDroppingLogsForFrustratingExperienceBugs(bug);

                            try {
                                State state = bug.getState();
                                if (state != null) {
                                    prepareState(context, state);
                                } else {
                                    bug.setState(State.getState(context));
                                }
                                saveBugToDB(bug);
                                sendBug();

                            } catch (JSONException e) {
                                InstabugSDKLogger.e(Constants.LOG_TAG, "Error while committing bug: ", e);
                                ReportUploadingStateEventBus.INSTANCE.postError(e);
                            } catch (IOException e) {
                                InstabugSDKLogger.e(Constants.LOG_TAG, "IOException while " +
                                        "committing bug", e);
                                ReportUploadingStateEventBus.INSTANCE.postError(e);
                            }

                            AndroidSchedulers.mainThread().scheduleDirect(new Runnable() {
                                @Override
                                public void run() {
                                    runOnSdkDismissedRunnable();
                                    releaseBug();
                                    triggerSync();
                                }
                            });
                        }
                    }
                })
                .orchestrate();
    }

    @WorkerThread
    private void prepareState(Context context, @NonNull State state) throws JSONException, IOException {
        final Bug bug = this.bug;
        if (bug != null) {
            File file = new File(bug.getAttachmentsPath()
                    + "/bug_state_"
                    + System.currentTimeMillis()
                    + ".txt");

            Uri uri = DiskUtils.with(context)
                    .writeOperation(new WriteStateToFileDiskOperation(file, state.toJson()))
                    .execute();
            state.setUri(uri);
            this.bug = bug;
        }
    }

    private void saveBugToDB(Bug bug) throws JSONException {
        long id = ServiceLocator.getBugReportsDbHelper().insert(bug.setBugState(Bug.BugState.READY_TO_BE_SENT));
        if (id != -1) {
            postFrustratingExperienceReportSaved(bug);
        }


    }

    @VisibleForTesting
    void postFrustratingExperienceReportSaved(Bug bug) {
        if (bug.getFrustratingExperienceInternalId() != 0 && bug.getFrustratingExperienceExternalId() == null) {
            FrustratingExperienceEventBus.INSTANCE.post(new FrustratingExperienceEvent.BugReportSaved(bug.getFrustratingExperienceInternalId()));
        }
    }

    private void triggerSync() {
        if (SynchronizationManager.getInstance() != null) {
            SynchronizationManager.getInstance().sync(false);
        }
    }

    public void updateBugState() {
        if (bug != null && bug.getState() != null) {
            Context context = Instabug.getApplicationContext();
            if (context != null && !MemoryUtils.isLowMemory(context) && getFeatureState(IBGFeature.USER_EVENTS) == Feature.State.ENABLED) {
                try {
                    bug.getState().setUserEvents(UserEvent.toJson(InstabugUserEventLogger.getInstance().getUserEvents())
                            .toString());
                } catch (JSONException e) {
                    InstabugSDKLogger.e(Constants.LOG_TAG, "Got error while parsing user events logs", e);
                }
            }
            State state = bug == null ? null : bug.getState();
            if (state != null) {
                if (SettingsManager.getInstance().getOnReportCreatedListener() == null) {
                    bug.getState().setTags(InstabugCore.getTagsAsString());

                    bug.getState().updateConsoleLog();
                    if (getFeatureState(IBGFeature.USER_DATA)
                            == Feature.State.ENABLED) {
                        bug.getState().setUserData(InstabugCore.getUserData());
                    }
                    if (getFeatureState(IBGFeature.INSTABUG_LOGS)
                            == Feature.State.ENABLED) {
                        bug.getState().setInstabugLog(InstabugLog.getLogs());
                    }
                }
                setUserAttributes(bug);

                if (ServiceLocator.getConfigurationsProvider().isReproStepsEnabled()) {
                    bug.getState().updateVisualUserSteps();
                }
                bug.getState().setScreenShotAnalytics(CoreServiceLocator.getScreenShotAnalyticsMapper().map(ServiceLocator.getBugsProductAnalyticsCollector().getAnalytics()).toString());
                bug.getState().setCurrentView(InstabugCore.getCurrentView());
            }
        }
    }

    private void setUserAttributes(@Nullable Bug bug) {
        if (bug == null || bug.getState() == null) return;
        Map<String, String> consentResponses = bug.getUserConsentResponses();
        Map<String, String> attributesMap = new HashMap<>();
        if (consentResponses != null) {
            attributesMap.putAll(consentResponses);
            bug.setUserConsentResponses(null);
        }

        bug.getState().setUserAttributes(
                UserAttributeCacheManager.getSDKUserAttributesAndAppendToIt(attributesMap)
        );
    }

    private void compressTheAttachedImages(Context context) {
        Bug currentBug = this.bug;
        if (currentBug != null) {
            List<Attachment> attachments = currentBug.getAttachments();
            for (Attachment attachment : attachments) {
                if (attachment.getType() != null && attachment.getLocalPath() != null) {
                    if (attachment.getType().equals(Attachment.Type.MAIN_SCREENSHOT)
                            || attachment.getType().equals(Attachment.Type.EXTRA_IMAGE)
                            || attachment.getType().equals(Attachment.Type.GALLERY_IMAGE)) {
                        try {
                            BitmapUtils.compressBitmapAndSave(context, new File(attachment.getLocalPath()));
                        } catch (Exception e) {
                            InstabugSDKLogger.e(Constants.LOG_TAG, "Failed to compress attachments");
                        }
                    }
                }
            }
        }
    }

    private void copyAndDeleteOriginalAttachments(Context context) {
        Bug currentBug = this.bug;
        if (currentBug != null) {
            List<Attachment> attachments = currentBug.getAttachments();
            for (Attachment attachment : attachments) {
                if (attachment.getType() != null && attachment.getLocalPath() != null) {
                    try {
                        attachment.setLocalPath(FileUtils.copyAndDeleteOriginalFile(context, attachment.getLocalPath(), currentBug.getAttachmentsPath()));
                    } catch (Exception e) {
                        InstabugSDKLogger.e(Constants.LOG_TAG, "Failed to compress attachments");
                    }
                }
            }
        }
    }

    private void addAttachmentFiles(Context context) {
        LinkedHashMap<Uri, String> extraAttachmentFiles = InstabugCore.getExtraAttachmentFiles();
        if (extraAttachmentFiles != null) {
            for (Map.Entry<Uri, String> entry : extraAttachmentFiles.entrySet()) {
                if (context != null) {
                    addExternalAttachment(context, entry.getKey(), entry.getValue(),
                            Attachment.Type.ATTACHMENT_FILE);
                }
            }
        }
    }

    private void addReprostepsAttachments(Context context) {
        Bug bug = this.bug;
        if (ServiceLocator.getConfigurationsProvider().isReproScreenshotsEnabled()) {
            if (bug != null && bug.getId() != null) {
                Disposable disposable = VisualUserStepsHelper.getVisualUserStepsFileObservable(context, bug.getId())
                        .subscribe(processedUri -> {
                            if ( processedUri.getUri() != null) {
                                bug.addAttachment(Uri.parse(FileUtils.copyAndDeleteOriginalFile(context, processedUri.getUri().getPath(),
                                                BugUtils.getBugAttachmentFolder(context, bug.getId()))),
                                        Attachment.Type.VISUAL_USER_STEPS, processedUri.isUriEncrypted());
                                this.bug = bug;
                            }
                        }, t -> InstabugSDKLogger.e(Constants.LOG_TAG, "something went wrong while getting Visual User Steps File Observable", t));
            }
        }
    }

    private void runOnSdkDismissedRunnable() {
        BugSettings settingsManager = BugSettings.getInstance();
        if (settingsManager.getOnSdkDismissCallback() != null && LiveBugManager.getInstance().getDismissType() != null &&
                LiveBugManager.getInstance().getBug() != null) {
            settingsManager.getOnSdkDismissCallback().call(
                    ReportingPluginWrapper.getDismissType(LiveBugManager.getInstance().getDismissType()),
                    ReportingPluginWrapper.getReportType(LiveBugManager.getInstance().getBug().getType())
            );
        }
    }

    public void onSdkDismissedForAttachment() {
        setIsInBackground(true);
        setDismissType(OnSdkDismissedCallback.DismissType.ADD_ATTACHMENT);
        runOnSdkDismissedRunnable();
    }

    public void sendBug() {
        if (Instabug.getApplicationContext() != null) {
            NormalBugsUploaderJob.getInstance().start();
            ProactiveReportsBugsUploaderJob.getInstance().start();
        }
    }

    public void setRecordingDuration(int recordingDuration) {
        this.recordingDuration = recordingDuration;
    }

    public int getAndResetRecordingDuration() {
        int duration = recordingDuration;
        recordingDuration = DURATION_NOT_EXISTS;
        return duration;
    }

    @VisibleForTesting
    void handleDroppingLogsForFrustratingExperienceBugs(Bug bug) {
        if (Objects.equals(bug.getType(), Constants.ReportType.FRUSTRATING_EXPERIENCE) && ServiceLocator.getProactiveReportingConfigProvider().getShouldDropLogs()) {
            InstabugSDKLogger.w(Constants.LOG_TAG, "Logs will be dropped from frustrating experience report.");
            State bugState = bug.getState();
            if (bugState != null) {
                dropLogs(bugState);
                bugState.setVisualUserSteps(null);
                bugState.setSessionProfilerTimeline(null);
            }

            Attachment attachmentToRemove = null;

            for (Attachment attachment : bug.getAttachments()) {
                if (attachment != null && attachment.getType() != null && attachment.getType().equals(Attachment.Type.VISUAL_USER_STEPS)) {
                    attachmentToRemove = attachment;
                    deleteAttachment(attachment, bug.getId());
                }
            }

            bug.getAttachments().remove(attachmentToRemove);
            addManuallyAddedConsoleLogsToBug(bug);
            addManuallyAddedInstabugLogsToBug(bug);
        }
    }

    void addManuallyAddedConsoleLogsToBug(Bug bug) {
        SettingsManager settingsManager = SettingsManager.getInstance();
        if (settingsManager.getOnReportCreatedListener() != null) {
            Report report = new Report();
            try {
                settingsManager.getOnReportCreatedListener().onReportCreated(report);
            } catch (Exception e) {
                InstabugSDKLogger.e(Constants.LOG_TAG, "Exception occurred in report Submit Handler ", e);
            }
            State state = bug.getState();
            if (state != null) {
                state.addConsoleLogs(report.getConsoleLog());
            }
        }
    }

    void addManuallyAddedInstabugLogsToBug(Bug bug) {
        SettingsManager settingsManager = SettingsManager.getInstance();
        if (settingsManager.getOnReportCreatedListener() != null) {
            Report report = new Report();
            try {
                settingsManager.getOnReportCreatedListener().onReportCreated(report);
            } catch (Exception e) {
                InstabugSDKLogger.e(Constants.LOG_TAG, "Exception occurred in report Submit Handler ", e);
            }
            State state = bug.getState();
            if (state != null) {
                JSONArray jsonArray = new JSONArray();
                ArrayList<InstabugLog.LogMessage> manuallyAddedLogs = report.getInstabugLogs();
                for(InstabugLog.LogMessage log: manuallyAddedLogs) {
                    jsonArray.put(log.toJson());
                }
                state.setInstabugLog(jsonArray.toString());
            }
        }
    }
}


