package com.instabug.chat.notification;

import static com.instabug.chat.RepliesWrapper.isMessagingServiceAvailable;
import static com.instabug.chat.notification.NotificationBarThemeApplierKt.applyNotificationBarTheme;

import android.animation.Animator;
import android.annotation.SuppressLint;
import android.app.Activity;
import android.content.Context;
import android.content.res.Resources;
import android.graphics.Bitmap;
import android.graphics.Rect;
import android.os.Build;
import android.os.Handler;
import android.os.Looper;
import android.view.Gravity;
import android.view.LayoutInflater;
import android.view.Surface;
import android.view.View;
import android.view.ViewGroup;
import android.widget.Button;
import android.widget.FrameLayout;
import android.widget.TextView;

import androidx.annotation.MainThread;
import androidx.annotation.NonNull;
import androidx.annotation.Nullable;
import androidx.annotation.VisibleForTesting;

import com.instabug.chat.Constants;
import com.instabug.bug.R;
import com.instabug.chat.cache.ChatsCacheManager;
import com.instabug.chat.model.NotificationMessage;
import com.instabug.chat.settings.ChatSettings;
import com.instabug.library.IBGFeature;
import com.instabug.library.Instabug;
import com.instabug.library.InstabugCustomTextPlaceHolder;
import com.instabug.library.InstabugState;
import com.instabug.library.PresentationManager;
import com.instabug.library.core.InstabugCore;
import com.instabug.library.core.eventbus.ActivityLifecycleSubscriber;
import com.instabug.library.core.eventbus.DefaultActivityLifeCycleEventHandler;
import com.instabug.library.core.eventbus.InstabugStateEventBus;
import com.instabug.library.core.eventbus.coreeventbus.IBGCoreEventSubscriber;
import com.instabug.library.core.eventbus.coreeventbus.IBGSdkCoreEvent;
import com.instabug.library.core.eventbus.eventpublisher.IBGCompositeDisposable;
import com.instabug.library.core.eventbus.eventpublisher.IBGDisposable;
import com.instabug.library.internal.servicelocator.CoreServiceLocator;
import com.instabug.library.model.AssetEntity;
import com.instabug.library.ui.custom.CircularImageView;
import com.instabug.library.util.BitmapUtils;
import com.instabug.library.util.InstabugSDKLogger;
import com.instabug.library.util.LocaleUtils;
import com.instabug.library.util.PlaceHolderUtils;
import com.instabug.library.util.ScreenUtility;
import com.instabug.library.util.threading.PoolProvider;

import java.lang.ref.WeakReference;

@SuppressLint("ERADICATE_FIELD_NOT_INITIALIZED")
public class NotificationBarInvoker implements DefaultActivityLifeCycleEventHandler {

    private WeakReference<View> viewWeakReference;
    private boolean isShown = false;
    private boolean isKeyboardOpen;
    private boolean shouldShowNotification;
    @Nullable
    private NotificationMessage notificationMessage;
    @Nullable
    private OnButtonsClickListener onButtonsClickListener;
    @Nullable
    private ActivityLifecycleSubscriber activityLifecycleSubscriber;
    @Nullable
    private IBGCompositeDisposable disposables;

    public NotificationBarInvoker() {
        subscribeToCarenActivityLifeCycle();
        subscribeToSDKStateEvents();
        subscribeToSDKCoreEvents();
    }

    private void subscribeToSDKStateEvents() {
        InstabugStateEventBus.getInstance().subscribe(instabugState -> {
            if (instabugState == InstabugState.DISABLED) {
                clearMessage();
                dismiss();
                if (onButtonsClickListener != null)
                    onButtonsClickListener.onDismiss();
            }
        });
    }

    private void subscribeToSDKCoreEvents() {
        getOrCreateDisposables().add(subscribeToCoreEvents());
    }

    @NonNull
    private IBGDisposable subscribeToCoreEvents() {
        return IBGCoreEventSubscriber.subscribe((this::handleCoreEvents));
    }

    private void handleCoreEvents(IBGSdkCoreEvent event) {
        boolean isRepliesDisabled = event instanceof IBGSdkCoreEvent.Features.Updated &&
                !InstabugCore.isFeatureEnabled(IBGFeature.REPLIES);

        if (event instanceof IBGSdkCoreEvent.User.LoggedOut || isRepliesDisabled) {
            clearMessagesAndDismiss();
        }
    }

    private void clearMessagesAndDismiss() {
        clearMessage();
        dismiss(false);
    }

    private IBGCompositeDisposable getOrCreateDisposables() {
        return disposables != null ? disposables : (disposables = new IBGCompositeDisposable());
    }

    private void initView(@NonNull final WeakReference<Activity> parentActivityWeakReference,
                          final OnViewAddedListener onViewAddedListener) {

        Activity parentActivity = parentActivityWeakReference.get();
        if (parentActivity == null) {
            return;
        }
        View foundedView = parentActivity.findViewById(R.id.instabug_in_app_notification);

        if (foundedView != null) {
            viewWeakReference = new WeakReference<>(foundedView);
            onViewAddedListener.onViewFounded();
            return;
        }

        dismiss(false);

        LayoutInflater inflater = (LayoutInflater) parentActivity
                .getSystemService(Context.LAYOUT_INFLATER_SERVICE);

        if (inflater == null) {
            InstabugSDKLogger.e(Constants.LOG_TAG, "Unable to inflate the InAppNotifications view due to null Inflater");
            return;
        }

        View view = inflater.inflate(R.layout.instabug_lyt_notification, null);
        viewWeakReference = new WeakReference<>(view);
        view.setVisibility(View.INVISIBLE);

        view.setOnClickListener(v -> {
        });

        final FrameLayout.LayoutParams viewLayoutParams = new FrameLayout.LayoutParams(
                FrameLayout.LayoutParams.MATCH_PARENT, FrameLayout.LayoutParams.WRAP_CONTENT);
        viewLayoutParams.gravity = Gravity.BOTTOM;

        Resources resources = parentActivity.getResources();
        if (android.os.Build.VERSION.SDK_INT >= android.os.Build.VERSION_CODES.LOLLIPOP &&
                ScreenUtility.isLandscape(parentActivity) && ScreenUtility.hasNavBar(
                parentActivity.getApplicationContext())) {

            int rotation = parentActivity.getWindowManager().getDefaultDisplay().getRotation();

            if (rotation == Surface.ROTATION_90) {
                viewLayoutParams.rightMargin = ScreenUtility.getNavigationBarWidth(resources);
            } else if (rotation == Surface.ROTATION_270) {
                if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.N) {
                    viewLayoutParams.leftMargin = ScreenUtility.getNavigationBarWidth
                            (resources);
                } else {
                    viewLayoutParams.rightMargin = ScreenUtility.getNavigationBarWidth
                            (resources);
                }
            }
        }

        view.setLayoutParams(viewLayoutParams);

        parentActivity.runOnUiThread(() -> {
            //removing parent first the view has any
            if (view.getParent() != null) {
                ((ViewGroup) view.getParent()).removeView(view);
            }

            ((ViewGroup) parentActivity.getWindow().getDecorView()).addView(view,
                    viewLayoutParams);
            view.postDelayed(onViewAddedListener, 100);
        });

        parentActivity.getWindow().getDecorView().getViewTreeObserver()
                .addOnGlobalLayoutListener(() -> {
                    Rect r = new Rect();
                    parentActivity.getWindow().getDecorView().getWindowVisibleDisplayFrame(r);
                    int screenHeight = parentActivity.getWindow().getDecorView().getRootView
                            ().getHeight();

                    // r.bottom is the position above soft keypad or device button.
                    // if keypad is shown, the r.bottom is smaller than that before.
                    int keypadHeight = screenHeight - r.bottom;

                    if (keypadHeight > screenHeight * 0.15) { // 0.15 ratio is perhaps enough
                        // to determine keypad height.
                        // keyboard is opened
                        isKeyboardOpen = true;
                    } else {
                        // keyboard is closed
                        isKeyboardOpen = false;
                        if (shouldShowNotification && !isShown) {
                            showWithAnimation();
                        }
                    }
                });
    }

    public void show(@NonNull final WeakReference<Activity> activity, final NotificationMessage notificationMessage,
                     OnButtonsClickListener onButtonsClickListener) {

        this.notificationMessage = notificationMessage;
        this.onButtonsClickListener = onButtonsClickListener;

        OnViewAddedListener onViewInitializedRunnable = new OnViewAddedListener() {

            @Override
            void onViewInitialized() {
                View view = viewWeakReference != null ? viewWeakReference.get() : null;
                if (view != null && InstabugCore.getTargetActivity() != null)
                    view.setY(ScreenUtility.getWindowHeight(InstabugCore.getTargetActivity()));
                showNotification(notificationMessage);
            }

            @Override
            public void onViewFounded() {
                showNotification(notificationMessage);
            }
        };

        initView(activity, onViewInitializedRunnable);
        setOnButtonsClickListener();
    }

    private void setOnButtonsClickListener() {
        View view = viewWeakReference != null ? viewWeakReference.get() : null;
        if (view == null) {
            return;
        }
        Button replyButton = view.findViewById(R.id.replyButton);
        Button dismissButton = view.findViewById(R.id.dismissButton);

        if (replyButton != null) {
            replyButton.setOnClickListener(v -> {
                clearMessagesAndDismiss();
                if (onButtonsClickListener != null)
                    onButtonsClickListener.onReply();
            });
        }

        if (dismissButton != null) {
            dismissButton.setOnClickListener(v -> {
                clearMessage();
                dismiss();
                if (onButtonsClickListener != null)
                    onButtonsClickListener.onDismiss();
            });
        }
    }

    private void clearMessage() {
        notificationMessage = null;
    }

    private void showNotification(NotificationMessage
                                          notificationMessage) {
        setData(notificationMessage);
    }

    private void setData(final NotificationMessage message) {
        View view = viewWeakReference != null ? viewWeakReference.get() : null;
        if (view == null) {
            return;
        }
        final CircularImageView senderAvatarImageView = view.findViewById(R.id
                .senderAvatarImageView);
        PoolProvider.postMainThreadTask(() -> {
            Context context = Instabug.getApplicationContext();
            applyNotificationBarTheme(viewWeakReference != null ? viewWeakReference.get() : null);

            // forcing locale on action buttons.
            Button replyButton = view.findViewById(R.id.replyButton);
            Button dismissButton = view.findViewById(R.id.dismissButton);
            String replyText = PlaceHolderUtils.getPlaceHolder(InstabugCustomTextPlaceHolder.Key.REPLIES_NOTIFICATION_REPLY_BUTTON,
                    LocaleUtils.getLocaleStringResource(InstabugCore.getLocale(context), R.string.instabug_str_reply, context));
            if (replyButton != null) {
                replyButton.setText(replyText);
                replyButton.setContentDescription(LocaleUtils.getLocaleStringResource(InstabugCore.getLocale(context), R.string.ibg_notification_reply_btn_content_description, context));
            }

            String dismissText = PlaceHolderUtils.getPlaceHolder(InstabugCustomTextPlaceHolder.Key.REPLIES_NOTIFICATION_DISMISS_BUTTON,
                    LocaleUtils.getLocaleStringResource(InstabugCore.getLocale(context), R.string.instabug_str_dismiss, context));
            if (dismissButton != null) {
                dismissButton.setText(dismissText);
                dismissButton.setContentDescription(LocaleUtils.getLocaleStringResource(InstabugCore.getLocale(context), R.string.ibg_notification_dismiss_btn_content_description, context));
            }

            senderAvatarImageView.setBackgroundResource(R.drawable.ibg_core_ic_avatar);
            TextView senderNameTextView = view.findViewById(R.id.senderNameTextView);
            TextView senderMessageTextView = view.findViewById(R.id.senderMessageTextView);
            if (message.getFrom() != null && senderNameTextView != null) {
                senderNameTextView.setText(message.getFrom());
            }
            if (message.getBody() != null && senderMessageTextView != null) {
                senderMessageTextView.setText(message.getBody());
            }
        });

        if (message.getAvatarURL() != null) {
            PoolProvider.postIOTask(() -> {
                if (message.getAvatarURL() != null) {
                    BitmapUtils.loadBitmapForAsset(Instabug.getApplicationContext(), message.getAvatarURL(), AssetEntity.AssetType.IMAGE,
                            new BitmapUtils.OnBitmapReady() {
                                @Override
                                public void onBitmapReady(@Nullable final Bitmap bitmap) {
                                    PoolProvider.postMainThreadTask(() -> {
                                        showAvatarImage(senderAvatarImageView, bitmap);
                                        if (!isShown) {
                                            showWithAnimation();
                                        } else if (view.getVisibility() != View.VISIBLE) {
                                            view.setVisibility(View.VISIBLE);
                                        }
                                    });
                                }

                                @Override
                                public void onBitmapFailedToLoad() {
                                    if (!isShown) {
                                        showWithAnimation();
                                    } else if (view.getVisibility() != View.VISIBLE) {
                                        view.setVisibility(View.VISIBLE);
                                    }
                                }
                            });
                }
            });
        }
    }


    private void showAvatarImage(CircularImageView senderAvatarImageView, @Nullable Bitmap bitmap) {
        if (bitmap != null) {
            senderAvatarImageView.setBackgroundResource(0);
            senderAvatarImageView.setImageBitmap(bitmap);
        }
    }

    private void dismiss() {
        dismiss(true);
    }

    private void dismiss(boolean animate) {
        View view = viewWeakReference != null ? viewWeakReference.get() : null;
        if (isShown && view != null) {
            int screenHeight = ScreenUtility.getWindowHeight((Activity) view.getContext());
            if (animate) {
                view.animate()
                        .y(screenHeight)
                        .setListener(new OnAnimationEndListener() {
                            @Override
                            public void onAnimationEnd(Animator animation) {
                                view.setVisibility(View.INVISIBLE);
                            }
                        })
                        .start();
            } else {
                view.setY(screenHeight);
                view.setVisibility(View.INVISIBLE);
            }
            isShown = false;
            shouldShowNotification = false;
            PresentationManager.getInstance().setNotificationShowing(false);
        }
    }

    private void showWithAnimation() {
        if (isKeyboardOpen) {
            shouldShowNotification = true;
            return;
        }

        new Handler(Looper.getMainLooper()).post(new Runnable() {

            @Override
            @SuppressLint("ERADICATE_PARAMETER_NOT_NULLABLE")
            public void run() {
                View view = viewWeakReference != null ? viewWeakReference.get() : null;
                if (view == null || InstabugCore.getTargetActivity() == null) {
                    return;
                }

                int screenHeight = ScreenUtility.getWindowHeight(InstabugCore.getTargetActivity());
                int viewHeight = view.getHeight() + ScreenUtility.getBottomInsets(InstabugCore.getTargetActivity());

                view.setVisibility(View.VISIBLE);
                view.animate()
                        .y(screenHeight - viewHeight)
                        .setListener(null)
                        .start();
                isShown = true;
            }
        });

        if (ChatSettings.isInAppNotificationSoundEnabled()) {
            NotificationManager.getInstance().playNotificationSound(Instabug.getApplicationContext());
        }
    }

    private void subscribeToCarenActivityLifeCycle() {
        if (activityLifecycleSubscriber != null) return;
        activityLifecycleSubscriber = CoreServiceLocator.createActivityLifecycleSubscriber(this);
        activityLifecycleSubscriber.subscribe();
    }

    @Override
    @MainThread
    public void handleActivityResumed() {
        onActivityResumed();
    }

    @Override
    @MainThread
    public void handleActivityPaused() {
        onActivityPaused();
    }

    private void onActivityResumed() {
        PoolProvider.postIOTask(() -> {
            int unreadMessages = ChatsCacheManager.getUnreadCount();
            if (unreadMessages > 0 && isMessagingServiceAvailable()) {
                PoolProvider.postMainThreadTask(() -> {
                    if (notificationMessage != null && onButtonsClickListener != null && InstabugCore.getTargetActivity() != null) {
                        show(new WeakReference<>(InstabugCore.getTargetActivity()), notificationMessage, onButtonsClickListener);
                    }
                });
            }
        });
    }

    private void onActivityPaused() {
        dismiss(false);
    }

    @VisibleForTesting
    void release() {
        if (activityLifecycleSubscriber != null) activityLifecycleSubscriber.unsubscribe();
        if (disposables != null) disposables.dispose();
    }

    public interface OnButtonsClickListener {
        void onReply();

        void onDismiss();
    }

    private class OnAnimationEndListener implements Animator.AnimatorListener {

        @Override
        public void onAnimationStart(Animator animation) {

        }

        @Override
        public void onAnimationEnd(Animator animation) {

        }

        @Override
        public void onAnimationCancel(Animator animation) {

        }

        @Override
        public void onAnimationRepeat(Animator animation) {

        }
    }

    public abstract class OnViewAddedListener implements Runnable {

        @Override
        public void run() {
            onViewInitialized();
        }

        abstract void onViewInitialized();

        abstract void onViewFounded();
    }

}
