package com.instabug.chat.network;

import androidx.annotation.Nullable;

import com.instabug.chat.Constants;
import com.instabug.chat.cache.ChatsCacheManager;
import com.instabug.chat.eventbus.ChatTimeUpdatedEventBus;
import com.instabug.chat.eventbus.ChatTriggeringEventBus;
import com.instabug.chat.eventbus.TriggeredChat;
import com.instabug.chat.model.Attachment;
import com.instabug.chat.model.Chat;
import com.instabug.chat.model.Message;
import com.instabug.chat.network.service.MessagingService;
import com.instabug.chat.settings.ChatSettings;
import com.instabug.library.IBGNetworkWorker;
import com.instabug.library.Instabug;
import com.instabug.library.InstabugNetworkJob;
import com.instabug.library.internal.storage.cache.InMemoryCache;
import com.instabug.library.networkv2.request.Request;
import com.instabug.library.util.InstabugDateFormatter;
import com.instabug.library.util.InstabugSDKLogger;

import org.json.JSONException;

import java.io.FileNotFoundException;
import java.io.IOException;
import java.util.Calendar;
import java.util.List;
import java.util.Locale;


public class InstabugMessageUploaderJob extends InstabugNetworkJob {

    @Nullable
    private static InstabugMessageUploaderJob INSTANCE;

    public synchronized static InstabugMessageUploaderJob getInstance() {
        if (INSTANCE == null) {
            INSTANCE = new InstabugMessageUploaderJob();
        }

        return INSTANCE;
    }

    private InstabugMessageUploaderJob() {
    }

    public void start() {
        enqueueJob(IBGNetworkWorker.CHATS, new Runnable() {
            @Override
            public void run() {
                if (Instabug.getApplicationContext() != null) {
                    try {
                        uploadChats();
                        uploadMessages(ChatsCacheManager.getOfflineMessages());
                    } catch (Exception e) {
                        InstabugSDKLogger.e(Constants.LOG_TAG, "Error " + e.getMessage() + " occurred while uploading messages", e);
                    }
                } else {
                    InstabugSDKLogger.e(Constants.LOG_TAG, "Context was null while uploading messages");
                }
            }
        });
    }

    private static void uploadChats() throws IOException, JSONException {
        List<Chat> offlineChats = ChatsCacheManager.getOfflineChats();
        InstabugSDKLogger.v(Constants.LOG_TAG, "Found " + offlineChats.size() + " offline chats in cache");
        for (final Chat offlineChat : ChatsCacheManager.getOfflineChats()) {
            if (offlineChat.getChatState() != null && offlineChat.getChatState().equals(Chat.ChatState.READY_TO_BE_SENT)
                    && offlineChat.getMessages().size() > 0) {
                InstabugSDKLogger.d(Constants.LOG_TAG, "Uploading offline Chat: " + offlineChat);
                MessagingService.getInstance().triggerChat(offlineChat.getState(),
                        new Request.Callbacks<String, Throwable>() {
                            @Override
                            public void onSucceeded(@Nullable String triggeredChatId) {
                                if (triggeredChatId != null) {
                                    // hold offline chat id
                                    final String offlineChatId = offlineChat.getId();
                                    // publish chatTriggering event.
                                    ChatTriggeringEventBus.getInstance().post(new TriggeredChat
                                            (offlineChatId, triggeredChatId));
                                    // replace local chat with updating chat.
                                    InstabugSDKLogger.v(Constants.LOG_TAG,
                                            "Updating local " +
                                                    "chat with id: " + offlineChatId + ", with synced" +
                                                    " chat " +
                                                    "with id: " + triggeredChatId);
                                    offlineChat.setId(triggeredChatId);
                                    offlineChat.setChatState(Chat.ChatState.LOGS_READY_TO_BE_UPLOADED);

                                    InMemoryCache<String, Chat> cache = ChatsCacheManager.getCache();
                                    if (cache != null) {
                                        cache.delete(offlineChatId);
                                        cache.put(offlineChat.getId(), offlineChat);
                                    }
                                    ChatsCacheManager.saveCacheToDisk();
                                    uploadChatLogs(offlineChat);
                                }
                            }

                            @Override
                            public void onFailed(Throwable throwable) {
                                InstabugSDKLogger.e(Constants.LOG_TAG,
                                        "Something went" +
                                                " wrong while triggering offline chat with id: " +
                                                offlineChat.getId(), throwable);
                            }
                        });
            } else if (offlineChat.getChatState() != null && offlineChat.getChatState().equals(Chat.ChatState
                    .LOGS_READY_TO_BE_UPLOADED)) {
                InstabugSDKLogger.d(Constants.LOG_TAG, "chat: " + offlineChat.toString() + " already uploaded but has unsent logs, uploading now");
                uploadChatLogs(offlineChat);
            }
        }
    }

    private static void uploadMessages(List<Message> offlineMessages) throws IOException, JSONException {
        InstabugSDKLogger.v(Constants.LOG_TAG, "Found " + offlineMessages.size() + " offline messages in cache");
        for (int i = 0; i < offlineMessages.size(); i++) {
            final Message offlineMessage = offlineMessages.get(i);
            if (offlineMessage.getMessageState() == Message.MessageState.READY_TO_BE_SENT) {
                InstabugSDKLogger.d(Constants.LOG_TAG, "Uploading message: " + offlineMessages.get(i));
                MessagingService.getInstance().sendMessage(offlineMessage, new Request.Callbacks<String, Throwable>() {
                    @Override
                    public void onSucceeded(@Nullable String response) {
                        if (response != null && !response.equals("") && !response.equals("null")) {
                            InstabugSDKLogger.d(Constants.LOG_TAG, "Send message Request succeeded");
                            Chat chat = ChatsCacheManager.getChat(offlineMessage.getChatId());

                            if (chat != null) {
                                chat.getMessages().remove(offlineMessage);
                                offlineMessage.setId(response);
                                if (offlineMessage.getAttachments().size() == 0) {
                                    offlineMessage.setMessageState(
                                            Message.MessageState.READY_TO_BE_SYNCED);
                                } else {
                                    offlineMessage.setMessageState(Message.MessageState.SENT);
                                }
                                InstabugSDKLogger.v(Constants.LOG_TAG,
                                        "Caching sent message:" + offlineMessage.toString());
                                chat.getMessages().add(offlineMessage);
                                InMemoryCache<String, Chat> cache = ChatsCacheManager.getCache();
                                if (cache != null) {
                                    cache.put(chat.getId(), chat);
                                }
                                ChatsCacheManager.saveCacheToDisk();
                                if (offlineMessage.getAttachments().size() == 0) {
                                    ChatSettings.setLastChatTime(Calendar.getInstance(Locale
                                            .ENGLISH).getTime().getTime());
                                    ChatTimeUpdatedEventBus.getInstance().post(InstabugDateFormatter
                                            .getCurrentUTCTimeStampInMiliSeconds());
                                } else {
                                    try {
                                        uploadAttachments(offlineMessage);
                                    } catch (JSONException | FileNotFoundException e) {
                                        InstabugSDKLogger.e(Constants.LOG_TAG,
                                                "Something went wrong while uploading " +
                                                        "messageattach attachments "
                                                        + e.getMessage());
                                    }
                                }
                            } else {
                                InstabugSDKLogger.e(Constants.LOG_TAG, "Chat is null so can't remove message from it");
                            }
                        }
                    }

                    @Override
                    public void onFailed(Throwable throwable) {
                        InstabugSDKLogger.e(Constants.LOG_TAG, "Something went " +
                                "wrong while uploading cached message", throwable);
                    }
                });

            } else if (offlineMessage.getMessageState() == Message.MessageState.SENT) {
                InstabugSDKLogger.d(Constants.LOG_TAG, "Uploading message's attachments : " + offlineMessages
                        .get(i));
                try {
                    uploadAttachments(offlineMessage);
                } catch (JSONException | FileNotFoundException e) {
                    InstabugSDKLogger.e(Constants.LOG_TAG, "Something went " +
                            "wrong while uploading message attachments " + e.getMessage());
                }
            }
        }
    }

    private static void uploadAttachments(final Message message) throws JSONException,
            FileNotFoundException {
        InstabugSDKLogger.d(Constants.LOG_TAG, "Found " + message.getAttachments().size() + " attachments " +
                "related to message: "
                + message.getBody());
        MessagingService.getInstance().uploadMessageAttachments(message, new Request.Callbacks<Boolean, Message>() {
            @Override
            public void onSucceeded(@Nullable Boolean isSucceeded) {
                InstabugSDKLogger.d(Constants.LOG_TAG, "Message attachments " +
                        "uploaded successfully");
                Chat chat = ChatsCacheManager.getChat(message.getChatId());

                if (chat != null) {
                    chat.getMessages().remove(message);
                    message.setMessageState(Message.MessageState.READY_TO_BE_SYNCED);
                    for (int i = 0; i < message.getAttachments().size(); i++) {
                        message.getAttachments()
                                .get(i)
                                .setState(Attachment.AttachmentState.STATE_SYNCED);
                    }
                    InstabugSDKLogger.v(Constants.LOG_TAG,
                            "Caching sent message:" + message.toString());
                    chat.getMessages().add(message);
                    InMemoryCache<String, Chat> cache = ChatsCacheManager.getCache();
                    if (cache != null) {
                        cache.put(chat.getId(), chat);
                    }
                    ChatsCacheManager.saveCacheToDisk();
                    ChatSettings.setLastChatTime(Calendar.getInstance(Locale.ENGLISH).getTime()
                            .getTime());
                    ChatTimeUpdatedEventBus.getInstance().post(InstabugDateFormatter
                            .getCurrentUTCTimeStampInMiliSeconds());

                } else {
                    InstabugSDKLogger.e(Constants.LOG_TAG, "Chat is null so can't remove message from it");
                }
            }

            @Override
            public void onFailed(Message failedMessage) {
                InstabugSDKLogger.e(Constants.LOG_TAG, "Something went wrong " +
                        "while uploading message attachments");
            }
        });
    }

    private static void uploadChatLogs(final Chat chat) {
        InstabugSDKLogger.d(Constants.LOG_TAG, "START uploading all logs related to this chat id = "
                + chat.getId());
        MessagingService.getInstance().uploadChatLogs(chat, new Request.Callbacks<Boolean, Chat>() {

            @Override
            public void onSucceeded(@Nullable Boolean isSucceeded) {
                InstabugSDKLogger.d(Constants.LOG_TAG, "chat logs uploaded " +
                        "successfully, " +
                        "changing its state");
                chat.setChatState(Chat.ChatState.SENT);
                ChatsCacheManager.saveCacheToDisk();
            }

            @Override
            public void onFailed(Chat failedChat) {
                InstabugSDKLogger.d(Constants.LOG_TAG, "Something went wrong " +
                        "while " +
                        "uploading chat logs");
            }
        });
    }

}
