package com.instabug.bug.internal.video.customencoding;
/**
 * Copyright (c) 2014 Yrom Wang <http://www.yrom.net>
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 * http://www.apache.org/licenses/LICENSE-2.0
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

import android.annotation.SuppressLint;
import android.annotation.TargetApi;
import android.hardware.display.DisplayManager;
import android.hardware.display.VirtualDisplay;
import android.media.MediaCodec;
import android.media.MediaFormat;
import android.media.MediaMuxer;
import android.media.projection.MediaProjection;
import android.os.Build;
import android.os.Handler;
import android.os.HandlerThread;
import android.os.Looper;
import android.os.Message;

import androidx.annotation.Nullable;

import com.instabug.library.Constants;
import com.instabug.library.instacapture.exception.ScreenCapturingFailedException;
import com.instabug.library.util.InstabugSDKLogger;

import java.io.IOException;
import java.nio.ByteBuffer;
import java.util.LinkedList;
import java.util.concurrent.atomic.AtomicBoolean;


@TargetApi(Build.VERSION_CODES.LOLLIPOP)
public class ScreenRecorder {
    private static final String TAG = "ScreenRecorder";
    private static final int INVALID_INDEX = -1;
    private int mWidth;
    private int mHeight;
    private int mDpi;
    private String mDstPath;
    @Nullable
    private MediaProjection mMediaProjection;
    @Nullable
    private VideoEncoder mVideoEncoder;
    @Nullable
    private MicRecorder mAudioEncoder;
    @Nullable
    private MediaFormat mVideoOutputFormat = null, mAudioOutputFormat = null;
    private int mVideoTrackIndex = INVALID_INDEX, mAudioTrackIndex = INVALID_INDEX;
    @Nullable
    private MediaMuxer mMuxer;
    private boolean mMuxerStarted = false;

    private AtomicBoolean mForceQuit = new AtomicBoolean(false);
    private AtomicBoolean mIsRunning = new AtomicBoolean(false);
    @Nullable
    private VirtualDisplay mVirtualDisplay;
    private final MediaProjection.Callback mProjectionCallback = new MediaProjection.Callback() {
        @Override
        public void onStop() {
            quit();
        }
    };

    @Nullable
    private HandlerThread mWorker;
    @Nullable
    private CallbackHandler mHandler;

    @Nullable
    private Callback mCallback;
    private LinkedList<Integer> mPendingVideoEncoderBufferIndices = new LinkedList<>();
    private LinkedList<Integer> mPendingAudioEncoderBufferIndices = new LinkedList<>();
    private LinkedList<MediaCodec.BufferInfo> mPendingAudioEncoderBufferInfos = new LinkedList<>();
    private LinkedList<MediaCodec.BufferInfo> mPendingVideoEncoderBufferInfos = new LinkedList<>();

    public ScreenRecorder(VideoEncodeConfig video,
                          @Nullable AudioEncodeConfig audio,
                          @Nullable MediaProjection mp,
                          String dstPath) {
        mWidth = video.getWidth();
        mHeight = video.getHeight();
        mDpi = video.getDensity();
        mMediaProjection = mp;
        mDstPath = dstPath;
        mVideoEncoder = new VideoEncoder(video);
        mAudioEncoder = audio == null ? null : new MicRecorder(audio);
    }

    /**
     * stop task
     */
    public synchronized final void quit() {
        mForceQuit.set(true);
        if (!mIsRunning.get()) {
            release();
        } else {
            signalStop(false);
        }

    }

    public synchronized void start() {
        if (mWorker != null) throw new IllegalStateException();
        mWorker = new HandlerThread(TAG);
        mWorker.start();
        mHandler = new CallbackHandler(mWorker.getLooper());
        mHandler.sendEmptyMessage(MSG_START);
    }

    public synchronized void setCallback(Callback callback) {
        mCallback = callback;
    }


    public interface Callback {
        void onStop(@Nullable Throwable error);

        void onStart();

        void onRecording(long presentationTimeUs);
    }

    private static final int MSG_START = 0;
    private static final int MSG_STOP = 1;
    private static final int MSG_ERROR = 2;
    private static final int STOP_WITH_EOS = 1;

    private class CallbackHandler extends Handler {
        CallbackHandler(Looper looper) {
            super(looper);
        }

        @Override
        public void handleMessage(Message msg) {
            switch (msg.what) {
                case MSG_START:
                    try {
                        record();
                        if (mCallback != null) {
                            mCallback.onStart();
                        }
                        break;
                    } catch (Exception e) {
                        msg.obj = e;
                    }
                case MSG_STOP:
                case MSG_ERROR:
                    stopEncoders();
                    if (msg.arg1 != STOP_WITH_EOS) signalEndOfStream();
                    if (mCallback != null) {
                        mCallback.onStop((Throwable) msg.obj);
                    }
                    mCallback = null;
                    release();
                    break;
            }
        }
    }

    private synchronized void signalEndOfStream() {
        MediaCodec.BufferInfo eos = new MediaCodec.BufferInfo();
        ByteBuffer buffer = ByteBuffer.allocate(0);
        eos.set(0, 0, 0, MediaCodec.BUFFER_FLAG_END_OF_STREAM);
        if (mVideoTrackIndex != INVALID_INDEX) {
            writeSampleData(mVideoTrackIndex, eos, buffer);
        }
        if (mAudioTrackIndex != INVALID_INDEX) {
            writeSampleData(mAudioTrackIndex, eos, buffer);
        }
        mVideoTrackIndex = INVALID_INDEX;
        mAudioTrackIndex = INVALID_INDEX;
    }

    @SuppressLint("ERADICATE_PARAMETER_NOT_NULLABLE")
    private synchronized void record() {
        if (mIsRunning.get() || mForceQuit.get()) {
            throw new IllegalStateException();
        }
        if (mMediaProjection == null) {
            throw new IllegalStateException("maybe release");
        }
        mIsRunning.set(true);
        if (mHandler != null && mMediaProjection != null) {
            mMediaProjection.registerCallback(mProjectionCallback, mHandler);
        }
        try {
            // create muxer
            mMuxer = new MediaMuxer(mDstPath, MediaMuxer.OutputFormat.MUXER_OUTPUT_MPEG_4);
            // create encoder and input surface
            prepareVideoEncoder();
            prepareAudioEncoder();
        } catch (IOException e) {
            throw new ScreenCapturingFailedException(e);
        }

        if (mVideoEncoder != null && mMediaProjection != null) {
            mVirtualDisplay = mMediaProjection.createVirtualDisplay(this + "-display",
                    mWidth, mHeight, mDpi, DisplayManager.VIRTUAL_DISPLAY_FLAG_PUBLIC,
                    mVideoEncoder.getInputSurface(), null, null);
        }
    }

    private synchronized void muxVideo(int index, MediaCodec.BufferInfo buffer) {
        if (!mIsRunning.get()) {
            return;
        }
        if (!mMuxerStarted || mVideoTrackIndex == INVALID_INDEX) {
            mPendingVideoEncoderBufferIndices.add(index);
            mPendingVideoEncoderBufferInfos.add(buffer);
            return;
        }
        VideoEncoder videoEncoder = mVideoEncoder;
        if (videoEncoder != null) {
            ByteBuffer encodedData = videoEncoder.getOutputBuffer(index);
            writeSampleData(mVideoTrackIndex, buffer, encodedData);
            videoEncoder.releaseOutputBuffer(index);
        }
        if ((buffer.flags & MediaCodec.BUFFER_FLAG_END_OF_STREAM) != 0) {
            // send release msg
            mVideoTrackIndex = INVALID_INDEX;
            signalStop(true);
        }
    }

    private void muxAudio(int index, MediaCodec.BufferInfo buffer) {
        if (!mIsRunning.get()) {
            return;
        }
        if (!mMuxerStarted || mAudioTrackIndex == INVALID_INDEX) {
            mPendingAudioEncoderBufferIndices.add(index);
            mPendingAudioEncoderBufferInfos.add(buffer);
            return;

        }
        MicRecorder micRecorder = mAudioEncoder;
        if (micRecorder != null) {
            ByteBuffer encodedData = micRecorder.getOutputBuffer(index);
            writeSampleData(mAudioTrackIndex, buffer, encodedData);
            micRecorder.releaseOutputBuffer(index);
        }
        if ((buffer.flags & MediaCodec.BUFFER_FLAG_END_OF_STREAM) != 0) {
            mAudioTrackIndex = INVALID_INDEX;
            signalStop(true);
        }
    }

    private void writeSampleData(int track, MediaCodec.BufferInfo buffer, @Nullable ByteBuffer encodedData) {
        if ((buffer.flags & MediaCodec.BUFFER_FLAG_CODEC_CONFIG) != 0) {
            // The codec config data was pulled out and fed to the muxer when we got
            // the INFO_OUTPUT_FORMAT_CHANGED status.
            // Ignore it.
            buffer.size = 0;
        }
        boolean eos = (buffer.flags & MediaCodec.BUFFER_FLAG_END_OF_STREAM) != 0;
        if (buffer.size == 0 && !eos) {
            encodedData = null;
        } else {
            if (buffer.presentationTimeUs != 0) { // maybe 0 if eos
                if (track == mVideoTrackIndex) {
                    resetVideoPts(buffer);
                } else if (track == mAudioTrackIndex) {
                    resetAudioPts(buffer);
                }
            }
            if (!eos && mCallback != null) {
                mCallback.onRecording(buffer.presentationTimeUs);
            }
        }
        if (encodedData != null) {
            encodedData.position(buffer.offset);
            encodedData.limit(buffer.offset + buffer.size);
            if (mMuxer != null) {
                mMuxer.writeSampleData(track, encodedData, buffer);
            }
        }
    }

    private long mVideoPtsOffset, mAudioPtsOffset;

    private synchronized void resetAudioPts(MediaCodec.BufferInfo buffer) {
        if (mAudioPtsOffset == 0) {
            mAudioPtsOffset = buffer.presentationTimeUs;
            buffer.presentationTimeUs = 0;
        } else {
            buffer.presentationTimeUs -= mAudioPtsOffset;
        }
    }

    private synchronized void resetVideoPts(MediaCodec.BufferInfo buffer) {
        if (mVideoPtsOffset == 0) {
            mVideoPtsOffset = buffer.presentationTimeUs;
            buffer.presentationTimeUs = 0;
        } else {
            buffer.presentationTimeUs -= mVideoPtsOffset;
        }
    }

    private synchronized void resetVideoOutputFormat(MediaFormat newFormat) {
        // should happen before receiving buffers, and should only happen once
        if (mVideoTrackIndex >= 0 || mMuxerStarted) {
            throw new IllegalStateException("output format already changed!");
        }
        mVideoOutputFormat = newFormat;
    }

    private synchronized void resetAudioOutputFormat(MediaFormat newFormat) {
        // should happen before receiving buffers, and should only happen once
        if (mAudioTrackIndex >= 0 || mMuxerStarted) {
            throw new IllegalStateException("output format already changed!");
        }
        mAudioOutputFormat = newFormat;
    }

    private synchronized void startMuxerIfReady() {
        if (mMuxerStarted || mVideoOutputFormat == null
                || (mAudioEncoder != null && mAudioOutputFormat == null)) {
            return;
        }

        if (mMuxer != null) {
            mVideoTrackIndex = mMuxer.addTrack(mVideoOutputFormat);
            if (mAudioOutputFormat != null) {
                mAudioTrackIndex = mAudioEncoder == null ? INVALID_INDEX : mMuxer.addTrack(mAudioOutputFormat);
            }
            mMuxer.start();
            mMuxerStarted = true;
        }
        if (mPendingVideoEncoderBufferIndices.isEmpty() && mPendingAudioEncoderBufferIndices.isEmpty()) {
            return;
        }
        MediaCodec.BufferInfo info;
        while ((info = mPendingVideoEncoderBufferInfos.poll()) != null) {
            if (mPendingVideoEncoderBufferIndices.peek() != null) {
                Integer index = mPendingVideoEncoderBufferIndices.poll();
                if (index != null) {
                    muxVideo(index, info);
                }
            }
        }
        if (mAudioEncoder != null) {
            while ((info = mPendingAudioEncoderBufferInfos.poll()) != null) {
                if (mPendingAudioEncoderBufferIndices.peek() != null) {
                    Integer index = mPendingAudioEncoderBufferIndices.poll();
                    if (index != null) {
                        muxAudio(index, info);
                    }
                }
            }
        }
    }

    private synchronized void prepareVideoEncoder() throws IOException {
        VideoEncoder.Callback callback = new VideoEncoder.Callback() {

            @Override
            public void onOutputBufferAvailable(BaseEncoder codec, int index, MediaCodec.BufferInfo info) {
                try {
                    muxVideo(index, info);
                } catch (Exception e) {
                    InstabugSDKLogger.e(Constants.LOG_TAG, "Muxer encountered an error! ", e);
                    if (mHandler != null) Message.obtain(mHandler, MSG_ERROR, e).sendToTarget();
                }
            }

            @Override
            public void onError(Encoder codec, Exception e) {
                InstabugSDKLogger.e(Constants.LOG_TAG, "VideoEncoder ran into an error! ", e);
                if (mHandler != null) Message.obtain(mHandler, MSG_ERROR, e).sendToTarget();
            }

            @Override
            public void onOutputFormatChanged(BaseEncoder codec, MediaFormat format) {
                resetVideoOutputFormat(format);
                startMuxerIfReady();
            }
        };
        if (mVideoEncoder != null) {
            mVideoEncoder.setCallback(callback);
            mVideoEncoder.prepare();
        }
    }

    private synchronized void prepareAudioEncoder() throws IOException {
        final MicRecorder micRecorder = mAudioEncoder;
        if (micRecorder == null) return;
        AudioEncoder.Callback callback = new AudioEncoder.Callback() {

            @Override
            public void onOutputBufferAvailable(BaseEncoder codec, int index, MediaCodec.BufferInfo info) {
                try {
                    muxAudio(index, info);
                } catch (Exception e) {
                    InstabugSDKLogger.e(Constants.LOG_TAG, "Muxer encountered an error! ", e);
                    Message.obtain(mHandler, MSG_ERROR, e).sendToTarget();
                }
            }

            @Override
            public void onOutputFormatChanged(BaseEncoder codec, MediaFormat format) {
                resetAudioOutputFormat(format);
                startMuxerIfReady();
            }

            @Override
            public void onError(Encoder codec, Exception e) {
                InstabugSDKLogger.e(Constants.LOG_TAG, "MicRecorder ran into an error! ", e);
                if (mHandler != null) Message.obtain(mHandler, MSG_ERROR, e).sendToTarget();
            }


        };
        micRecorder.setCallback(callback);
        micRecorder.prepare();
    }

    private synchronized void signalStop(boolean stopWithEOS) {
        if (mHandler != null) {
            Message msg = Message.obtain(mHandler, MSG_STOP, stopWithEOS ? STOP_WITH_EOS : 0, 0);
            mHandler.sendMessageAtFrontOfQueue(msg);
        }
    }

    private synchronized void stopEncoders() {
        mIsRunning.set(false);
        mPendingAudioEncoderBufferInfos.clear();
        mPendingAudioEncoderBufferIndices.clear();
        mPendingVideoEncoderBufferInfos.clear();
        mPendingVideoEncoderBufferIndices.clear();
        // maybe called on an error has been occurred
        if (mVideoEncoder != null)
            mVideoEncoder.stop();

        if (mAudioEncoder != null)
            mAudioEncoder.stop();
    }

    private synchronized void release() {
        if (mMediaProjection != null) {
            mMediaProjection.unregisterCallback(mProjectionCallback);
        }
        if (mVirtualDisplay != null) {
            mVirtualDisplay.release();
            mVirtualDisplay = null;
        }

        mVideoOutputFormat = mAudioOutputFormat = null;
        mVideoTrackIndex = mAudioTrackIndex = INVALID_INDEX;
        mMuxerStarted = false;

        if (mWorker != null) {
            mWorker.quitSafely();
            mWorker = null;
        }
        if (mVideoEncoder != null) {
            mVideoEncoder.release();
            mVideoEncoder = null;
        }
        if (mAudioEncoder != null) {
            mAudioEncoder.release();
            mAudioEncoder = null;
        }

        if (mMediaProjection != null) {
            mMediaProjection.stop();
            mMediaProjection = null;
        }
        if (mMuxer != null) {
            try {
                mMuxer.stop();
                mMuxer.release();
            } catch (Exception e) {
                InstabugSDKLogger.e(Constants.LOG_TAG, "Something went wrong, " + e.getMessage(), e);
            }
            mMuxer = null;
        }
        mHandler = null;
    }

    @Override
    protected synchronized void finalize() throws Throwable {
        if (mMediaProjection != null) {
            release();
        }
        super.finalize();
    }

}
