package com.instabug.bug.invocation;

import static com.instabug.library.invocation.InstabugInvocationEvent.SCREENSHOT;

import android.annotation.SuppressLint;
import android.app.Activity;
import android.content.Context;
import android.graphics.Rect;
import android.os.Build;
import android.view.MotionEvent;

import androidx.annotation.NonNull;
import androidx.annotation.Nullable;
import androidx.annotation.VisibleForTesting;

import com.instabug.bug.di.ServiceLocator;
import com.instabug.bug.invocation.invoker.AbstractInvoker;
import com.instabug.bug.invocation.invoker.FloatingButtonInvoker;
import com.instabug.bug.invocation.invoker.ScreenshotCaptorRegistry;
import com.instabug.bug.invocation.invoker.ScreenshotGestureInvoker;
import com.instabug.bug.invocation.invoker.ShakeInvoker;
import com.instabug.bug.invocation.invoker.ThreeFingerSwipeLeftInvoker;
import com.instabug.bug.invocation.invoker.TwoFingerSwipeLeftInvoker;
import com.instabug.bug.invocation.util.AtomicArray;
import com.instabug.library.Constants;
import com.instabug.library.Feature;
import com.instabug.library.IBGFeature;
import com.instabug.library.Instabug;
import com.instabug.library.InstabugState;
import com.instabug.library.InstabugStateProvider;
import com.instabug.library._InstabugActivity;
import com.instabug.library.core.InstabugCore;
import com.instabug.library.core.eventbus.ActivityLifecycleSubscriber;
import com.instabug.library.core.eventbus.DefaultActivityLifeCycleEventHandler;
import com.instabug.library.core.plugin.PluginPromptOption;
import com.instabug.library.internal.servicelocator.CoreServiceLocator;
import com.instabug.library.invocation.InstabugInvocationEvent;
import com.instabug.library.invocation.InvocationManagerContract;
import com.instabug.library.invocation.InvocationMode;
import com.instabug.library.settings.SettingsManager;
import com.instabug.library.tracking.InstabugInternalTrackingDelegate;
import com.instabug.library.util.InstabugSDKLogger;
import com.instabug.library.util.threading.PoolProvider;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.HashSet;
import java.util.List;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.concurrent.atomic.AtomicReference;
import java.util.concurrent.atomic.AtomicReferenceArray;


/**
 * @author mesbah.
 * <p>
 * A class to manage the invocations, first it initialize the invokers with its settings and delegate any invocation
 * requests to the invocation request listener
 * DEPENDENCIES
 * ************
 * 1- {@link InvocationSettings} this holds invocation setting such as threshold of shaking, floating button position,...
 * 2- {@link InstabugInvocationEvent} this is a list of invocation events such as shaking, floating button and others..
 * 3- {@link AbstractInvoker} this one for the invokers and one for the last invoker.
 * 4- {@link com.instabug.library.core.eventbus.eventpublisher.IBGDisposable}   activity lifecycle changes.
 * 4- {@link InvocationListener} to handle invocation requests
 */


public class InvocationManager implements DefaultActivityLifeCycleEventHandler, InvocationManagerContract {
    // private instances
    private static InvocationManager INSTANCE;
    private InvocationSettings currentInvocationSettings;
    private AtomicReferenceArray<InstabugInvocationEvent> currentInstabugInvocationEvents;
    private List<AbstractInvoker> currentInvokersList = new ArrayList<>();
    private static final String LIFECYCLE_OPS_EXEC = "invocation_lifecycle_ops_exec";

    @Nullable
    private AtomicReferenceArray<AbstractInvoker> currentInvokers;
    @Nullable
    private AtomicReference<AbstractInvoker> lastUsedInvoker = new AtomicReference<>();

    @VisibleForTesting
    @Nullable
    ActivityLifecycleSubscriber currentActivityLifeCycleSubscriber = null;
    private final InvocationManagerSubscribers invocationManagerSubscribers;
    // switch invocation off/on
    private AtomicBoolean isInvocationAvailable = new AtomicBoolean(true);
    @Nullable
    private AtomicReference<InvocationRequestListenerImp> invocationRequestListenerImp;
    private boolean hasSwipeGestureRegistered = false;

    private InvocationManager() {
        currentInvocationSettings = new InvocationSettings();
        currentInstabugInvocationEvents = new AtomicReferenceArray<>(1);
        currentInstabugInvocationEvents.set(0, InstabugInvocationEvent.SHAKE);
        currentInvokers = new AtomicReferenceArray<>(createInvokersList());
        invocationManagerSubscribers = new InvocationManagerSubscribersImpl(this);
        invocationManagerSubscribers.subscribe();
        subscribeToCurrentActivityLifeCycle();
        invocationRequestListenerImp = new AtomicReference<>(new InvocationRequestListenerImp());
    }

    @Override
    public void getInvocationRequested() {
        if (invocationRequestListenerImp != null && invocationRequestListenerImp.get() != null) {
            invocationRequestListenerImp.get().onInvocationRequested();
        } else {
            InstabugSDKLogger.e(Constants.LOG_TAG, "invocationRequestListenerImp == null ");
        }
    }

    @NonNull
    private AbstractInvoker[] createInvokersList() {
        currentInvokersList = new ArrayList<>();
        return currentInvokersList.toArray(new AbstractInvoker[currentInvokersList.size()]);
    }

    @VisibleForTesting
    void addToCurrentInvokers(AbstractInvoker invoker) {
        currentInvokersList.add(invoker);
        currentInvokers = new AtomicReferenceArray<>(currentInvokersList.toArray(new AbstractInvoker[currentInvokersList.size()]));
    }

    public synchronized static void init() {
        if (INSTANCE == null) {
            INSTANCE = new InvocationManager();
        } else if (!SettingsManager.getInstance().isInBackground()) {
            INSTANCE.listen();
        }
    }

    /**
     * Returns the current singleton instance of this class.
     *
     * @return current instance of InvocationManager.
     */
    public synchronized static InvocationManager getInstance() {
        if (INSTANCE == null) {
            init();
        }
        return INSTANCE;
    }

    public void release() {
        if (currentActivityLifeCycleSubscriber != null) {
            currentActivityLifeCycleSubscriber.unsubscribe();
        }
        invocationRequestListenerImp = null;
        if (invocationManagerSubscribers != null) {
            invocationManagerSubscribers.unsubscribe();
        }
    }

    public void setInstabugInvocationEvent(InstabugInvocationEvent... instabugInvocationEvent) {
        if (instabugInvocationEvent == null) {
            InstabugSDKLogger.e(Constants.LOG_TAG, "Passed invocation events" +
                    " has null value, no change will take effect to the previous set invocation events");
            return;
        }

        currentInstabugInvocationEvents = removeDuplicates(instabugInvocationEvent);

        if (currentInvokers != null) {
            for (int i = 0; i < currentInvokers.length(); i++) {
                currentInvokers.get(i).sleep();
            }
            currentInvokers = new AtomicReferenceArray<>(createInvokersList());
        }
        hasSwipeGestureRegistered = false;
        for (int i = 0; i < currentInstabugInvocationEvents.length(); i++) {

            InstabugInvocationEvent event = currentInstabugInvocationEvents.get(i);

            InstabugSDKLogger.v(Constants.LOG_TAG, "set instabug invocation event: " + event);
            //set the list with null if the InstabugInvocationEvent array has none value
            if (event == InstabugInvocationEvent.NONE && instabugInvocationEvent.length == 1) {
                currentInvokers = null;
                break;
            }

            //re-initialize the list if it was sat with null before due to a NONE event passed
            if (currentInvokers == null) {
                currentInvokers = new AtomicReferenceArray<>(createInvokersList());
            }

            Context context = Instabug.getApplicationContext();
            if (invocationRequestListenerImp != null) {
                switch (event) {
                    case SHAKE:
                        if (context != null && invocationRequestListenerImp.get() != null) {
                            AbstractInvoker shakeInvoker = new ShakeInvoker(context, invocationRequestListenerImp.get());
                            ((ShakeInvoker) shakeInvoker).setShakingThreshold(currentInvocationSettings
                                    .getShakeThreshold());
                            if (currentInvokers != null) {
                                addToCurrentInvokers(shakeInvoker);
                            }
                        } else
                            InstabugSDKLogger.e(Constants.LOG_TAG, "did not add ShakeInvoker due to null appContext");
                        break;
                    case FLOATING_BUTTON:
                        if (currentInvokers != null && invocationRequestListenerImp.get() != null) {
                            addToCurrentInvokers(new FloatingButtonInvoker(invocationRequestListenerImp.get()));
                        }
                        break;
                    case TWO_FINGER_SWIPE_LEFT:
                        if (context != null && invocationRequestListenerImp.get() != null) {
                            TwoFingerSwipeLeftInvoker swipeLeftInvoker = new TwoFingerSwipeLeftInvoker(
                                    context, invocationRequestListenerImp.get());
                            if (currentInvokers != null) {
                                addToCurrentInvokers(swipeLeftInvoker);
                            }
                            hasSwipeGestureRegistered = true;
                        } else
                            InstabugSDKLogger.e(Constants.LOG_TAG, "did not add TwoFingerSwipeLeftInvoker due to null appContext");
                        break;
                    case THREE_FINGER_SWIPE_LEFT:
                        if (context != null && invocationRequestListenerImp.get() != null) {
                            ThreeFingerSwipeLeftInvoker swipeLeftInvoker = new ThreeFingerSwipeLeftInvoker(
                                    context, invocationRequestListenerImp.get());
                            if (currentInvokers != null) {
                                addToCurrentInvokers(swipeLeftInvoker);
                            }
                            hasSwipeGestureRegistered = true;
                        } else
                            InstabugSDKLogger.e(Constants.LOG_TAG, "did not add ThreeFingerSwipeLeftInvoker due to null appContext");
                        break;
                    case SCREENSHOT:
                        if (Build.VERSION.SDK_INT < Build.VERSION_CODES.UPSIDE_DOWN_CAKE && currentInvokers != null && invocationRequestListenerImp.get() != null) {
                            addToCurrentInvokers(new ScreenshotGestureInvoker(invocationRequestListenerImp.get()));
                        } else if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.UPSIDE_DOWN_CAKE) {
                            registerScreenshotCaptor();
                        }
                        break;
                    default:
                        break;
                }
            }
        }
        //Waking up the current invokers if they get changed during a running session
        if (currentInvokers != null) {
            setLastUsedInvoker(null);
            listen();
        }
    }

    private void registerScreenshotCaptor() {
        ScreenshotCaptorRegistry screenshotCaptorRegistry = ServiceLocator.getScreenshotCaptorRegistry();
        Activity currentActivity = InstabugInternalTrackingDelegate.getInstance().getCurrentActivity();
        if (screenshotCaptorRegistry != null && currentActivity != null) {
            screenshotCaptorRegistry.startSdkCaptureScreenShot(currentActivity);
        }
    }

    @VisibleForTesting
    AtomicReferenceArray<InstabugInvocationEvent> removeDuplicates(InstabugInvocationEvent[] events) {

        // Store unique items in result.
        ArrayList<InstabugInvocationEvent> result = new ArrayList<>();

        // Record encountered Strings in HashSet.
        HashSet<InstabugInvocationEvent> set = new HashSet<>();

        // Loop over argument list.
        for (InstabugInvocationEvent item : events) {

            // If String is not in set, add it to the list and the set.
            if (!set.contains(item)) {
                result.add(item);
                set.add(item);
            }
        }
        InstabugInvocationEvent[] eventsArray = new InstabugInvocationEvent[result.size()];
        AtomicReferenceArray<InstabugInvocationEvent> eventAtomicReferenceArray = new AtomicReferenceArray<>(result.toArray(eventsArray));
        return eventAtomicReferenceArray;
    }

    @Nullable
    public InstabugInvocationEvent[] getCurrentInstabugInvocationEvents() {
        if (InstabugCore.getFeatureState(IBGFeature.BUG_REPORTING) == Feature.State.DISABLED) {
            return null;
        }
        InstabugInvocationEvent[] currentInvocationEvents = AtomicArray.toArray(currentInstabugInvocationEvents, InstabugInvocationEvent.class);
        if (currentInvocationEvents != null) {
            return Arrays.copyOf(currentInvocationEvents, currentInstabugInvocationEvents.length());
        }
        return null;
    }

    public boolean isScreenshotEnabled() {
        if (getCurrentInstabugInvocationEvents() == null)
            return false;
        for (InstabugInvocationEvent event : getCurrentInstabugInvocationEvents()) {
            if (SCREENSHOT.name().equals(event.name())) {
                return true;
            }
        }
        return false;
    }

    public InvocationSettings getCurrentInvocationSettings() {
        return currentInvocationSettings;
    }

    @Nullable
    List<AbstractInvoker> getCurrentInvokers() {
        AbstractInvoker[] invokers = AtomicArray.toArray(currentInvokers, AbstractInvoker.class);
        if (invokers == null) return null;
        return Arrays.asList(invokers);
    }

    @Nullable
    public AbstractInvoker getLastUsedInvoker() {
        if (this.lastUsedInvoker == null) return null;
        return this.lastUsedInvoker.get();
    }

    @SuppressLint("ERADICATE_PARAMETER_NOT_NULLABLE")
    public void setLastUsedInvoker(@Nullable AbstractInvoker lastUsedInvoker) {
        if (this.lastUsedInvoker != null) {
            this.lastUsedInvoker.set(lastUsedInvoker);
        }
    }

    public void listen() {
        if (Instabug.isEnabled()
                && isInvocationAvailable.get()
                && isPromptOptionsAvailable()
                && currentInvokers != null
                && InstabugCore.getTargetActivity() != null
                && InstabugCore.getRunningSession() != null
                && !SettingsManager.getInstance().isProcessingForeground()) {
            for (int i = 0; i < currentInvokers.length(); i++) {
                AbstractInvoker abstractInvoker = currentInvokers.get(i);
                if (!abstractInvoker.isActive()) abstractInvoker.listen();
            }
        }
    }

    @Override
    public void handle(MotionEvent motionEvent) {
        // ignoring un-needed processing for touch events if no swipe gesterues registered
        if(!hasSwipeGestureRegistered) return;
        if (InstabugStateProvider.getInstance().getState().equals(InstabugState.ENABLED)
                && InstabugCore.isForegroundNotBusy()) {
            handleSwipeEvents(motionEvent);
        }
    }

    private void handleSwipeEvents(MotionEvent motionEvent) {
        if (currentInvokers == null) return;
        for (int i = 0; i < currentInvokers.length(); i++) {
            AbstractInvoker invoker = currentInvokers.get(i);
            if (invoker instanceof TwoFingerSwipeLeftInvoker || invoker instanceof ThreeFingerSwipeLeftInvoker) {
                invoker.handle(motionEvent);
                break;
            }

        }
    }

    public void sleep() {
        if (currentInvokers != null) {
            for (int i = 0; i < currentInvokers.length(); i++) {
                AbstractInvoker abstractInvoker = currentInvokers.get(i);
                if (abstractInvoker.isActive()) abstractInvoker.sleep();
            }
        }
    }

    @Override
    public void switchOffInvocation() {
        isInvocationAvailable.set(false);
    }

    @Override
    public void switchOnInvocation() {
        isInvocationAvailable.set(true);
    }

    public ArrayList<PluginPromptOption> getAvailablePromptOptions() {
        return InstabugCore.getPluginsPromptOptions();
    }

    private void subscribeToCurrentActivityLifeCycle() {
        if (currentActivityLifeCycleSubscriber == null) {
            currentActivityLifeCycleSubscriber = CoreServiceLocator.createActivityLifecycleSubscriber(this);
            currentActivityLifeCycleSubscriber.subscribe();
        }
    }

    @Override
    public void handleActivityStarted() {
        PoolProvider.postOrderedIOTask(LIFECYCLE_OPS_EXEC, () -> {
            ScreenshotCaptorRegistry screenshotCaptorRegistry = ServiceLocator.getScreenshotCaptorRegistry();
            Activity currentActivity = InstabugInternalTrackingDelegate.getInstance().getCurrentActivity();

            if (screenshotCaptorRegistry != null && currentActivity != null && isNotInstabugActivity(currentActivity)) {
                screenshotCaptorRegistry.startSdkCaptureScreenShot(currentActivity);
            }
        });
    }

    @Override
    public void handleActivityResumed() {
        PoolProvider.postOrderedIOTask(LIFECYCLE_OPS_EXEC, this::listen);
    }

    @Override
    public void handleActivityPaused() {
        PoolProvider.postOrderedIOTask(LIFECYCLE_OPS_EXEC, this::sleep);
    }

    @Override
    public void handleActivityStopped() {
        PoolProvider.postOrderedIOTask(LIFECYCLE_OPS_EXEC, () -> {
            ScreenshotCaptorRegistry screenshotCaptorRegistry = ServiceLocator.getScreenshotCaptorRegistry();
            Activity stoppedActivity = InstabugInternalTrackingDelegate.getInstance().getLastStoppedActivity();

            if (screenshotCaptorRegistry != null && stoppedActivity != null && isNotInstabugActivity(stoppedActivity)) {
                screenshotCaptorRegistry.stopSdkCaptureScreenShot(stoppedActivity);
            }
        });
    }

    private boolean isNotInstabugActivity(Activity activity) {
        return !(activity instanceof _InstabugActivity);
    }

    private boolean isPromptOptionsAvailable() {
        return getAvailablePromptOptions().size() > 0;
    }

    public void notifyPrimaryColorChanged() {
        if (Instabug.isEnabled() && currentInvokers != null) {
            for (int i = 0; i < currentInvokers.length(); i++) {
                final AbstractInvoker currentInvoker = currentInvokers.get(i);
                if (InstabugCore.getTargetActivity() != null
                        && currentInvoker instanceof FloatingButtonInvoker) {
                    PoolProvider.postMainThreadTask(new Runnable() {
                        @Override
                        public void run() {
                            currentInvoker.sleep();
                            currentInvoker.listen();
                        }
                    });
                }
            }
        }
    }

    /**
     * A method triggered when any invocation option is changed to show/hide floating button
     * if all invocation options are disabled then hide the button otherwise show it.
     */
    public void notifyInvocationOptionChanged() {
        boolean allOptionsAreDisabled = !isPromptOptionsAvailable();
        FloatingButtonInvoker floatingButtonInvoker = getFloatingButtonInvoker();
        if (floatingButtonInvoker != null) {
            if (allOptionsAreDisabled) {
                floatingButtonInvoker.sleep();
            } else {
                floatingButtonInvoker.updateButtonLocation();
            }
        }
    }

    @Nullable
    private FloatingButtonInvoker getFloatingButtonInvoker() {
        FloatingButtonInvoker floatingButtonInvoker = null;
        if (currentInvokers != null) {
            for (int i = 0; i < currentInvokers.length(); i++) {
                AbstractInvoker currentInvoker = currentInvokers.get(i);
                if (currentInvoker instanceof FloatingButtonInvoker) {
                    floatingButtonInvoker = (FloatingButtonInvoker) currentInvoker;
                    break;
                }
            }
        }
        return floatingButtonInvoker;
    }

    @Override
    @SuppressLint("ERADICATE_PARAMETER_NOT_NULLABLE")
    public void show() {
        if (invocationRequestListenerImp != null && invocationRequestListenerImp.get() != null) {
            invocationRequestListenerImp.get().onInvocationRequested();
        }
        lastUsedInvoker = new AtomicReference<>(null);
    }

    public void invoke(@InvocationMode int invocationMode) {
        InstabugSDKLogger.d(Constants.LOG_TAG, "[InvocationManager#invoke] Invoking with mode: " + invocationMode);
        if (invocationRequestListenerImp != null && invocationRequestListenerImp.get() != null) {
            InstabugSDKLogger.d(Constants.LOG_TAG, "[InvocationManager#invoke] InvocationRequestListener is not null, proceeding ...");
            invocationRequestListenerImp.get().invokeWithMode(invocationMode);
        }
    }

    @VisibleForTesting
    @SuppressLint("ERADICATE_FIELD_NOT_NULLABLE")
    public static void tearDown() {
        INSTANCE = null;
    }

    public void forceInvoke(@InvocationMode int invocationMode) {
        if (invocationRequestListenerImp != null && invocationRequestListenerImp.get() != null) {
            invocationRequestListenerImp.get().forceInvoke(invocationMode);
        }

    }


    @Override
    public boolean fabBoundsContain(int x, int y) {
        if (currentInvokers != null) {
            for (int i = 0; i < currentInvokers.length(); i++) {
                AbstractInvoker currentInvoker = currentInvokers.get(i);
                if (currentInvoker instanceof FloatingButtonInvoker) {
                    FloatingButtonInvoker floatingButtonInvoker = (FloatingButtonInvoker) currentInvoker;
                    Rect buttonBounds = floatingButtonInvoker.getButtonBounds();
                    return buttonBounds.contains(x, y);
                }
            }
        }
        return false;
    }


    @Override
    public boolean isInvocationEventScreenShot() {
        if (currentInstabugInvocationEvents == null) return false;
        if (!InstabugStateProvider.getInstance().getState().equals(InstabugState.DISABLED)) {
            for (int i = 0; i < currentInstabugInvocationEvents.length(); i++) {
                InstabugInvocationEvent invocationEvent = currentInstabugInvocationEvents.get(i);
                if (SCREENSHOT.equals(invocationEvent)) {
                    return true;
                }
            }
        }
        return false;
    }
}
