package com.instabug.chat.ui.chat;

import static android.app.Activity.RESULT_OK;

import android.annotation.SuppressLint;
import android.app.Activity;
import android.content.Intent;
import android.net.Uri;
import android.os.Build;
import android.text.TextUtils;
import android.util.Pair;

import androidx.annotation.NonNull;
import androidx.annotation.RequiresApi;

import com.instabug.chat.ChatPlugin;
import com.instabug.chat.Constants;
import com.instabug.chat.cache.ChatsCacheManager;
import com.instabug.chat.cache.ReadQueueCacheManager;
import com.instabug.chat.eventbus.ChatTriggeringEventBus;
import com.instabug.chat.eventbus.TriggeredChat;
import com.instabug.chat.model.Attachment;
import com.instabug.chat.model.Chat;
import com.instabug.chat.model.FlatMessage;
import com.instabug.chat.model.Message;
import com.instabug.chat.model.ReadMessage;
import com.instabug.chat.network.InstabugMessageUploaderJob;
import com.instabug.chat.notification.NotificationManager;
import com.instabug.chat.screenrecording.ExternalScreenRecordHelper;
import com.instabug.chat.screenshot.ScreenshotHelper;
import com.instabug.chat.settings.ChatSettings;
import com.instabug.chat.synchronization.NewMessagesHandler;
import com.instabug.chat.synchronization.OnNewMessagesReceivedListener;
import com.instabug.chat.synchronization.SynchronizationManager;
import com.instabug.library.core.InstabugCore;
import com.instabug.library.core.plugin.Plugin;
import com.instabug.library.core.ui.BasePresenter;
import com.instabug.library.diagnostics.IBGDiagnostics;
import com.instabug.library.internal.InstabugMediaProjectionIntent;
import com.instabug.library.internal.storage.AttachmentsUtility;
import com.instabug.library.internal.storage.cache.CacheChangedListener;
import com.instabug.library.internal.storage.cache.CacheManager;
import com.instabug.library.internal.storage.cache.InMemoryCache;
import com.instabug.library.internal.video.RequestPermissionActivity;
import com.instabug.library.settings.SettingsManager;
import com.instabug.library.user.UserManagerWrapper;
import com.instabug.library.util.FileUtils;
import com.instabug.library.util.InstabugDateFormatter;
import com.instabug.library.util.InstabugSDKLogger;
import com.instabug.library.util.VideoManipulationUtils;

import java.io.File;
import java.util.ArrayList;
import java.util.Collections;
import java.util.List;
import java.util.concurrent.TimeUnit;

import io.reactivexport.android.schedulers.AndroidSchedulers;
import io.reactivexport.disposables.Disposable;
import io.reactivexport.functions.Consumer;
import io.reactivexport.subjects.PublishSubject;

@SuppressLint("ERADICATE_FIELD_NOT_INITIALIZED")
class ChatPresenter extends BasePresenter<ChatContract.View> implements ChatContract.Presenter,
        CacheChangedListener<Chat>, OnNewMessagesReceivedListener {

    public static final String TAG = "ChatPresenter";
    public static final int REQUEST_SCREEN_RECORDING_CAPTURE_PERMISSIONS = 0xF32;

    public static final long MAX_FILE_SIZE_IN_MB = 50L;
    public static final long MAX_VIDEO_LENGTH_MILLI = 60000;
    private PublishSubject<String> chatCacheUpdatePublishSubject;
    private Disposable chatCacheUpdateDisposable;
    private Disposable chatTriggeringDisposable;

    private Chat chat;

    ChatPresenter(ChatContract.View view) {
        super(view);
    }

    public void init(String chatNumber) {
        // content
        chat = getChatFromCache(chatNumber);
        // UI
        displayProperLeftTopIcon();
        handleAttachmentButtonVisibility();
        updateUI(chat);
        // read chat
        readChat(chat);

        if (SynchronizationManager.getInstance() != null) {
            SynchronizationManager.getInstance().sync(false);
        }
    }

    @Override
    public void listen() {
        reversHangingChat();
        subscribeToChatsCacheUpdates();
        subscribeToChatsCache(this);
        NewMessagesHandler.getInstance().addOnNewMessagesReceivedListener(this);
        subscribeToChatTriggeringEvents();
    }

    @Override
    public void sleep() {
        CacheManager.getInstance().unSubscribe(ChatsCacheManager.CHATS_MEMORY_CACHE_KEY, this);
        NewMessagesHandler.getInstance().removeOnNewMessagesReceivedListener(this);
        unSubscribeFromChatTriggeringEvents();
        unSubscribeToChatsCacheUpdates();
    }

    @Override
    public List<FlatMessage> flattenMessages(List<Message> messages) {
        List<FlatMessage> flatMessages = new ArrayList<>();
        for (Message message : messages) {
            if (message.getAttachments() != null && message.getAttachments().size() > 0) {
                for (Attachment attachment : message.getAttachments()) {
                    FlatMessage attachmentMessage = new FlatMessage()
                            .setBody(message.getBody())
                            .setSenderAvatarUrl(message.getSenderAvatarUrl())
                            .setMessagedAt(message.getMessagedAt())
                            .setUrl(attachment.getUrl())
                            .setPath(attachment.getLocalPath())
                            .setInbound(message.isInbound());
                    if (attachment.getType() != null) {
                        switch (attachment.getType()) {
                            case Attachment.AttachmentType.TYPE_IMAGE:
                            case Attachment.AttachmentType.TYPE_IMAGE_GALLERY:
                            case Attachment.AttachmentType.TYPE_IMAGE_SCREENSHOT:
                                attachmentMessage.setType(FlatMessage.Type.IMAGE);
                                break;
                            case Attachment.AttachmentType.TYPE_AUDIO:
                                attachmentMessage.setType(FlatMessage.Type.AUDIO);
                                attachmentMessage.setMode(FlatMessage.Mode.NONE);
                                break;
                            case Attachment.AttachmentType.TYPE_VIDEO_GALLERY:
                            case Attachment.AttachmentType.TYPE_VIDEO_RECORD:
                            case Attachment.AttachmentType.TYPE_VIDEO:
                                attachmentMessage.setType(FlatMessage.Type.VIDEO);
                                break;
                            default:
                                break;
                        }
                    }
                    flatMessages.add(attachmentMessage);
                }
            }

            FlatMessage textMessage;
            if (!TextUtils.isEmpty(message.getBody())) {
                textMessage = new FlatMessage();
                textMessage.setBody(message.getBody()).setSenderAvatarUrl(message
                                .getSenderAvatarUrl())
                        .setMessagedAt(message.getMessagedAt()).setInbound(message.isInbound())
                        .setType(FlatMessage.Type.MESSAGE);
                if (message.getActions() != null && message.getActions().size() > 0) {
                    textMessage.setMessageActions(message.getActions());
                }
                flatMessages.add(textMessage);

            } else {
                if (!message.isInbound() && message.getActions() != null && message.getActions()
                        .size() > 0) {
                    textMessage = new FlatMessage();
                    textMessage.setBody(message.getBody()).setSenderAvatarUrl(message
                                    .getSenderAvatarUrl())
                            .setMessagedAt(message.getMessagedAt()).setInbound(message.isInbound
                                    ()).setType(FlatMessage.Type.MESSAGE);
                    textMessage.setMessageActions(message.getActions());
                    flatMessages.add(textMessage);
                }
            }
        }
        return flatMessages;
    }

    @Override
    public void sendMessage(Message message) {
        chat.getMessages().add(message);
        // may looks weird to you but believe me removing this line and sending a chat from dashboard
        // then trying to to reply with a file, it will reply on a chat id -1
        // for more context check github PR#1936 or JIRA MOB-2608
        if (chat.getState() == null) chat.setChatState(Chat.ChatState.SENT);
        InMemoryCache<String, Chat> chatCache = ChatsCacheManager.getCache();
        if (chatCache != null) {
            chatCache.put(chat.getId(), chat);
            ChatsCacheManager.saveCacheToDisk();
        }
        if (view != null) {
            ChatContract.View viewRef = view.get();
            if (viewRef != null && viewRef.getViewContext().getContext() != null) {
                InstabugMessageUploaderJob.getInstance().start();
            }
        }
    }

    @Override
    public void handleSentAttachment(Attachment attachment) {
        if (attachment.getType() != null && attachment.getLocalPath() != null) {
            switch (attachment.getType()) {
                case Attachment.AttachmentType.TYPE_IMAGE_GALLERY:
                case Attachment.AttachmentType.TYPE_IMAGE_SCREENSHOT:
                    if (view != null) {
                        ChatContract.View viewRef = view.get();
                        if (ChatSettings.shouldSkipImageAttachmentAnnotation()) {
                            sendMessage(createAttachmentMessage(chat.getId(), attachment));
                        } else {
                            if (viewRef != null) {
                                viewRef.annotateAttachmentImage(
                                        Uri.fromFile(new File(attachment.getLocalPath())),
                                        attachment.getType());
                            }
                        }
                    }
                    break;
                case Attachment.AttachmentType.TYPE_AUDIO:
                case Attachment.AttachmentType.TYPE_VIDEO_GALLERY:
                case Attachment.AttachmentType.TYPE_VIDEO_RECORD:
                default:
                    sendMessage(createAttachmentMessage(chat.getId(), attachment));
            }
        }
    }

    @Override
    public Message createMessage(String chatNumber, String messageBody) {
        Message message = new Message(UserManagerWrapper.getUserName(), UserManagerWrapper.getUserEmail(), InstabugCore.getPushNotificationToken());
        message.setChatId(chatNumber)
                .setBody(messageBody)
                .setMessagedAt(InstabugDateFormatter.getCurrentUTCTimeStampInSeconds())
                .setReadAt(InstabugDateFormatter.getCurrentUTCTimeStampInSeconds())
                .setDirection(Message.Direction.INBOUND)
                .setSenderName(InstabugCore.getIdentifiedUsername())
                .setMessageState(Message.MessageState.READY_TO_BE_SENT);
        return message;
    }

    @Override
    public Message createAttachmentMessage(String chatNumber, Attachment attachment) {
        Message attachmentMessage = createMessage(chatNumber, "");
        attachmentMessage.addAttachment(attachment);
        return attachmentMessage;
    }

    @Override
    public Attachment createImageAttachment(Uri imageUri, String attachmentType) {
        Attachment imageAttachment = createOfflineAttachment();
        imageAttachment.setType(attachmentType)
                .setLocalPath(imageUri.getPath())
                .setName(imageUri.getLastPathSegment());
        return imageAttachment;
    }

    @Override
    public Attachment createVideoAttachment(Uri videoUri) {
        Attachment videoAttachment = createOfflineAttachment();
        videoAttachment.setType(Attachment.AttachmentType.TYPE_VIDEO_GALLERY)
                .setLocalPath(videoUri.getPath());
        videoAttachment.setVideoEncoded(true);
        return videoAttachment;
    }

    @Override
    public Chat getChat() {
        return chat;
    }

    private Attachment createOfflineAttachment() {
        Attachment offlineAttachment = new Attachment();
        offlineAttachment.setState(Attachment.AttachmentState.STATE_OFFLINE);
        return offlineAttachment;
    }

    @Override
    public void readChat(Chat chat) {
        chat.markRead();
        if (ChatsCacheManager.getCache() != null)
            ChatsCacheManager.getCache().put(chat.getId(), chat);
    }

    @Override
    public void attachScreenshot() {
        if (isMediaProjectionEnabled()) {
            ChatContract.View viewRef = view.get();
            if (viewRef != null) {
                viewRef.requestMediaProjectionPermission();
            }
        } else {
            startScreenshotHelper();
        }
    }

    private boolean isMediaProjectionEnabled() {
        return SettingsManager.getInstance().isScreenshotByMediaProjectionEnabled();
    }

    @Override
    public void startScreenshotHelper() {
        SettingsManager.getInstance().setProcessingForeground(false);
        ChatPlugin plugin = (ChatPlugin) InstabugCore.getXPlugin(ChatPlugin.class);
        if (plugin != null && plugin.getAppContext() != null) {
            if (chat != null) {
                InstabugSDKLogger.v(Constants.LOG_TAG, "taking extra screenshot");
                plugin.setState(Plugin.STATE_PROCESSING_ATTACHMENT);
                chat.setChatState(Chat.ChatState.WAITING_ATTACHMENT_MESSAGE);
                ScreenshotHelper.getInstance().startScreenshotCapturing(plugin.getAppContext(),
                        chat.getId());
                if (view != null) {
                    ChatContract.View viewRef = view.get();
                    if (viewRef != null) {
                        viewRef.finishActivity();
                    }
                }
            }
        }
    }

    @Override
    public void startScreenRecordingHelper() {
        ChatPlugin plugin = (ChatPlugin) InstabugCore.getXPlugin(ChatPlugin.class);
        if (plugin != null && plugin.getAppContext() != null) {
            if (chat != null) {
                InstabugSDKLogger.v(Constants.LOG_TAG, "start recording screen");
                plugin.setState(Plugin.STATE_PROCESSING_ATTACHMENT);
                chat.setChatState(Chat.ChatState.WAITING_ATTACHMENT_MESSAGE);
                ExternalScreenRecordHelper.getInstance().start(chat.getId());
                if (view != null) {
                    ChatContract.View viewRef = view.get();
                    if (viewRef != null) {
                        viewRef.finishActivity();
                    }
                }
            }
        }
    }

    @Override
    public void startImageFromGalleryPicker() {
        ChatPlugin plugin = (ChatPlugin) InstabugCore.getXPlugin(ChatPlugin.class);
        if (plugin != null && plugin.getAppContext() != null) {
            if (chat != null) {
                InstabugSDKLogger.v(Constants.LOG_TAG, "picking image from gallery");
                plugin.setState(Plugin.STATE_PROCESSING_ATTACHMENT);
                chat.setChatState(Chat.ChatState.WAITING_ATTACHMENT_MESSAGE);
                ChatContract.View viewRef = view.get();
                if (viewRef != null) {
                    viewRef.pickImageFromGallery();
                }
            }
        }
    }

    @Override
    public void cleanOfflineChats() {
        if (chat != null && chat.getMessages().size() == 0 && chat.getChatState() != Chat.ChatState
                .WAITING_ATTACHMENT_MESSAGE) {
            InMemoryCache<String, Chat> chatCache = ChatsCacheManager.getCache();
            if (chatCache != null) {
                chatCache.delete(chat.getId());
            }
        }
    }

    @Override
    public void openVideoRecorder(Intent data) {
        if (view != null) {
            ChatContract.View viewRef = view.get();
            ExternalScreenRecordHelper.getInstance().start(chat.getId());
            chat.setChatState(Chat.ChatState.WAITING_ATTACHMENT_MESSAGE);
            if (viewRef != null) {
                viewRef.finishActivity();
            }

            ChatPlugin bugPlugin = (ChatPlugin) InstabugCore.getXPlugin(ChatPlugin.class);
            if (bugPlugin != null) {
                bugPlugin.setState(Plugin.STATE_PROCESSING_ATTACHMENT);
            }
        }
    }

    @Override
    public void onActivityResult(int requestCode, int resultCode, Intent data) {
        ChatContract.View viewRef = view.get();
        if (viewRef != null) {
            Activity activity = viewRef.getViewContext().getActivity();
            switch (requestCode) {
                case ChatFragment.PICK_IMAGE_REQUEST_CODE:
                    if (resultCode == RESULT_OK && data != null && data.getData() != null && activity != null) {
                        //Todo post this to BG Thread to fix DiskWriteViolation
                        if (Build.VERSION.SDK_INT >= 21) {
                            handleGalleryFileAbove21(viewRef, data);
                        } else {
                            handleGalleryFileBelow21(viewRef, data);
                        }
                    }

                    ChatPlugin chatPlugin = (ChatPlugin) InstabugCore.getXPlugin(ChatPlugin.class);
                    if (chatPlugin != null) {
                        chatPlugin.setState(Plugin.STATE_FOREGROUND);
                    }
                    break;
                case REQUEST_SCREEN_RECORDING_CAPTURE_PERMISSIONS:
                    if (resultCode == Activity.RESULT_OK && data != null) {
                        InstabugMediaProjectionIntent.setMediaProjectionIntent(data);
                        openVideoRecorder(data);
                    }
                    break;

                case RequestPermissionActivity.REQUEST_MEDIA_PROJECTION_PERMISSION:
                    if (data != null && data.getBooleanExtra(RequestPermissionActivity.KEY_IS_PERMISSION_GRANDTED, false))
                        startScreenshotHelper();
                    break;
                default:
                    break;
            }
        }
    }

    @RequiresApi(api = Build.VERSION_CODES.KITKAT)
    private void handleGalleryFileAbove21(ChatContract.View viewRef, Intent data) {
        if (viewRef.getViewContext() != null && viewRef.getViewContext().getContext() != null) {
            Pair<String, String> fileNameAndSize = AttachmentsUtility.getFileNameAndSize(viewRef.getViewContext().getContext(), data.getData());
            if (fileNameAndSize != null) {
                String extension = null, size = "0";

                String fileName = fileNameAndSize.first;

                if (fileNameAndSize.first != null)
                    extension = FileUtils.getExtension(fileName);
                if (fileNameAndSize.second != null)
                    size = fileNameAndSize.second;

                if (extension == null) {
                    InstabugSDKLogger.e(Constants.LOG_TAG, "Selected file extension is null");
                    return;
                }

                if (FileUtils.isImageExtension(extension)) {
                    listen();

                    File imageFile = AttachmentsUtility.getFileFromContentProvider(viewRef.getViewContext().getContext(), data.getData(), fileName);

                    if (imageFile != null)
                        handleSentAttachment(createImageAttachment(Uri.fromFile(imageFile), Attachment.AttachmentType.TYPE_IMAGE_GALLERY));

                } else if (FileUtils.isVideoExtension(extension)) {
                    try {
                        long file_size_in_mb = (Long.parseLong(size) / 1024L / 1024L);
                        if (file_size_in_mb > MAX_FILE_SIZE_IN_MB) {
                            viewRef.showMediaFileSizeAlert();
                            InstabugSDKLogger.e(Constants.LOG_TAG, "Selected video size exceeded the limit");

                            ChatPlugin chatPlugin = (ChatPlugin) InstabugCore.getXPlugin(ChatPlugin.class);
                            if (chatPlugin != null) {
                                chatPlugin.setState(Plugin.STATE_FOREGROUND);
                            }
                            return;
                        }

                        File file = AttachmentsUtility.getFileFromContentProvider(viewRef.getViewContext().getContext(), data.getData(), fileName);

                        if (file != null) {
                            long videoLength = VideoManipulationUtils.extractVideoDuration(file.getPath());
                            if (videoLength > MAX_VIDEO_LENGTH_MILLI) {
                                viewRef.showVideoLengthAlert();
                                InstabugSDKLogger.e(Constants.LOG_TAG, "Selected video length exceeded the limit");

                                if (file.delete()) {
                                    InstabugSDKLogger.v(Constants.LOG_TAG, "file deleted");
                                }
                            } else {
                                listen();
                                handleSentAttachment(createVideoAttachment(Uri.fromFile(file)));
                            }
                        } else {
                            InstabugSDKLogger.e(Constants.LOG_TAG, "Selected video file was null");
                        }

                    } catch (Exception e) {
                        InstabugSDKLogger.e(Constants.LOG_TAG, "Error while selecting video from gallery", e);
                    }
                }
            }
        }
    }

    @SuppressLint("ERADICATE_PARAMETER_NOT_NULLABLE")
    private void handleGalleryFileBelow21(ChatContract.View viewRef, @NonNull Intent data) {
        if (viewRef.getViewContext() == null || viewRef.getViewContext().getActivity() == null)
            return;

        String filePath = AttachmentsUtility.getGalleryImagePath(viewRef.getViewContext().getActivity(), data.getData());
        if (filePath == null && data.getData() != null) {
            filePath = data.getData().getPath();
        }
        String extension = FileUtils.getExtension(filePath);
        Uri selectedMediaUri = AttachmentsUtility.getNewFileAttachmentUri(viewRef.getViewContext().getContext(), Uri.fromFile(new File(filePath)));
        if (selectedMediaUri == null) {
            return;
        }
        if (FileUtils.isImageExtension(extension)) {
            listen();
            handleSentAttachment(createImageAttachment(selectedMediaUri, Attachment.AttachmentType.TYPE_IMAGE_GALLERY));
        } else if (FileUtils.isVideoExtension(extension)) {
            File file = new File(filePath);
            long file_size_in_mb = (file.length() / 1024L / 1024L);
            if (file_size_in_mb > MAX_FILE_SIZE_IN_MB) {
                viewRef.showMediaFileSizeAlert();
            } else if (VideoManipulationUtils.extractVideoDuration(filePath) > MAX_VIDEO_LENGTH_MILLI) {
                viewRef.showVideoLengthAlert();
            } else {
                listen();
                handleSentAttachment(createVideoAttachment(selectedMediaUri));
            }
        }
    }

    @Override
    public List<Message> onNewMessagesReceived(@NonNull List<Message> newMessages) {
        if (view != null) {
            ChatContract.View v = view.get();
            if (v != null) {
                if (v.getViewContext().getActivity() != null) {
                    for (Message message : newMessages) {
                        if (message.getChatId() != null && message.getChatId().equals(chat.getId())) {
                            newMessages.remove(message);
                            NotificationManager.getInstance().playNotificationSound(v
                                    .getViewContext().getActivity());
                            readChat(chat);
                        }
                    }
                    return newMessages;
                }
            }
        }
        return newMessages;
    }

    @Override
    public void onCachedItemRemoved(final Chat removedElement) {
        handleChatCacheUpdate(removedElement.getId());
    }

    @Override
    public void onCachedItemAdded(final Chat addedElement) {
        handleChatCacheUpdate(addedElement.getId());
    }

    @Override
    public void onCachedItemUpdated(final Chat oldElement, final Chat updatedElement) {
        handleChatCacheUpdate(updatedElement.getId());
    }

    @Override
    public void onCacheInvalidated() {
        InstabugSDKLogger.d(Constants.LOG_TAG, "Chats cache was invalidated, Time: " + System
                .currentTimeMillis());
    }

    private void displayProperLeftTopIcon() {
        if (view != null) {
            ChatContract.View viewRef = view.get();
            if (viewRef != null) {
                if (ChatsCacheManager.getValidChats().size() > 0) {
                    viewRef.makeLeftTopButtonBackButton();
                } else {
                    viewRef.makeLeftTopButtonCloseButton();
                }
            }
        }
    }

    private void handleAttachmentButtonVisibility() {
        if (view != null) {
            ChatContract.View viewRef = view.get();
            if (viewRef != null) {
                if (ChatSettings.isAttachmentsTypesEnabled()) {
                    viewRef.showAttachmentButton();
                } else {
                    viewRef.hideAttachmentButton();
                }
            }
        }
    }

    private void handleChatCacheUpdate(String chatId) {
        if (chatId.equals(chat.getId())) {
            chatCacheUpdatePublishSubject.onNext(chatId);
        }
    }

    private void subscribeToChatsCacheUpdates() {
        chatCacheUpdatePublishSubject = PublishSubject.create();
        chatCacheUpdateDisposable = chatCacheUpdatePublishSubject
                .debounce(300, TimeUnit.MILLISECONDS)
                .observeOn(AndroidSchedulers.mainThread())
                .subscribe(new Consumer<String>() {
                    @Override
                    public void accept(String chatNumber) throws Exception {
                        updateChatFromCache(chatNumber);
                    }
                });
    }

    private void subscribeToChatTriggeringEvents() {
        if (!isSubscribed(chatTriggeringDisposable)) {
            chatTriggeringDisposable = ChatTriggeringEventBus.getInstance().subscribe(new Consumer<TriggeredChat>() {

                @Override
                public void accept(TriggeredChat triggeredChat) {
                    if (chat.getId().equals(triggeredChat.getOldChatId())) {
                        chat.setId(triggeredChat.getNewChatId());
                    }
                }
            });
        }
    }

    private void unSubscribeFromChatTriggeringEvents() {
        if (isSubscribed(chatTriggeringDisposable)) {
            chatTriggeringDisposable.dispose();
        }
    }

    private void unSubscribeToChatsCacheUpdates() {
        if (isSubscribed(chatCacheUpdateDisposable)) {
            chatCacheUpdateDisposable.dispose();
        }
    }

    private boolean isSubscribed(Disposable disposable) {
        return disposable != null && !disposable.isDisposed();
    }

    private void updateChatFromCache(String chatNumber) {
        chat = getChatFromCache(chatNumber);
        if (chat != null) {
            updateUI(chat);
        }
    }

    @SuppressLint("ERADICATE_RETURN_NOT_NULLABLE")
    private Chat getChatFromCache(String chatNumber) {
        Chat chat;
        if (ChatsCacheManager.getCache() != null
                && ChatsCacheManager.getChat(chatNumber) != null) {
            chat = ChatsCacheManager.getChat(chatNumber);
        } else {
            chat = new Chat();
        }
        return chat;
    }

    private void reversHangingChat() {
        if (chat.getChatState() == Chat.ChatState.WAITING_ATTACHMENT_MESSAGE) {
            chat.setChatState(Chat.ChatState.READY_TO_BE_SENT);
        }
    }

    private void addToReadQueue(List<Message> messages) {
        // TODO verify this logic, get messages sorted by date
        for (int i = messages.size() - 1; i >= 0; i--) {
            if (!messages.get(i).isInbound() && !messages.get(i).isRead()) {
                ReadMessage readMessage = new ReadMessage();
                readMessage.setChatNumber(messages.get(i).getChatId());
                readMessage.setMessageId(messages.get(i).getId());
                readMessage.setReadAt(InstabugDateFormatter.getCurrentUTCTimeStampInSeconds());
                ReadQueueCacheManager.getInstance().add(readMessage);
                break;
            }
        }
    }

    private void updateUI(Chat chat) {
        addToReadQueue(chat.getMessages());
        Collections.sort(chat.getMessages(), new Message.Comparator());
        if (view != null) {
            ChatContract.View v = view.get();
            if (v != null) {
                v.updateChatMessages(chat.getMessages());
                v.notifyUIForUpdates();
            }
        }
    }

    private void subscribeToChatsCache(CacheChangedListener<Chat> listener) {
        try {
            CacheManager.getInstance().subscribe(ChatsCacheManager.CHATS_MEMORY_CACHE_KEY, listener);
        } catch (IllegalArgumentException exception) {
            InstabugSDKLogger.e(TAG, "Couldn't subscribe to cache", exception);
            IBGDiagnostics.reportNonFatal(exception, "Couldn't subscribe to cache");
        }
    }
}
