package com.instabug.chat

import android.content.Context
import com.instabug.chat.cache.CacheUtility
import com.instabug.chat.cache.ChatsCacheManager
import com.instabug.chat.network.InstabugMessageUploaderJob
import com.instabug.chat.settings.ChatSettings
import com.instabug.chat.synchronization.SynchronizationManager
import com.instabug.library.Feature
import com.instabug.library.IBGFeature
import com.instabug.library.core.InstabugCore
import com.instabug.library.core.eventbus.coreeventbus.IBGSdkCoreEvent
import com.instabug.library.util.threading.PoolProvider

object ChatCoreEventHandler {
    private val isChatFeatureEnabled: Boolean
        get() = InstabugCore.getFeatureState(IBGFeature.IN_APP_MESSAGING) == Feature.State.ENABLED

    @JvmStatic
    fun handleCoreEvents(context: Context?, coreEvent: IBGSdkCoreEvent) {
        if (isChatFeatureEnabled)
            when (coreEvent) {
                IBGSdkCoreEvent.NetworkActivated -> handleNetworkActivated()
                IBGSdkCoreEvent.Session.SessionStarted -> handleSessionStart(context)
                IBGSdkCoreEvent.Session.SessionFinished -> handleSessionFinished()
                IBGSdkCoreEvent.User.LoggedIn -> handleUserLogin()
                IBGSdkCoreEvent.User.LoggedOut -> handleUserLogout()
                IBGSdkCoreEvent.EncryptionStateChanged -> handleEncryptionStateChange()
                else -> {}
            }
    }

    private fun handleEncryptionStateChange() {
        ChatsCacheManager.migrateOnEncryptionStateChange()
    }

    private fun handleNetworkActivated() {
        CacheUtility.dumpCache()
        startBackgroundTasks()
        SynchronizationManager.getInstance().sync(false)
    }

    private fun handleUserLogin() {
        SynchronizationManager.getInstance().sync(true)
    }

    private fun handleUserLogout() {
        ChatSettings.setLastChatTime(0L)
    }

    private fun startBackgroundTasks() {
        PoolProvider.postIOTask {
            val offlineChats = ChatsCacheManager.getOfflineChats()
            val offlineMessages = ChatsCacheManager.getOfflineMessages()
            if (offlineChats.isNotEmpty() || offlineMessages.isNotEmpty()) {
                InstabugMessageUploaderJob.getInstance().start()
            }
        }
    }


    private fun handleSessionStart(context: Context?) {
        context?.let(CacheUtility::prepareCache)
        SynchronizationManager.getInstance().sync(false)
        ChatPluginWrapper.sendPushNotificationToken(false)
    }

    private fun handleSessionFinished() {
        CacheUtility.dumpCache()
        startBackgroundTasks()
        SynchronizationManager.getInstance().stop()
    }

}