package com.instabug.bug.proactivereporting.ui

import android.net.Uri
import androidx.annotation.VisibleForTesting
import androidx.annotation.WorkerThread
import com.instabug.bug.BugPlugin
import com.instabug.bug.Constants
import com.instabug.bug.LiveBugManager
import com.instabug.bug.R
import com.instabug.bug.di.ServiceLocator
import com.instabug.library.core.InitialScreenshotHelper
import com.instabug.library.core.InstabugCore
import com.instabug.library.core.plugin.Plugin
import com.instabug.library.core.plugin.Plugin.PluginState
import com.instabug.library.core.ui.BasePresenter
import com.instabug.library.diagnostics.IBGDiagnostics
import com.instabug.library.frustratingexperience.FrustratingExperienceType
import com.instabug.library.internal.video.MediaProjectionHelper.isMediaProjectionServiceAvailable
import com.instabug.library.settings.SettingsManager
import com.instabug.library.util.threading.PoolProvider

@Suppress("BASE_CLASS_FIELD_WITH_DIFFERENT_SIGNATURE_THAN_DERIVED_CLASS_PROPERTY")
class ProactiveReportingDialogActivityPresenter(val view: ProactiveReportingDialogActivityContract.View?) :
    BasePresenter<ProactiveReportingDialogActivityContract.View>(
        view
    ), ProactiveReportingDialogActivityContract.Presenter {

    @VisibleForTesting
    val SDK_EVENT_DIALOG_DISMISSED = "proactive_bugs_modal_ignored"

    @VisibleForTesting
    val SDK_EVENT_DIALOG_ACCEPTED = "proactive_bugs_modal_reported"

    @VisibleForTesting
    val SDK_EVENT_DIALOG_OPENED = "proactive_bugs_modal_triggers"

    private val executor = PoolProvider.getInstance().ioExecutor

    override fun getTitleResId(type: String?) = when (type) {
        FrustratingExperienceType.FORCE_RESTART -> R.string.ib_frustrating_experience_force_restart_dialog_title
        else -> null
    }

    override fun onDialogShown() {
        executor.execute {
            logSdkEvent(SDK_EVENT_DIALOG_OPENED)
            ServiceLocator.getProactiveReportingConfigProvider().lastModalTime =
                System.currentTimeMillis()
            setBugPluginState(Plugin.STATE_FOREGROUND)
        }
    }

    override fun onNegativeButtonClicked() {
        executor.execute {
            logSdkEvent(SDK_EVENT_DIALOG_DISMISSED)
            setBugPluginState(Plugin.STATE_BACKGROUND)
        }
        view?.get()?.finishActivity()
    }

    override fun onPositiveButtonClicked(type: @FrustratingExperienceType String?, id: Long?) {
        executor.execute {
            logSdkEvent(SDK_EVENT_DIALOG_ACCEPTED)
            handleBugReportingInvocation(type, id)
        }
    }

    private fun handleBugReportingInvocation(type: @FrustratingExperienceType String?, id: Long?) {
        callOnSdkInvoked()
        createBug(type, id ?: 0L)
    }

    private fun callOnSdkInvoked() {
        InstabugCore.getOnSdkInvokedCallback()?.onSdkInvoked()
    }

    private fun takeInitialScreenshotIfNeeded() {
        if (isInitialScreenshotRequired()) {
            InitialScreenshotHelper.captureScreenshot(object : InitialScreenshotHelper(),
                InitialScreenshotHelper.InitialScreenshotCapturingListener {
                override fun onScreenshotCapturedSuccessfully(screenshotUri: Uri?) {
                    LiveBugManager.getInstance().bug?.initialScreenshotPath = screenshotUri?.path
                    navigateToReportingActivityAndFinishActivity()

                }

                override fun onScreenshotCapturingFailed(throwable: Throwable?) {
                    navigateToReportingActivityAndFinishActivity()
                }

            })
        } else {
            navigateToReportingActivityAndFinishActivity()
        }

    }

    private fun createBug(type: @FrustratingExperienceType String?, id: Long) {
        view.get()?.viewContext?.let { activity ->
            LiveBugManager.getInstance().createBug(activity)
            LiveBugManager.getInstance().bug?.addCategory(Constants.ReportType.FRUSTRATING_EXPERIENCE)
            if (type != null) {
                LiveBugManager.getInstance().bug?.addCategory(type)
            }
            LiveBugManager.getInstance().bug?.frustratingExperienceInternalId = id
        }
        takeInitialScreenshotIfNeeded()
    }

    private fun navigateToReportingActivityAndFinishActivity() {
        PoolProvider.postMainThreadTask {
            view?.get()?.navigateToBugReportingActivity()
            view?.get()?.finishActivity()
        }
    }


    @WorkerThread
    private fun logSdkEvent(eventName: String) {
        IBGDiagnostics.logEvent(eventName)
    }


    @WorkerThread
    private fun setBugPluginState(state: @PluginState Int) {
        val plugin = InstabugCore.getXPlugin(BugPlugin::class.java) as? BugPlugin
        plugin?.state = state;
    }

    private fun isInitialScreenshotRequired(): Boolean {
        return if (SettingsManager.getInstance().isScreenshotByMediaProjectionEnabled) {
            val context = view.get()?.viewContext
            if (context != null) {
                SettingsManager.isInitialScreenShotAllowed() && isMediaProjectionServiceAvailable(
                    context
                )
            } else {
                false
            }
        } else {
            SettingsManager.isInitialScreenShotAllowed()
        }
    }

}