package com.instabug.bug.internal.video;

import android.annotation.SuppressLint;
import android.media.MediaPlayer;
import android.net.Uri;
import android.os.Bundle;
import android.view.View;
import android.widget.VideoView;

import androidx.annotation.NonNull;
import androidx.annotation.Nullable;
import androidx.appcompat.app.ActionBar;
import androidx.appcompat.app.AppCompatActivity;
import androidx.fragment.app.FragmentActivity;

import com.instabug.bug.R;
import com.instabug.library.Constants;
import com.instabug.library.InstabugBaseFragment;

import com.instabug.library.util.InstabugSDKLogger;
import com.instabug.library.view.IBGProgressDialog;

public class VideoPlayerFragment extends InstabugBaseFragment implements InstabugMediaController.OnVisibilityChangedListener {

    public static final String TAG = "VideoPlayerFragment";
    private static final String VIDEO_PATH = "video.uri";
    @Nullable
    private View videoViewToolbar;
    @Nullable
    private VideoView videoView;
    private int position = 0;
    @Nullable
    private IBGProgressDialog progressDialog;
    @Nullable
    private InstabugMediaController mediaControls;
    @Nullable
    private String videoUri;

    public static VideoPlayerFragment newInstance(String videoUri) {
        VideoPlayerFragment fragment = new VideoPlayerFragment();
        Bundle bundle = new Bundle();
        bundle.putString(VIDEO_PATH, videoUri);
        fragment.setArguments(bundle);
        return fragment;
    }

    @Override
    protected int getLayout() {
        return R.layout.instabug_lyt_video_view;
    }

    @Override
    public void onViewCreated(@NonNull View view, @Nullable Bundle savedInstanceState) {
        super.onViewCreated(view, savedInstanceState);
        videoView = view.findViewById(R.id.video_view);
        videoViewToolbar = view.findViewById(R.id.ib_bg_toolbar_video);

        if (videoViewToolbar != null) {
            videoViewToolbar.setOnClickListener(v -> {
                if (getActivity() != null) {
                    getActivity().onBackPressed();
                }
            });
        }
    }

    @Override
    public void onResume() {
        super.onResume();
        showToolbar(false);
    }

    @Override
    public void onActivityCreated(@Nullable Bundle savedInstanceState) {
        super.onActivityCreated(savedInstanceState);

        //set the media controller buttons
        FragmentActivity activity = getActivity();
        if (activity != null) {
            if (mediaControls == null) {
                mediaControls = new InstabugMediaController(activity, this);
            }

            // create a progress bar while the video file is loading
            progressDialog = new IBGProgressDialog.Builder()
                    .setMessage("Loading...")
                    .build(activity);
            // show the progress bar
            progressDialog.show();
            try {
                //set the media controller in the VideoView
                if (videoView != null && videoUri != null) {
                    videoView.setMediaController(mediaControls);
                    //set the uri of the video to be played
                    videoView.setVideoURI(Uri.parse(videoUri));
                }

            } catch (Exception e) {
                InstabugSDKLogger.e(Constants.LOG_TAG, "Couldn't play video due to: ", e);
            }

            if (videoView != null) {
                videoView.requestFocus();
                //we also set an setOnPreparedListener in order to know when the video file is ready for playback
                videoView.setOnPreparedListener(new MediaPlayer.OnPreparedListener() {

                    @Override
                    @SuppressLint("NULL_DEREFERENCE")
                    public void onPrepared(MediaPlayer mediaPlayer) {
                        // close the progress bar and play the video
                        if (progressDialog != null) {
                            progressDialog.dismiss();
                        }
                        //if we have a position on savedInstanceState, the video playback should start from here
                        if (videoView != null) {
                            videoView.seekTo(position);
                            if (position == 0) {
                                videoView.start();
                                if (mediaControls != null) {
                                    mediaControls.show();
                                }
                            } else {
                                //if we come from a resumed activity, video playback will be paused
                                videoView.pause();
                            }
                        }
                    }
                });

                videoView.setOnErrorListener((mp, what, extra) -> {
                    if (progressDialog != null) {
                        progressDialog.dismiss();
                    }
                    return false;
                });
            }
        }
    }


    @Override
    @SuppressLint("ERADICATE_RETURN_NOT_NULLABLE")
    protected String getTitle() {
        return getLocalizedString(R.string.instabug_str_video_player);
    }

    @Override
    public void onDestroy() {
        super.onDestroy();
        showToolbar(true);
    }

    @Override
    protected void consumeNewInstanceSavedArguments() {
        videoUri = getArguments() == null ? null : getArguments().getString(VIDEO_PATH);
    }

    @Override
    protected void restoreState(Bundle savedInstanceState) {
        position = savedInstanceState.getInt("Position");
        if (videoView != null) {
            videoView.seekTo(position);
        }

    }

    @Override
    protected void saveState(Bundle outState) {
        if (videoView != null) {
            outState.putInt("Position", videoView.getCurrentPosition());
            videoView.pause();
        }
    }

    @Override
    public void onDestroyView() {
        mediaControls = null;
        videoView = null;
        videoViewToolbar = null;
        super.onDestroyView();
    }

    private void showToolbar(boolean shouldShow) {
        FragmentActivity activity = getActivity();
        if (activity != null) {
            ActionBar supportActionBar = ((AppCompatActivity) activity).getSupportActionBar();
            if (supportActionBar != null) {
                if (shouldShow)
                    supportActionBar.show();
                else
                    supportActionBar.hide();
            }
        }
    }

    @Override
    public void isVisible(boolean isVisible) {
        if (videoViewToolbar != null) {
            videoViewToolbar.setVisibility(isVisible ? View.VISIBLE : View.GONE);
        }
    }


}
