package com.instabug.bug.configurations

import com.instabug.bug.preferences.PrefSpecs
import com.instabug.bug.preferences.bugsPref
import com.instabug.bug.preferences.bugsPreferences
import com.instabug.bug.settings.BugSettings
import com.instabug.library.IBGFeature
import com.instabug.library.core.InstabugCore
import com.instabug.library.internal.sharedpreferences.corePref
import com.instabug.library.visualusersteps.ReproConfigurationsProvider
import com.instabug.library.visualusersteps.ReproProxyAuthID

interface BugReportingConfigurationsProvider : ReproConfigurationsProvider {
    var isBugReportingUsageExceeded: Boolean
    fun restoreFromCache()
    fun setLastBugRequestStartedAt(time: Long)
    fun setBugReportingLimitedUntil(period: Int)
    val isBugReportingRateLimited: Boolean
    var isUserConsentAvailable: Boolean
    var consentKeyLimitCharacterLimit: Int
    var consentDescriptionChaLimit: Int
    var consentsLimit: Int
    var lastFetchedLocale: String
}

object BugReportingConfigurationsProviderImpl : BugReportingConfigurationsProvider {
    private const val BUG_REPORTING_USAGE_EXCEEDED_KEY = "bug_reporting_usage_exceeded"
    private const val BUG_REPORTING_REQUEST_STARTED_AT_KEY = "last_bug_reporting_request_started_at"
    private const val BUG_REPORTING_LIMITED_UNTIL_KEY = "bug_reporting_rate_limited_until"
    private const val BUG_REPORTING_USAGE_EXCEEDED_DEFAULT = false
    const val USER_CONSENT_FEATURE_KEY = "user_consent"
    const val USER_CONSENT_LIMIT_KEY = "user_consent_limit"
    const val USER_CONSENT_DEFAULT_VALUE = true
    const val USER_CONSENT_LIMIT_DEFAULT_VALUE = 6

    @Volatile
    private var inMemoryUsageExceeded = BUG_REPORTING_USAGE_EXCEEDED_DEFAULT
    private var isCacheLoaded = false
    private val sharedPreferences
        get() = bugsPreferences
    private val editor
        get() = sharedPreferences?.edit()
    override var isBugReportingUsageExceeded: Boolean
        get() {
            return if (isCacheLoaded) inMemoryUsageExceeded
            else {
                restoreUsageExceededFromCache()
                inMemoryUsageExceeded
            }
        }
        set(value) {
            inMemoryUsageExceeded = value
            isCacheLoaded = true
            editor
                ?.putBoolean(BUG_REPORTING_USAGE_EXCEEDED_KEY, value)
                ?.apply()
        }
    private val lastRequestStartedAt: Long
        get() = sharedPreferences
            ?.getLong(BUG_REPORTING_REQUEST_STARTED_AT_KEY, 0L)
            ?: 0L
    private val bugReportingLimitedUntil: Long
        get() = sharedPreferences
            ?.getLong(BUG_REPORTING_LIMITED_UNTIL_KEY, 0L)
            ?: 0L
    override val isBugReportingRateLimited: Boolean
        get() {
            val start = lastRequestStartedAt
            val limitedUntil = bugReportingLimitedUntil
            val currentTime = System.currentTimeMillis()
            return start != 0L && limitedUntil != 0L &&
                    currentTime > start && currentTime < limitedUntil
        }
    override var isUserConsentAvailable: Boolean
        get() = sharedPreferences?.getBoolean(USER_CONSENT_FEATURE_KEY, USER_CONSENT_DEFAULT_VALUE)
            ?: USER_CONSENT_DEFAULT_VALUE
        set(value) {
            editor?.putBoolean(USER_CONSENT_FEATURE_KEY, value)?.apply()
        }

    override var consentKeyLimitCharacterLimit: Int = 24

    override var consentDescriptionChaLimit: Int = 125

    override var consentsLimit: Int by corePref(
        key = USER_CONSENT_LIMIT_KEY,
        defaultValue = USER_CONSENT_LIMIT_DEFAULT_VALUE
    )

    override var lastFetchedLocale: String by bugsPref(PrefSpecs.LastFetchedLocale)
    override fun restoreFromCache() {
        restoreUsageExceededFromCache()
    }

    override fun setLastBugRequestStartedAt(time: Long) {
        editor
            ?.putLong(BUG_REPORTING_REQUEST_STARTED_AT_KEY, time)
            ?.apply()
    }

    override fun setBugReportingLimitedUntil(period: Int) {
        period.toLong()
            .times(1000)
            .plus(lastRequestStartedAt)
            .let { limitedUntil ->
                editor
                    ?.putLong(BUG_REPORTING_LIMITED_UNTIL_KEY, limitedUntil)
                    ?.apply()
            }
    }

    private fun restoreUsageExceededFromCache() = synchronized(this) {
        val isUsageExceeded = sharedPreferences
            ?.getBoolean(
                BUG_REPORTING_USAGE_EXCEEDED_KEY,
                BUG_REPORTING_USAGE_EXCEEDED_DEFAULT
            )
            ?: BUG_REPORTING_USAGE_EXCEEDED_DEFAULT
        isCacheLoaded = true
        inMemoryUsageExceeded = isUsageExceeded
    }

    override val reproProxyAuthId: Int = ReproProxyAuthID.Bugs

    override var isReproStepsEnabledSDK: Boolean = true
    override val isReproStepsEnabled: Boolean
        get() = isReproStepsEnabledSDK && isReproFeaturesAvailable && isBugReportingEnabled

    override var isReproScreenshotsAvailable: Boolean by bugsPref(PrefSpecs.ReproAttachmentsAvailability)
    override var isReproScreenShotsEnabledSDK: Boolean = true

    override val isReproScreenshotsEnabled: Boolean
        get() = isReproScreenShotsEnabledSDK && isReproScreenshotsAvailable &&
                isReproFeaturesAvailable && isBugReportingEnabled


    private val isBugReportingEnabled: Boolean
        get() = BugSettings.getInstance().isBugReportingStateEnabled
    private val isReproFeaturesAvailable: Boolean
        get() = InstabugCore.isFeatureAvailable(IBGFeature.REPRO_STEPS)
}