package com.instabug.chat;

import android.annotation.SuppressLint;
import android.content.Context;
import android.content.Intent;
import android.net.Uri;

import androidx.annotation.Nullable;

import com.instabug.bug.R;
import com.instabug.chat.cache.CacheUtility;
import com.instabug.chat.cache.ChatsCacheManager;
import com.instabug.chat.network.InstabugPushNotificationTokenJob;
import com.instabug.chat.settings.ChatSettings;
import com.instabug.chat.synchronization.SynchronizationManager;
import com.instabug.library.Feature;
import com.instabug.library.IBGFeature;
import com.instabug.library.Instabug;
import com.instabug.library.core.InstabugCore;
import com.instabug.library.core.plugin.PluginPromptOption;
import com.instabug.bug.invocation.InvocationManager;
import com.instabug.library.invocation.InvocationMode;
import com.instabug.bug.invocation.invocationdialog.InstabugDialogActivity;

import java.util.ArrayList;

class ChatPluginWrapper {

    private static void initSynchronization(Context context) {
        SynchronizationManager.init(context);
    }

    static void wake(Context context) {
        ChatSettings.init(context);
        CacheUtility.prepareCache(context);
        CacheUtility.cleanupChats();
        initSynchronization(context);
    }

    static void release() {
        releaseSynchronization();
        CacheUtility.dumpCache();
        ChatSettings.release();
    }

    @SuppressLint("NULL_DEREFERENCE")
    private static void releaseSynchronization() {
        SynchronizationManager.getInstance().release();
    }

    static long getLastActivityTime() {
        return ChatSettings.getLastChatTime();
    }

    static ArrayList<PluginPromptOption> getPromptOptions(Context context) {
        ArrayList<PluginPromptOption> chatPluginPromptOptions = new ArrayList<>();
        if (isChatFeatureEnabled()) {
            if (InstabugCore.isFeatureEnabled(IBGFeature.REPLIES) && RepliesWrapper.hasChats()) {
                chatPluginPromptOptions.add(getChatsListOption(context));
            }
        }
        return chatPluginPromptOptions;
    }

    private static PluginPromptOption getChatsListOption(final Context context) {
        PluginPromptOption promptOption = new PluginPromptOption();
        promptOption.setOrder(PluginPromptOption.NO_ORDER);
        promptOption.setInvocationMode(InvocationMode.CHATS);
        promptOption.setInitialScreenshotRequired(false);
        promptOption.setNotificationCount(ChatsCacheManager.getUnreadCount());
        promptOption.setPromptOptionIdentifier(PluginPromptOption.PromptOptionIdentifier.CHAT_LIST);
        promptOption.setIcon(R.drawable.ibg_core_ic_talk_to_us);
        promptOption.setOnInvocationListener(new PluginPromptOption.OnInvocationListener() {
            @Override
            public void onInvoke(@Nullable Uri initialScreenshotUri, String... reportCategories) {
                // to close prompt dialog if opened
                Intent intent = InstabugDialogActivity.getIntent(context, null, null, null, true);
                context.startActivity(intent);
                RepliesWrapper.show();
            }
        });
        return promptOption;
    }


    private static boolean isChatFeatureEnabled() {
        return InstabugCore.getFeatureState(IBGFeature.IN_APP_MESSAGING) == Feature.State.ENABLED;
    }

    public static void initDefaultPromptOptionAvailabilityState() {
        InvocationManager.getInstance().notifyInvocationOptionChanged();

    }

    public static void sendPushNotificationToken(boolean forceSend) {

        if(forceSend) {
            InstabugCore.setPushNotificationTokenSent(false);
        }

        String token = InstabugCore.getPushNotificationToken();
        if (!InstabugCore.isPushNotificationTokenSent()
                && token != null
                && !token.isEmpty()) {
            Context context = Instabug.getApplicationContext();
            if (context != null) {
                InstabugPushNotificationTokenJob.getInstance().start();

            }
        }
    }

}
