package com.instabug.bug.userConsent

import com.instabug.bug.Constants
import com.instabug.bug.configurations.BugReportingConfigurationsProvider
import com.instabug.library.util.InstabugSDKLogger

interface UserConsentsManager {
    fun addUserConsent(
        key: String?,
        description: String?,
        isMandatory: Boolean,
        isDefaultChecked: Boolean,
        type: @ActionType String? = null
    )

    fun getConsents(): List<UserConsent>?
}


class UserConsentsManagerImpl(
    private val repository: UserConsentRepository,
    private val userConsentValidator: UserConsentValidator,
    private val configurationsProvider: BugReportingConfigurationsProvider
) : UserConsentsManager {

    override fun addUserConsent(
        key: String?,
        description: String?,
        isMandatory: Boolean,
        isDefaultChecked: Boolean,
        type: String?
    ) {
        synchronized(this) {
            if (!configurationsProvider.isUserConsentAvailable) {
                InstabugSDKLogger.e(
                    Constants.LOG_TAG,
                    Constants.USER_CONSENTS_FEATURE_IS_NOT_AVAILABLE
                )
                return
            }

            userConsentValidator(
                UserConsent(
                    key,
                    description,
                    isMandatory = isMandatory,
                    isChecked = isDefaultChecked,
                    actionType  = type
                ), repository.getConsents().keys
            )?.let { consent ->
                removeConsentByType(type)
                repository.addUserConsent(consent)
            }
        }
    }

    override fun getConsents(): List<UserConsent>? =
        synchronized(this) {
            repository.getConsents().takeIf { configurationsProvider.isUserConsentAvailable }
                ?.values
                ?.map { it.copy() }
                ?.toList()
        }


    private fun removeConsentByType(consentType: @ActionType String?) {
        getConsentKeyByType(consentType)?.let { key ->
            repository.removeConsent(key)
        }
    }

    private fun getConsentKeyByType(consentType: @ActionType String?): String? =
        consentType?.let {
            repository.getConsents().filter { entry -> entry.value.actionType == it }
        }?.takeIf { filteredMap ->
            filteredMap.isNotEmpty()
        }?.keys?.first()
}