package com.instabug.bug.view.actionList.service;

import android.content.Context;

import androidx.annotation.NonNull;
import androidx.annotation.Nullable;
import androidx.annotation.VisibleForTesting;

import com.instabug.bug.Constants;
import com.instabug.bug.di.ServiceLocator;
import com.instabug.bug.settings.BugSettings;
import com.instabug.library.IBGNetworkWorker;
import com.instabug.library.Instabug;
import com.instabug.library.InstabugNetworkJob;
import com.instabug.library.networkv2.NetworkManager;
import com.instabug.library.networkv2.RequestResponse;
import com.instabug.library.networkv2.request.Endpoints;
import com.instabug.library.networkv2.request.Request;
import com.instabug.library.networkv2.request.RequestMethod;
import com.instabug.library.networkv2.request.RequestParameter;
import com.instabug.library.networkv2.request.RequestType;
import com.instabug.library.util.InstabugSDKLogger;
import com.instabug.library.util.LocaleUtils;
import com.instabug.library.util.TaskDebouncer;
import com.instabug.library.util.TimeUtils;

import org.json.JSONArray;
import org.json.JSONException;


public class FetchReportCategoriesJob extends InstabugNetworkJob {
    private static final String LOCALE_KEY = "locale";
    private static final long TIME_24_HOURS = 86_400_000;
    private final TaskDebouncer taskDebouncer;
    private final NetworkManager networkManager;
    @Nullable
    private static FetchReportCategoriesJob INSTANCE;

    private FetchReportCategoriesJob() {
        networkManager = ServiceLocator.getNetworkManager();
        taskDebouncer = ServiceLocator.getTaskDebouncer();
    }

    public static synchronized FetchReportCategoriesJob getInstance() {
        if (INSTANCE == null) {
            INSTANCE = new FetchReportCategoriesJob();
        }

        return INSTANCE;
    }

    private void getReportCategories(@NonNull String locale) {
        InstabugSDKLogger.d(Constants.LOG_TAG, "Getting report categories for this application");
        // create getReportCategories request.
        Request categoriesRequest = new Request.Builder()
                .endpoint(Endpoints.REPORT_CATEGORIES)
                .method(RequestMethod.GET)
                .addParameter(new RequestParameter<>(LOCALE_KEY, locale))
                .hasUuid(false)
                .build();

        // do request with NORMAL request type.
        networkManager.doRequest(IBGNetworkWorker.CORE, RequestType.NORMAL, categoriesRequest, new Request.Callbacks<RequestResponse, Throwable>() {
            @Override
            public void onSucceeded(RequestResponse requestResponse) {
                InstabugSDKLogger.d(Constants.LOG_TAG,
                        "getReportCategories request Succeeded, Response code: "
                                + requestResponse.getResponseCode());
                InstabugSDKLogger.v(Constants.LOG_TAG,
                        "getReportCategories request Succeeded, Response body: "
                                + requestResponse.getResponseBody());
                setLastFetchedTime(System.currentTimeMillis());
                ServiceLocator.getConfigurationsProvider().setLastFetchedLocale(locale);
                String response = (String) requestResponse.getResponseBody();
                try {
                    if (response != null) {
                        JSONArray jsonArray = new JSONArray(response);
                        if (jsonArray.length() == 0) {
                            onGetResponse(null);
                        } else {
                            onGetResponse(response);
                        }
                    }
                } catch (JSONException e) {
                    e.printStackTrace();
                }
            }

            @Override
            public void onFailed(Throwable error) {
                InstabugSDKLogger.e(Constants.LOG_TAG, "getReportCategories request got error"
                        , error);
            }
        });
    }


    @VisibleForTesting
    public static long getLastFetchedAt() {
        return BugSettings.getInstance().getReportCategoriesLastFetchedTime();
    }

    @VisibleForTesting
    static void setLastFetchedTime(long time) {
        BugSettings.getInstance().setReportCategoriesLastFetchedTime(time);
    }

    public void resetLastFetchTimeout() {
        setLastFetchedTime(0);
        taskDebouncer.resetLastRun();
    }

    public void start() {

        String currentLocale = getCurrentLocale();

        if (shouldFetch(currentLocale)) {
            enqueueJob(IBGNetworkWorker.CORE, () -> taskDebouncer.debounce(() -> {
                if (Instabug.getApplicationContext() != null && currentLocale != null) {
                    try {
                        getReportCategories(currentLocale);
                    } catch (Exception exception) {
                        InstabugSDKLogger.e(Constants.LOG_TAG, "Error occurred while getting report categories", exception);
                    }
                } else {
                    InstabugSDKLogger.d(Constants.LOG_TAG, "Context was null while getting report categories");
                }
            }));
        }
    }

    @Nullable
    private static String getCurrentLocale() {
        Context applicationContext = Instabug.getApplicationContext();
        if (applicationContext != null)
            return LocaleUtils.getCurrentLocaleResolved(applicationContext);
        return null;
    }

    /**
     * Determines whether to fetch report categories based on the provided locale.
     *
     * @param locale The current locale resolved from the SDK/device.
     * @return true if the current locale is different from the last fetched locale or if 24 hours have passed since the last fetch.
     */
    private static boolean shouldFetch(@Nullable String locale) {
        String lastFetchedLocale = ServiceLocator.getConfigurationsProvider().getLastFetchedLocale();
        return (locale != null && !locale.equals(lastFetchedLocale)) ||
                TimeUtils.hasXHoursPassed(getLastFetchedAt(), TIME_24_HOURS);
    }

    private static void onGetResponse(@Nullable String response) {
        BugSettings.getInstance().setRemoteReportCategories(response);
    }

    @VisibleForTesting
    static void reset() {
        INSTANCE = null;
    }
}
