package com.instabug.bug.screenshot.viewhierarchy.utilities;

import android.annotation.SuppressLint;
import android.content.Context;
import android.net.Uri;
import android.view.View;

import androidx.annotation.Nullable;

import com.instabug.bug.Constants;
import com.instabug.bug.screenshot.viewhierarchy.ViewHierarchy;
import com.instabug.bug.screenshot.viewhierarchy.ViewHierarchyInspector;
import com.instabug.library.internal.storage.DiskUtils;
import com.instabug.library.util.InstabugSDKLogger;

import java.io.File;
import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.util.zip.ZipEntry;
import java.util.zip.ZipOutputStream;

public class ViewHierarchyDiskUtils {

    public static void saveViewHierarchyImage(ViewHierarchy viewHierarchy) {
        View view = viewHierarchy.getView();
        if (view != null) {
            File bitmapFile = new File(getViewHierarchyImagesDirectory(view.getContext()).getAbsolutePath()
                    + File.separator + viewHierarchy.getId() + ".png");
            try {
                if (viewHierarchy.getImage() != null) {
                    DiskUtils.saveBitmapOnDisk(viewHierarchy.getImage(), bitmapFile);
                    viewHierarchy.setImageUriOnDisk(Uri.fromFile(bitmapFile));
                } else {
                    InstabugSDKLogger.e(Constants.LOG_TAG, "trying to save a null value bitmap");
                }
            } catch (IOException e) {
                InstabugSDKLogger.e(Constants.LOG_TAG, "save viewHierarchy image got error: " + e.getMessage() , e);
            }
        }
    }

    @Nullable
    @SuppressLint("RESOURCE_LEAK")
    public static synchronized Uri zipViewHierarchyImages(ViewHierarchy viewHierarchy) {
        View view = viewHierarchy.getView();
        if (view != null && view.getContext() != null) {
            InstabugSDKLogger.v(Constants.LOG_TAG, "zip viewHierarchy images just started");

            ZipOutputStream zipOutputStream = null;
            try {
                File zipFile = new File(
                        DiskUtils.getInstabugInternalDirectory(view.getContext()) + File.separator +
                                "view_hierarchy_attachment_" +
                                System.currentTimeMillis() + ".zip");
                OutputStream outputStream = new FileOutputStream(zipFile);
                zipOutputStream = new ZipOutputStream(outputStream);
                for (ViewHierarchy childViewHierarchy : ViewHierarchyInspector
                        .convertViewHierarchyToList(viewHierarchy)) {
                    if (childViewHierarchy.getImageUriOnDisk() != null &&
                            childViewHierarchy.getImageUriOnDisk().getPath() != null) {
                        File child = new File(childViewHierarchy.getImageUriOnDisk().getPath());
                        zipOutputStream.putNextEntry(new ZipEntry(childViewHierarchy.getId() + ".png"));
                        copy(child, zipOutputStream);
                        zipOutputStream.closeEntry();
                        if (child.delete())
                            InstabugSDKLogger.v(Constants.LOG_TAG, "file zipped successfully");
                    }
                }
                InstabugSDKLogger.v(Constants.LOG_TAG, "zip viewHierarchy images done successfully");
                return Uri.fromFile(zipFile);
            } catch (IOException e) {
                InstabugSDKLogger.e(Constants.LOG_TAG, "zip viewHierarchy images got error: " + e.getMessage() , e);
                return null;
            } finally {
                try {
                    if (zipOutputStream != null) {
                        zipOutputStream.close();
                    }
                } catch (IOException e) {
                    e.printStackTrace();
                }
            }
        }
        return null;
    }

    private static void copy(File file, OutputStream out) throws IOException {
        InputStream in = new FileInputStream(file);
        try {
            copy(in, out);
        } finally {
            in.close();
        }
    }

    private static void copy(InputStream in, OutputStream out) throws IOException {
        byte[] buffer = new byte[2048];
        while (true) {
            int readCount = in.read(buffer);
            if (readCount < 0) {
                break;
            }
            out.write(buffer, 0, readCount);
        }
    }

    public static File getViewHierarchyImagesDirectory(Context context) {
        String viewHierarchyImagesDirectoryPath =
                DiskUtils.getInstabugInternalDirectory(context)
                        + "/view-hierarchy-images/";
        File viewHierarchyImagesDirectory = new File(viewHierarchyImagesDirectoryPath);
        if (!viewHierarchyImagesDirectory.exists()) {
            if (viewHierarchyImagesDirectory.mkdir())
                InstabugSDKLogger.v(Constants.LOG_TAG, "Temp directory for view hierarchy images created successfully");
        }
        return viewHierarchyImagesDirectory;
    }
}
