package com.instabug.bug.proactivereporting

import androidx.annotation.VisibleForTesting
import com.instabug.bug.LiveBugManager
import com.instabug.bug.di.ServiceLocator
import com.instabug.bug.network.ProactiveReportsBugsUploaderJob
import com.instabug.bug.proactivereporting.configs.ProactiveReportingConfigProvider
import com.instabug.bug.proactivereporting.ui.ProactiveReportingDialogActivity
import com.instabug.library.PresentationManager
import com.instabug.library.frustratingexperience.FrustratingExperienceEvent
import com.instabug.library.tracking.InstabugInternalTrackingDelegate
import com.instabug.library.util.TimeUtils
import com.instabug.library.util.threading.PoolProvider
import java.util.concurrent.TimeUnit

interface FrustratingExperienceEventsHandler {
    fun handle(event: FrustratingExperienceEvent)
}

class FrustratingExperienceEventsHandlerImpl(
    private val configsProvider: ProactiveReportingConfigProvider
) : FrustratingExperienceEventsHandler {

    @VisibleForTesting
    public val pendingFrustratingExperienceReportsIds = mutableMapOf<Long, String?>()

    override fun handle(event: FrustratingExperienceEvent) {
        PoolProvider.postIOTask {
            when (event) {
                is FrustratingExperienceEvent.Detected -> handleFrustratingExperienceDetected(event)
                is FrustratingExperienceEvent.Synced -> handleFrustratingExperienceSynced(event)
                is FrustratingExperienceEvent.BugReportSaved -> handleFrustratingExperienceSaved(
                    event
                )
            }
        }
    }

    private fun handleFrustratingExperienceDetected(event: FrustratingExperienceEvent.Detected) {
        if (shouldShow()) {
            PoolProvider.postDelayedTask({
                PresentationManager.getInstance().show {
                    InstabugInternalTrackingDelegate.getInstance().currentActivity?.let {
                        it.startActivity(
                            ProactiveReportingDialogActivity.getIntent(
                                it,
                                event.type,
                                event.internalId
                            )
                        )
                    }
                }
            }, TimeUnit.SECONDS.toMillis(configsProvider.detectionGap))
        }
    }

    private fun shouldShow() = configsProvider.isEnabled && TimeUtils.hasXHoursPassed(
        configsProvider.lastModalTime,
        TimeUnit.SECONDS.toMillis(configsProvider.modalsGap)
    )

    private fun handleFrustratingExperienceSynced(event: FrustratingExperienceEvent.Synced) {
        if (event.externalId != null) {


            if (LiveBugManager.getInstance().bug != null && LiveBugManager.getInstance().bug?.frustratingExperienceInternalId == event.internalId) {
                LiveBugManager.getInstance().bug?.frustratingExperienceExternalId = event.externalId
            } else {
                val updated = ServiceLocator.getBugReportsDbHelper()
                    .updateFrustratingExperienceExternalId(event.internalId, event.externalId)


                if (!updated) {
                    pendingFrustratingExperienceReportsIds[event.internalId] = event.externalId
                } else {
                    ProactiveReportsBugsUploaderJob.getInstance().start()
                }
            }
        }
    }

    private fun handleFrustratingExperienceSaved(event: FrustratingExperienceEvent.BugReportSaved) {
        if (pendingFrustratingExperienceReportsIds.containsKey(event.internalId)) {
            pendingFrustratingExperienceReportsIds[event.internalId]?.let { externalId ->
                ServiceLocator.getBugReportsDbHelper()
                    .updateFrustratingExperienceExternalId(event.internalId, externalId)
            }?.also {
                pendingFrustratingExperienceReportsIds.remove(event.internalId)
                ProactiveReportsBugsUploaderJob.getInstance().start()
            }

        }
    }
}