package com.instabug.bug.view.visualusersteps.visitedscreens;

import static com.instabug.library.util.DiskUtils.listFilesInDirectory;

import android.content.Context;
import android.graphics.Bitmap;
import android.graphics.BitmapFactory;
import android.net.Uri;

import androidx.annotation.Nullable;

import com.instabug.bug.Constants;
import com.instabug.bug.di.ServiceLocator;
import com.instabug.bug.model.VisitedScreen;
import com.instabug.library.core.InstabugCore;
import com.instabug.library.core.ui.BasePresenter;
import com.instabug.library.internal.storage.DiskUtils;
import com.instabug.library.internal.storage.ProcessedBytes;
import com.instabug.library.internal.storage.operation.DeleteUriDiskOperation;
import com.instabug.library.internal.storage.operation.DiskOperationCallback;
import com.instabug.library.util.FileUtils;
import com.instabug.library.util.InstabugSDKLogger;
import com.instabug.library.visualusersteps.VisualUserStep;
import com.instabug.library.visualusersteps.VisualUserStepsHelper;

import java.io.File;
import java.util.ArrayList;
import java.util.concurrent.TimeUnit;

import io.reactivexport.Observable;
import io.reactivexport.android.schedulers.AndroidSchedulers;
import io.reactivexport.disposables.Disposable;
import io.reactivexport.schedulers.Schedulers;

/**
 * Created by Barakat on 05/05/2019
 */
public class VisualUserStepsListPresenter extends BasePresenter<VisualUserStepsListContract.View>
        implements VisualUserStepsListContract.Presenter {

    private ArrayList<VisitedScreen> uniqueVisitedScreens = new ArrayList<>();
    @Nullable
    private Disposable visitedScreenDisposable;

    public VisualUserStepsListPresenter(VisualUserStepsListContract.View view) {
        super(view);
    }

    @Override
    public void getVisualUserSteps() {
        if (view != null) {
            final VisualUserStepsListContract.View viewRef = this.view.get();
            if (viewRef != null && !uniqueVisitedScreens.isEmpty()) {
                viewRef.populateVisualUserSteps(uniqueVisitedScreens);
                return;
            }
            if (viewRef != null) {
                viewRef.showPreparingDialog();
                visitedScreenDisposable = visitedScreensObservable()
                        .subscribeOn(Schedulers.io())
                        .delay(1, TimeUnit.SECONDS)
                        .observeOn(AndroidSchedulers.mainThread())
                        .subscribe(visitedScreens -> {
                            uniqueVisitedScreens = visitedScreens;
                            viewRef.dismissPreparingDialog();
                            viewRef.populateVisualUserSteps(visitedScreens);
                        });
            }
        }
    }

    private Observable<ArrayList<VisitedScreen>> visitedScreensObservable() {
        return Observable.fromCallable(this::fetchThumbnails);
    }

    public ArrayList<VisitedScreen> fetchThumbnails() {
        ArrayList<VisitedScreen> uniqueVisitedScreens = new ArrayList<>();
        int trueIndex = 1;
        ArrayList<VisualUserStep> visualUserSteps = VisualUserStepsHelper.fetchSteps();

        // Get files in Reprosteps directory
        File directory = ServiceLocator.getReproScreenshotsCacheDir().getCurrentSpanDirectory();
        ArrayList<File> files = directory != null ? listFilesInDirectory(directory) : new ArrayList<>();

        for (VisualUserStep visualUserStep : visualUserSteps) {
            if (visualUserStep.getScreenshotId() != null) {
                String absolutePath = getPathOfScreenshotId(files, visualUserStep.getScreenshotId());
                if (absolutePath != null) {
                    boolean screenshotExist = new File(absolutePath).exists();
                    if (!screenshotExist && visualUserStep.getScreenshotId() != null) {
                        // Increment to skip deleted index
                        trueIndex++;
                    }
                    if (visualUserStep.getScreenId() != null && visualUserStep.getScreenshotId() != null && screenshotExist) {
                        ProcessedBytes processedBytes = InstabugCore.decryptOnTheFly(absolutePath);
                        if (processedBytes.isProcessSuccessful()) {
                            byte[] bitmapBytes = processedBytes.getFileBytes();
                            BitmapFactory.Options options = new BitmapFactory.Options();
                            //decrease decode rate to create a thumb version 1/2 of the original bitmap dimensions, 1/4 in size
                            options.inSampleSize = 2;
                            Bitmap screenshotBitmap = BitmapFactory.decodeByteArray(bitmapBytes, 0, bitmapBytes.length, options);

                            uniqueVisitedScreens.add(
                                    new VisitedScreen(trueIndex++,
                                            visualUserStep.getScreenName(),
                                            visualUserStep.getScreenshotId(),
                                            FileUtils.getPathWithDecryptedFlag(absolutePath),
                                            screenshotBitmap));
                        }
                    }
                }
            }
        }
        return uniqueVisitedScreens;
    }

    @Nullable
    private String getPathOfScreenshotId(ArrayList<File> files, String screenshotId) {
        int extensionIndex = FileUtils.getIndexOfExtension(screenshotId);
        String pathWithoutExtension = screenshotId.substring(0, extensionIndex);
        for (File file : files) {
            if (file.getPath().contains(pathWithoutExtension)) {
                return file.getPath();
            }
        }
        return null;
    }

    @Override
    public void deleteVisualUserStep(Context context, int position, VisitedScreen visitedScreen) {
        InstabugSDKLogger.v(Constants.LOG_TAG, "Deleting visual user step, Screen name: " + visitedScreen);
        if (position >= 0 && uniqueVisitedScreens.size() > position) {
            VisualUserStepsHelper.removeScreenshotId(visitedScreen.getScreenshotId());
            uniqueVisitedScreens.remove(position);
            DiskUtils.with(context)
                    .deleteOperation(new DeleteUriDiskOperation(Uri.parse(visitedScreen.getScreenshotUri())))
                    .executeAsync(new DiskOperationCallback<Boolean>() {
                        @Override
                        public void onSuccess(Boolean args) {
                            InstabugSDKLogger.v(Constants.LOG_TAG, "Deleting visual user steps operation succeeded");
                        }

                        @Override
                        public void onFailure(Throwable t) {
                            InstabugSDKLogger.e(Constants.LOG_TAG, "Deleting visual user steps operation failed due to: " + t.getMessage());
                        }
                    });
            if (view != null) {
                VisualUserStepsListContract.View viewRef = this.view.get();
                if (viewRef != null) {
                    viewRef.populateVisualUserSteps(uniqueVisitedScreens);
                }
            }
        }
    }

    @Override
    public void release() {
        if (visitedScreenDisposable != null && visitedScreenDisposable.isDisposed()) {
            visitedScreenDisposable.dispose();
        }

        VisualUserStepsHelper.encryptExistingSteps();
    }
}
