package com.instabug.bug.model;

import android.annotation.SuppressLint;
import android.content.Context;
import android.net.Uri;

import androidx.annotation.NonNull;
import androidx.annotation.Nullable;

import com.instabug.bug.Constants;
import com.instabug.bug.screenrecording.ExternalAutoScreenRecordHelper;
import com.instabug.bug.userConsent.ActionType;
import com.instabug.library.Feature;
import com.instabug.library.IBGFeature;
import com.instabug.library.core.InstabugCore;
import com.instabug.library.internal.storage.cache.Cacheable;
import com.instabug.library.model.Attachment;
import com.instabug.library.model.BaseReport;
import com.instabug.library.model.State;
import com.instabug.library.util.InstabugSDKLogger;
import com.instabug.library.util.StringUtility;

import org.json.JSONArray;
import org.json.JSONException;
import org.json.JSONObject;

import java.io.Serializable;
import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.concurrent.CopyOnWriteArrayList;

/**
 * @author mesbah.
 */
@SuppressLint("ERADICATE_FIELD_NOT_INITIALIZED")
public class Bug extends BaseReport implements Cacheable, Serializable {

    private static final long serialVersionUID = 1905162041950251407L;

    // keys used by toJson() & fromJson() methods
    static final String KEY_ID = "id";
    static final String KEY_TEMPORARY_SERVER_TOKEN = "temporary_server_token";
    static final String KEY_TYPE = "type";
    static final String KEY_MESSAGE = "message";
    static final String KEY_ATTACHMENTS = "attachments";
    static final String KEY_STATE = "state";
    static final String KEY_BUG_STATE = "bug_state";
    static final String KEY_VIEW_HIERARCHY = "view_hierarchy";
    static final String KEY_CATEGORIES_LIST = "categories_list";
    static final String KEY_CONNECTION_ERROR = "connection_error";
    public static final String KEY_ACTIONABLE_CONSENT = "actionable_consents";
    public static final String CONNECTION_ERROR_DISCONNECTED = "Disconnected";
    public static final String KEY_FRUSTRATING_EXPERIENCE_INTERNAL_ID = "frustrating_experience_internal_id";
    public static final String KEY_FRUSTRATING_EXPERIENCE_EXTERNAL_ID = "frustrating_experience_external_id";
    @Nullable
    private String id;
    @Nullable
    private String temporaryServerToken;
    private String type;
    private String message;
    private List<Attachment> attachments;
    private BugState bugState;
    @Nullable
    private String viewHierarchy;
    private boolean requiredViewHierarchy;
    @Nullable
    private ViewHierarchyInspectionState viewHierarchyInspectionState;
    @Nullable
    private transient List<ExtraReportField> extraReportFields;
    private ArrayList<String> categories;
    @Nullable
    private String initialScreenshotPath;
    @Nullable
    private String attachmentsPath;
    @Nullable
    private Map<String, String> consentResponses;

    @Nullable
    private List<@ActionType String> actionableConsents;

    private String connectionError;

    private long frustratingExperienceInternalId;

    @Nullable
    private String frustratingExperienceExternalId;

    @Nullable
    public String getAttachmentsPath() {
        return attachmentsPath;
    }

    public void setAttachmentsPath(@Nullable String attachmentsPath) {
        this.attachmentsPath = attachmentsPath;
    }


    public enum BugState {
        IN_PROGRESS, READY_TO_BE_SENT, LOGS_READY_TO_BE_UPLOADED, ATTACHMENTS_READY_TO_BE_UPLOADED,
        NOT_AVAILABLE;
    }


    public enum ViewHierarchyInspectionState {
        IN_PROGRESS, FAILED, DONE;
    }

    public Bug() {
        this.bugState = BugState.NOT_AVAILABLE;
        this.type = Constants.ReportType.NOT_AVAILABLE;
    }

    public Bug(@NonNull String id, @Nullable State state, @NonNull BugState bugState) {
        this.id = id;
        this.state = state;
        this.bugState = bugState;
        this.type = Constants.ReportType.NOT_AVAILABLE;
        this.attachments = new CopyOnWriteArrayList<>();
        this.categories = new ArrayList<>();
    }

    @Nullable
    public String getId() {
        return id;
    }

    public Bug setId(@Nullable String id) {
        this.id = id;
        return this;
    }

    @Nullable
    public String getTemporaryServerToken() {
        return temporaryServerToken;
    }

    public Bug setTemporaryServerToken(@Nullable String temporaryServerToken) {
        this.temporaryServerToken = temporaryServerToken;
        return this;
    }

    public void setInitialScreenshotPath(@Nullable String initialScreenshotPath) {
        this.initialScreenshotPath = initialScreenshotPath;
    }

    @Nullable
    public String getInitialScreenshotPath() {
        return initialScreenshotPath;
    }

    @Constants.ReportType
    public String getType() {
        return type;
    }

    public Bug setType(@Constants.ReportType String type) {
        this.type = type;
        return this;
    }

    public String getMessage() {
        return message;
    }

    public Bug setMessage(String message) {
        this.message = message;
        return this;
    }

    public Bug addAttachment(Uri attachmentUri, Attachment.Type type) {
        return addAttachment(attachmentUri, type, false);
    }

    public Bug addAttachment(@Nullable Uri attachmentUri, Attachment.Type type, boolean isReproStepsUriEncrypted) {
        InstabugSDKLogger.v(Constants.LOG_TAG, "Started adding attachments to bug");
        if (attachmentUri == null) {
            InstabugSDKLogger.w(Constants.LOG_TAG, "Adding attachment with a null Uri, ignored.");
            return this;
        }
        if (type == null) {
            InstabugSDKLogger.w(Constants.LOG_TAG, "Adding attachment with a null Attachment.Type, ignored.");
            return this;
        }

        String attachmentFileName = attachmentUri.getLastPathSegment();
        String attachmentFilePath = attachmentUri.getPath();

        for (Attachment addedAttachment : attachments) {
            if (addedAttachment.getType() == type
                    && Objects.equals(attachmentFileName, addedAttachment.getName())
                    && Objects.equals(attachmentFilePath, addedAttachment.getLocalPath())) {
                InstabugSDKLogger.w(Constants.LOG_TAG, "Adding duplicated attachment, ignored.");
                return this;
            }
        }

        Attachment attachment = new Attachment();
        if (attachmentFileName != null) {
            attachment.setName(attachmentFileName);
        }
        if (attachmentFilePath != null) {
            attachment.setLocalPath(attachmentFilePath);
        }
        attachment.setType(type);

        if (attachment.getLocalPath() != null && attachment.getLocalPath().contains("attachments")) {
            attachment.setEncrypted(true);
        }
        if (type == Attachment.Type.VISUAL_USER_STEPS) {
            attachment.setEncrypted(isReproStepsUriEncrypted);
            InstabugSDKLogger.i(Constants.LOG_TAG, "Adding  visual user steps attachments to bug");
        }
        attachments.add(attachment);
        return this;
    }

    public synchronized List<Attachment> getAttachments() {
        return attachments;
    }

    public Bug setAttachments(List<Attachment> attachments) {
        this.attachments = new CopyOnWriteArrayList<>(attachments);
        return this;
    }

    public Bug setState(@Nullable State state) {
        this.state = state;
        return this;
    }

    public BugState getBugState() {
        return bugState;
    }

    public Bug setBugState(BugState bugState) {
        this.bugState = bugState;
        return this;
    }

    @Nullable
    public String getViewHierarchy() {
        return viewHierarchy;
    }

    public Bug setViewHierarchy(String viewHierarchy) {
        this.viewHierarchy = viewHierarchy;
        return this;
    }

    public ArrayList<String> getCategories() {
        return categories;
    }

    public void setCategories(ArrayList<String> categories) {
        this.categories = categories;
    }

    @Nullable
    public String getConnectionError() {
        return connectionError;
    }

    public void setConnectionError(String connectionError) {
        this.connectionError = connectionError;
    }

    @Nullable
    public List<String> getActionableConsents() {
        return actionableConsents;
    }

    public void setActionableConsents(@Nullable List<String> actionableConsents) {
        this.actionableConsents = actionableConsents;
    }


    public long getFrustratingExperienceInternalId() {
        return frustratingExperienceInternalId;
    }

    public void setFrustratingExperienceInternalId(long frustratingExperienceInternalId) {
        this.frustratingExperienceInternalId = frustratingExperienceInternalId;
    }

    @Nullable
    public String getFrustratingExperienceExternalId() {
        return frustratingExperienceExternalId;
    }

    public void setFrustratingExperienceExternalId(@Nullable String frustratingExperienceExternalId) {
        this.frustratingExperienceExternalId = frustratingExperienceExternalId;
    }

    @Override
    @SuppressLint("ERADICATE_PARAMETER_NOT_NULLABLE")
    public String toJson() throws JSONException {
        JSONObject bug = new JSONObject();
        bug.put(KEY_ID, getId())
                .put(KEY_TEMPORARY_SERVER_TOKEN, getTemporaryServerToken())
                .put(KEY_TYPE, getType().toString())
                .put(KEY_MESSAGE, getMessage())
                .put(KEY_BUG_STATE, getBugState().toString())
                .put(KEY_ATTACHMENTS, Attachment.toJson(getAttachments()))
                .put(KEY_VIEW_HIERARCHY, getViewHierarchy())
                .put(KEY_CATEGORIES_LIST, getCategoriesAsJSONArray())
                .put(KEY_ACTIONABLE_CONSENT, getActionableConsentsAsJsonArray())
                .put(KEY_FRUSTRATING_EXPERIENCE_INTERNAL_ID, getFrustratingExperienceInternalId())
                .put(KEY_FRUSTRATING_EXPERIENCE_EXTERNAL_ID, getFrustratingExperienceExternalId());
        if (getState() != null) {
            bug.put(KEY_STATE, getState().toJson());
        }

        if (getConnectionError() != null) {
            bug.put(KEY_CONNECTION_ERROR, connectionError);
        }
        return bug.toString();
    }

    @Override
    @SuppressLint("NULL_DEREFERENCE")
    public void fromJson(String bugASJson) throws JSONException {
        JSONObject bugJsonObject = new JSONObject(bugASJson);
        if (bugJsonObject.has(KEY_ID))
            setId(bugJsonObject.getString(KEY_ID));
        if (bugJsonObject.has(KEY_TEMPORARY_SERVER_TOKEN))
            setTemporaryServerToken(bugJsonObject.getString(KEY_TEMPORARY_SERVER_TOKEN));
        if (bugJsonObject.has(KEY_TYPE)) {
            String type;
            switch (bugJsonObject.getString(KEY_TYPE)) {
                case Constants.ReportType.BUG:
                    type = Constants.ReportType.BUG;
                    break;
                case Constants.ReportType.FEEDBACK:
                    type = Constants.ReportType.FEEDBACK;
                    break;
                case Constants.ReportType.ASK_QUESTION:
                    type = Constants.ReportType.ASK_QUESTION;
                    break;
                case Constants.ReportType.FRUSTRATING_EXPERIENCE:
                    type = Constants.ReportType.FRUSTRATING_EXPERIENCE;
                    break;
                default:
                    type = Constants.ReportType.NOT_AVAILABLE;
                    break;
            }
            setType(type);
        }
        if (bugJsonObject.has(KEY_MESSAGE))
            setMessage(bugJsonObject.getString(KEY_MESSAGE));
        if (bugJsonObject.has(KEY_BUG_STATE))
            setBugState(BugState.valueOf(bugJsonObject.getString(KEY_BUG_STATE)));
        if (bugJsonObject.has(KEY_STATE)) {
            State state = new State();
            state.fromJson(bugJsonObject.getString(KEY_STATE));
            setState(state);
        }
        if (bugJsonObject.has(KEY_ATTACHMENTS))
            setAttachments(Attachment.fromJson(bugJsonObject.getJSONArray(KEY_ATTACHMENTS)));
        if (bugJsonObject.has(KEY_VIEW_HIERARCHY))
            setViewHierarchy(bugJsonObject.getString(KEY_VIEW_HIERARCHY));
        if (bugJsonObject.has(KEY_CATEGORIES_LIST)) {
            setCategoriesFromJSONArray(bugJsonObject.getJSONArray(KEY_CATEGORIES_LIST));
        }

        if(bugJsonObject.has(KEY_ACTIONABLE_CONSENT)) {
            setActionableConsentsFromJSONArray(bugJsonObject.getJSONArray(KEY_ACTIONABLE_CONSENT));
        }

        if (bugJsonObject.has(KEY_CONNECTION_ERROR)) {
            setConnectionError(bugJsonObject.getString(KEY_CONNECTION_ERROR));
        }

        if (bugJsonObject.has(KEY_FRUSTRATING_EXPERIENCE_INTERNAL_ID)) {
            setFrustratingExperienceInternalId(bugJsonObject.getLong(KEY_FRUSTRATING_EXPERIENCE_INTERNAL_ID));
        }

        if (bugJsonObject.has(KEY_FRUSTRATING_EXPERIENCE_EXTERNAL_ID)) {
            setFrustratingExperienceExternalId(bugJsonObject.getString(KEY_FRUSTRATING_EXPERIENCE_EXTERNAL_ID));
        }
    }

    public JSONArray getCategoriesAsJSONArray() {
        JSONArray jsonArray = new JSONArray();
        if (categories != null) {
            for (String category : categories) {
                jsonArray.put(category);
            }
        }
        return jsonArray;
    }

    public void setCategoriesFromJSONArray(JSONArray categoriesArray) throws JSONException {
        ArrayList<String> categories = new ArrayList<>();

        for (int i = 0; i < categoriesArray.length(); i++) {
            categories.add(categoriesArray.getString(i));
        }

        setCategories(categories);
    }


    public JSONArray getActionableConsentsAsJsonArray() {
        JSONArray jsonArray = new JSONArray();
        if (actionableConsents != null) {
            for (String type : actionableConsents) {
                jsonArray.put(type);
            }
        }
        return jsonArray;
    }

    public void setActionableConsentsFromJSONArray(JSONArray actionableConsents) throws JSONException {
        ArrayList<String> actions = new ArrayList<>();

        for (int i = 0; i < actionableConsents.length(); i++) {
            actions.add(actionableConsents.getString(i));
        }

        setActionableConsents(actions);
    }

    public int getVisibleAttachmentsCount() {
        int count = 0;
        for (Attachment attachment : getAttachments()) {
            if (attachment.getType() == Attachment.Type.MAIN_SCREENSHOT
                    || attachment.getType() == Attachment.Type.EXTRA_IMAGE
                    || attachment.getType() == Attachment.Type.GALLERY_IMAGE
                    || attachment.getType() == Attachment.Type.EXTRA_VIDEO
                    || attachment.getType() == Attachment.Type.GALLERY_VIDEO
                    || attachment.getType() == Attachment.Type.AUDIO)
                count++;
        }
        return count;
    }

    public boolean hasMainScreenshot() {
        for (Attachment attachment : getAttachments()) {
            if (attachment.getType() == Attachment.Type.MAIN_SCREENSHOT) {
                return true;
            }
        }
        return false;
    }

    public boolean isRequiredViewHierarchy() {
        return requiredViewHierarchy;
    }

    public Bug setRequiredViewHierarchy(boolean requiredViewHierarchy) {
        this.requiredViewHierarchy = requiredViewHierarchy;
        return this;
    }

    @Nullable
    public ViewHierarchyInspectionState getViewHierarchyInspectionState() {
        return viewHierarchyInspectionState;
    }

    public Bug setViewHierarchyInspectionState(ViewHierarchyInspectionState viewHierarchyInspectionState) {
        this.viewHierarchyInspectionState = viewHierarchyInspectionState;
        return this;
    }

    @Nullable
    public List<ExtraReportField> getExtraReportFields() {
        return extraReportFields;
    }

    public void setExtraReportFields(@Nullable List<ExtraReportField> extraReportFields) {
        this.extraReportFields = extraReportFields;
    }

    public void addCategory(String category) {
        categories.add(category);
    }

    public String getCategoriesString() {
        return StringUtility.toCommaSeparated(categories);
    }

    public void setUserConsentResponses(@Nullable Map<String, String> consentResponses) {
        this.consentResponses = consentResponses;
    }

    @Nullable
    public Map<String, String> getUserConsentResponses() {
        return consentResponses;
    }

    @Override
    public String toString() {
        return "Internal Id: " + id + ", TemporaryServerToken:" + temporaryServerToken
                + ", Message:" + message + ", Type:" + type + ", Connection Error: " + connectionError;
    }

    @Override
    @SuppressLint("ERADICATE_INCONSISTENT_SUBCLASS_PARAMETER_ANNOTATION")
    public boolean equals(Object bug) {
        if (bug != null && bug instanceof Bug) {
            Bug comparedBug = (Bug) bug;
            if (String.valueOf(comparedBug.getId()).equals(String.valueOf(getId()))
                    && String.valueOf(comparedBug.getMessage()).equals(String.valueOf(getMessage()))
                    && String.valueOf(comparedBug.getTemporaryServerToken()).equals(String.valueOf(getTemporaryServerToken()))
                    && comparedBug.getBugState() == getBugState()
                    && comparedBug.getState() != null && comparedBug.getState().equals(getState())
                    && comparedBug.getType() != null && comparedBug.getType().equals(getType())
                    && comparedBug.getAttachments() != null
                    && comparedBug.getAttachments().size() == getAttachments().size()) {
                for (int i = 0; i < comparedBug.getAttachments().size(); i++) {
                    if (!(comparedBug.getAttachments().get(i).equals(getAttachments().get(i))))
                        return false;
                }
                return true;
            } else {
                return false;
            }
        } else {
            return false;
        }
    }

    @Override
    public int hashCode() {
        if (getId() != null)
            return getId().hashCode();
        else
            return -1;
    }

    public static class Factory {
        @SuppressLint("CheckResult")
        public Bug create(final Context context) {
            final Bug bug = new Bug(System.currentTimeMillis() + "", null,
                    BugState.IN_PROGRESS);

            if (ExternalAutoScreenRecordHelper.getInstance().isEnabled()) {
                Uri uri = ExternalAutoScreenRecordHelper.getInstance().getAutoScreenRecordingFileUri();
                ExternalAutoScreenRecordHelper.getInstance().clear();
                if (uri != null) {
                    Attachment attachment = new Attachment();
                    attachment.setName(uri.getLastPathSegment());
                    attachment.setLocalPath(uri.getPath());
                    attachment.setType(Attachment.Type.AUTO_SCREEN_RECORDING_VIDEO);

                    bug.getAttachments().add(attachment);
                }
            }

            boolean isRequiredViewHierarchy = InstabugCore.getFeatureState(IBGFeature.VIEW_HIERARCHY_V2) == Feature.State.ENABLED;
            bug.setRequiredViewHierarchy(isRequiredViewHierarchy);
            return bug;
        }
    }
}
