package com.instabug.apm.networkinterception.configuration

import com.instabug.apm.configuration.APMConfigurationProvider
import com.instabug.apm.configuration.APMPreferencePropertyFactory
import com.instabug.apm.di.Provider

interface NetworkInterceptionConfigurationProvider {

    var keywordSanitizationFeatureEnabled: Boolean
    var iBGSanitizationKeywords: Set<String>
    var networkSpansFeatureEnabled: Boolean
    var cpNativeInterceptionFeatureEnabled: Boolean
    val isNetworkSpansEnabled: Boolean
    val isCPNativeInterceptionEnabled: Boolean
    val isNativeInterceptionEnabled: Boolean
    val isGrpcInterceptionEnabled: Boolean
    val isUrlConnectionInterceptionEnabled: Boolean
    fun isKeywordSanitizationEnabled(): Boolean
    fun resetIBGSanitizationKeywords()
    fun resetKeywordSanitizationConfigurations()
    fun reset()
}

class NetworkInterceptionConfigurationProviderImpl(
    private val apmConfig: APMConfigurationProvider,
    preferencePropertyFactory: APMPreferencePropertyFactory,
    private val isCurrentPlatformCpProvider: Provider<Boolean>,
) : NetworkInterceptionConfigurationProvider {

    private val isNetworkEnabled
        get() = apmConfig.isNetworkEnabled

    override var keywordSanitizationFeatureEnabled: Boolean by preferencePropertyFactory.create(
        KEY_IBG_KEYWORD_SANITIZATION_FEATURE_ENABLED,
        DEFAULT_IBG_KEYWORD_SANITIZATION_FEATURE_ENABLED
    )

    override var iBGSanitizationKeywords: Set<String> by preferencePropertyFactory.create(
        KEY_IBG_SANITIZATION_KEYWORDS,
        DEFAULT_IBG_SANITIZATION_KEYWORDS
    )

    override var networkSpansFeatureEnabled: Boolean by preferencePropertyFactory.create(
        KEY_NETWORK_SPANS_ENABLED,
        DEFAULT_NETWORK_SPANS_ENABLED
    )
    override var cpNativeInterceptionFeatureEnabled: Boolean by preferencePropertyFactory.create(
        KEY_CP_NATIVE_INTERCEPTION_ENABLED,
        DEFAULT_CP_NATIVE_INTERCEPTION_ENABLED
    )

    override val isNetworkSpansEnabled: Boolean
        get() = isNativeInterceptionEnabled(cpNativeInterceptionFeatureEnabled && networkSpansFeatureEnabled)

    override val isCPNativeInterceptionEnabled: Boolean
        get() = isNetworkEnabled && cpNativeInterceptionFeatureEnabled

    override val isNativeInterceptionEnabled: Boolean
        get() = isNativeInterceptionEnabled(cpNativeInterceptionFeatureEnabled)

    override val isGrpcInterceptionEnabled: Boolean
        get() = isNativeInterceptionEnabled && apmConfig.isNetworkGrpcInterceptionFeatureEnabled

    override val isUrlConnectionInterceptionEnabled: Boolean
        get() = isNetworkEnabled && !isCurrentPlatformCpProvider()

    override fun isKeywordSanitizationEnabled(): Boolean =
        apmConfig.isAPMEnabled && keywordSanitizationFeatureEnabled

    override fun resetIBGSanitizationKeywords() {
        iBGSanitizationKeywords = DEFAULT_IBG_SANITIZATION_KEYWORDS
    }

    override fun resetKeywordSanitizationConfigurations() {
        keywordSanitizationFeatureEnabled = DEFAULT_IBG_KEYWORD_SANITIZATION_FEATURE_ENABLED
        resetIBGSanitizationKeywords()
    }

    override fun reset() {
        resetKeywordSanitizationConfigurations()
        networkSpansFeatureEnabled = DEFAULT_NETWORK_SPANS_ENABLED
        cpNativeInterceptionFeatureEnabled = DEFAULT_CP_NATIVE_INTERCEPTION_ENABLED
    }

    private fun isNativeInterceptionEnabled(cpCondition: Boolean) =
        isNetworkEnabled && (!isCurrentPlatformCpProvider() || (cpCondition))
}
