package com.instabug.apm.handler.uitrace.uiloading;

import static com.instabug.apm.constants.UiLoadingMetric.ACTIVITY_EVENTS_COUNT;
import static com.instabug.apm.constants.UiLoadingMetric.ON_ACTIVITY_CREATED;
import static com.instabug.apm.constants.UiLoadingMetric.ON_ACTIVITY_RESUMED;
import static com.instabug.apm.constants.UiLoadingMetric.ON_ACTIVITY_RESUMED_POST_RUN;
import static com.instabug.apm.constants.UiLoadingMetric.ON_ACTIVITY_STARTED;
import static com.instabug.apm.constants.UiLoadingMetric.ON_CUSTOM_SCREEN_LOADING_END;
import static com.instabug.apm.util.ArrayKtxKt.replaceNulls;

import androidx.annotation.Nullable;

import com.instabug.apm.cache.model.UiLoadingModel;
import com.instabug.apm.constants.UiLoadingMetric;
import com.instabug.apm.model.EmptyTimeMetric;
import com.instabug.apm.model.EventTimeMetricCapture;

import java.util.concurrent.TimeUnit;

public class DefaultActivityEventToUiLoadingModelMapper implements ActivityEventToUiLoadingModelMapper {

    @Nullable
    @Override
    public UiLoadingModel getUiLoadingModel(EventTimeMetricCapture[] activityTimeEvents) {
        final EmptyTimeMetric defaultTimeMetric = new EmptyTimeMetric();
        replaceNulls(activityTimeEvents, defaultTimeMetric);
        if (activityTimeEvents.length == ACTIVITY_EVENTS_COUNT) {
            UiLoadingModel uiLoadingModel = new UiLoadingModel();
            uiLoadingModel.setStartTimeStampMicro(getStartTimeStampMicro(activityTimeEvents));
            uiLoadingModel.setDurationInMicro(
                    TimeUnit.NANOSECONDS.toMicros(getTotalTimeNano(activityTimeEvents))
            );
            uiLoadingModel.addStage(
                    UiLoadingMetric.STAGE_CREATED_DURATION_MICRO,
                    TimeUnit.NANOSECONDS.toMicros(getCreatedDurationNano(activityTimeEvents))
            );
            uiLoadingModel.addStage(
                    UiLoadingMetric.STAGE_CREATED_TIMESTAMP_MICRO,
                    getCreatedTimestampMicro(activityTimeEvents)
            );
            uiLoadingModel.addStage(
                    UiLoadingMetric.STAGE_STARTED_DURATION_MICRO,
                    TimeUnit.NANOSECONDS.toMicros(getStartedDurationNano(activityTimeEvents))
            );
            uiLoadingModel.addStage(
                    UiLoadingMetric.STAGE_STARTED_TIMESTAMP_MICRO,
                    getStartedTimestampMicro(activityTimeEvents)
            );
            uiLoadingModel.addStage(
                    UiLoadingMetric.STAGE_RESUMED_DURATION_MICRO,
                    TimeUnit.NANOSECONDS.toMicros(getResumedDurationNano(activityTimeEvents))
            );
            uiLoadingModel.addStage(
                    UiLoadingMetric.STAGE_RESUMED_TIMESTAMP_MICRO,
                    getResumedTimestampMicro(activityTimeEvents)
            );
            if (hasCustomEndScreenLoading(activityTimeEvents)) {
                long durationInMicro =
                        TimeUnit.NANOSECONDS.toMicros(getCustomEndScreenLoadingDurationNano(activityTimeEvents));
                uiLoadingModel.addStage(
                        UiLoadingMetric.STAGE_CUSTOM_END_LOADING_DURATION_MICRO,
                        durationInMicro
                );
                if (durationInMicro != 0) {
                    uiLoadingModel.addStage(
                            UiLoadingMetric.STAGE_CUSTOM_END_LOADING_TIMESTAMP_MICRO,
                            getCustomEndScreenLoadingTimestampMicro(activityTimeEvents)
                    );
                }
            }
            return uiLoadingModel;
        }
        return null;
    }

    private boolean hasCustomEndScreenLoading(EventTimeMetricCapture[] activityTimeEvents) {
        return !(activityTimeEvents[ON_CUSTOM_SCREEN_LOADING_END] instanceof EmptyTimeMetric);
    }

    private boolean isCustomEndScreenLoadingValid(EventTimeMetricCapture[] activityTimeEvents) {
        return activityTimeEvents[ON_CUSTOM_SCREEN_LOADING_END].getNanoTime() >
                activityTimeEvents[ON_ACTIVITY_RESUMED_POST_RUN].getNanoTime();
    }

    private long getStartTimeStampMicro(EventTimeMetricCapture[] activityTimeEvents) {
        return activityTimeEvents[ON_ACTIVITY_CREATED].getTimeStampMicro();
    }

    private long getCreatedDurationNano(EventTimeMetricCapture[] activityTimeEvents) {
        return activityTimeEvents[ON_ACTIVITY_STARTED].getNanoTime()
                - activityTimeEvents[ON_ACTIVITY_CREATED].getNanoTime();
    }

    private long getCreatedTimestampMicro(EventTimeMetricCapture[] activityTimeEvents) {
        return activityTimeEvents[ON_ACTIVITY_CREATED].getTimeStampMicro();
    }

    private long getStartedDurationNano(EventTimeMetricCapture[] activityTimeEvents) {
        return activityTimeEvents[ON_ACTIVITY_RESUMED].getNanoTime()
                - activityTimeEvents[ON_ACTIVITY_STARTED].getNanoTime();
    }

    private long getStartedTimestampMicro(EventTimeMetricCapture[] activityTimeEvents) {
        return activityTimeEvents[ON_ACTIVITY_STARTED].getTimeStampMicro();
    }

    private long getResumedDurationNano(EventTimeMetricCapture[] activityTimeEvents) {
        return activityTimeEvents[ON_ACTIVITY_RESUMED_POST_RUN].getNanoTime()
                - activityTimeEvents[ON_ACTIVITY_RESUMED].getNanoTime();
    }

    private long getResumedTimestampMicro(EventTimeMetricCapture[] activityTimeEvents) {
        return activityTimeEvents[ON_ACTIVITY_RESUMED].getTimeStampMicro();
    }

    private long getCustomEndScreenLoadingDurationNano(EventTimeMetricCapture[] activityTimeEvents) {
        if (isCustomEndScreenLoadingValid(activityTimeEvents)) {
            return activityTimeEvents[ON_CUSTOM_SCREEN_LOADING_END].getNanoTime() -
                    activityTimeEvents[ON_ACTIVITY_RESUMED_POST_RUN].getNanoTime();
        }
        return 0;
    }

    private long getCustomEndScreenLoadingTimestampMicro(EventTimeMetricCapture[] activityTimeEvents) {
        return activityTimeEvents[ON_ACTIVITY_RESUMED_POST_RUN].getTimeStampMicro();
    }

    private long getTotalTimeNano(EventTimeMetricCapture[] activityTimeEvents) {
        if (hasCustomEndScreenLoading(activityTimeEvents) && isCustomEndScreenLoadingValid(activityTimeEvents)) {
            return activityTimeEvents[ON_CUSTOM_SCREEN_LOADING_END].getNanoTime()
                    - activityTimeEvents[ON_ACTIVITY_CREATED].getNanoTime();
        } else {
            return activityTimeEvents[ON_ACTIVITY_RESUMED_POST_RUN].getNanoTime()
                    - activityTimeEvents[ON_ACTIVITY_CREATED].getNanoTime();
        }
    }
}
