package com.instabug.apm.cache.handler.applaunch;

import androidx.annotation.NonNull;
import androidx.annotation.Nullable;

import com.instabug.apm.cache.model.AppLaunchCacheModel;
import com.instabug.apm.constants.AppLaunchType;

import java.util.List;

public interface AppLaunchCacheHandler {

    /**
     * Inserts a new app launch into cache
     *
     * @param sessionID           id of the session which this app launch belongs to
     * @param appLaunchCacheModel app launch model to be cached
     * @return id of the inserted app launch
     */
    long insertAppLaunch(String sessionID, AppLaunchCacheModel appLaunchCacheModel);

    /**
     * Gets a list of all cached app launches with limit
     *
     * @param limit max count of retrieved app launches
     * @return app launches list
     */
    @Nullable
    List<AppLaunchCacheModel> getAppLaunches(long limit);

    /**
     * Removes cached app launches with a limit
     *
     * @param limit max number of app launches to be removed
     */
    void removeAppLaunches(long limit);

    /**
     * Removes cached app launches for a specific type with a limit
     *
     * @param type type of the app launches to be removed from
     * @param limit max number of app launches to be removed
     *
     * @see AppLaunchType
     */
    void removeAppLaunches(@AppLaunchType String type, long limit);


    /**
     * Removes all cached app launches for a specific type
     *
     * @param type the type of launches to be cleared
     *
     * @see AppLaunchType
     */
    void removeAppLaunches(@NonNull @AppLaunchType String type);

    /**
     * Removes all app launches with its stages
     */
    void removeAll();

    /**
     * Gets list of cached app launches for passed session id
     *
     * @param sessionID id of the session to get cached app launches for
     * @return List of {@link AppLaunchCacheModel}
     */
    @Nullable
    List<AppLaunchCacheModel> getAppLaunchesForSession(String sessionID);

    /**
     * Removes the oldest app launches by type till the table reach the limit passed for that type
     *
     * @param limit number of traces to be remained
     * @param appLaunchType app launch type to apply the trimming logic on
     */
    void trimAppLaunchesToLimit(long limit, @AppLaunchType String appLaunchType);

    /**
     * Removes oldest app launches by type till
     * total app launches for the passed sessionID reaches the limit passed for that type
     *
     * @param sessionID Session id to limit its app launches
     * @param limit     number of traces to be remained
     * @param appLaunchType app launch type to apply the trimming logic on
     */
    int trimAppLaunchesToLimit(String sessionID, long limit, @AppLaunchType String appLaunchType);

    /**
     * Update an existing AppLaunch
     *
     * @param appLaunch represents the updated app launch model
     */
    void updateAppLaunch(AppLaunchCacheModel appLaunch);
}
