package com.instabug.apm.appflow.handler

import android.content.ContentValues
import android.database.Cursor
import com.instabug.apm.appflow.model.AppFlowAttributeInsertionCacheModel
import com.instabug.apm.appflow.model.AppFlowCacheModel
import com.instabug.apm.appflow.model.AppFlowEndReason
import com.instabug.apm.appflow.model.AppFlowInsertionCacheModel
import com.instabug.apm.logger.internal.Logger
import com.instabug.library.diagnostics.IBGDiagnostics
import com.instabug.library.internal.storage.cache.db.DatabaseManager
import com.instabug.library.internal.storage.cache.db.InstabugDbContract.APMSessionEntry
import com.instabug.library.internal.storage.cache.db.SQLiteDatabaseWrapper
import com.instabug.library.internal.storage.cache.db.contract.apm.ApmAppFlowEntry
import com.instabug.library.internal.storage.cache.db.contract.apm.ApmAppFlowTableAttributeEntry
import com.instabug.library.internal.storage.cache.dbv2.joinToArgs
import com.instabug.library.map.Mapper
import com.instabug.library.model.common.SessionVersion
import com.instabug.library.parse.Parser

class AppFlowCacheHandlerImpl(
    private val databaseManager: DatabaseManager,
    private val logger: Logger,
    private val appFlowContentValuesMapper: Mapper<AppFlowInsertionCacheModel, ContentValues>,
    private val attributeContentValuesMapper: Mapper<AppFlowAttributeInsertionCacheModel, ContentValues>,
    private val parser: Parser<Cursor?, List<AppFlowCacheModel>?>
) : AppFlowCacheHandler {

    companion object {
        private const val appFlowTable = ApmAppFlowEntry.TABLE_NAME
        private const val filterActiveFlows: String =
            "$appFlowTable.${ApmAppFlowEntry.COLUMN_IS_ENDED} = 0"

        private const val filterActiveFlowsWithLaunchId =
            "$appFlowTable.${ApmAppFlowEntry.COLUMN_APP_LAUNCH_ID} = ? AND $filterActiveFlows"

        private const val filterActiveAppFlowsByNameAndAppLaunchId =
            "$appFlowTable.${ApmAppFlowEntry.COLUMN_NAME} = ? " +
                    "AND $filterActiveFlowsWithLaunchId"

        private const val queryMostRecentAppFlowIdWithName =
            "SELECT ${ApmAppFlowEntry.COLUMN_ID} " +
                    "FROM ${ApmAppFlowEntry.TABLE_NAME} " +
                    "WHERE $filterActiveAppFlowsByNameAndAppLaunchId " +
                    "ORDER BY ${ApmAppFlowEntry.COLUMN_ID} DESC " +
                    "LIMIT 1"
    }

    private val databaseWrapper: SQLiteDatabaseWrapper
        get() = databaseManager.openDatabase()

    override fun insert(
        insertionModel: AppFlowInsertionCacheModel
    ): Long = runCatching {
        databaseWrapper.insert(
            ApmAppFlowEntry.TABLE_NAME,
            null,
            appFlowContentValuesMapper.map(insertionModel)
        )
    }.onFailure {
        it.reportAndLog("insert new app flow")
    }.getOrNull() ?: -1L

    override fun end(name: String, timeMicro: Long, appLaunchId: String): Int =
        end(name, appLaunchId, "end app flow with duration") {
            put(ApmAppFlowEntry.COLUMN_END_TIME_MU, timeMicro)
        }

    override fun end(name: String, appLaunchId: String): Int =
        end(name, appLaunchId, "end app flow")

    override fun endActiveFlowsWithEndReason(
        appLaunchId: String,
        @AppFlowEndReason endReason: Int
    ): Int =
        "$filterActiveFlowsWithLaunchId AND ${ApmAppFlowEntry.COLUMN_END_REASON} = ?".runCatching {
            databaseWrapper.update(
                ApmAppFlowEntry.TABLE_NAME,
                ContentValues().apply {
                    put(ApmAppFlowEntry.COLUMN_IS_ENDED, true)
                },
                this,
                arrayOf(appLaunchId, endReason.toString())
            )
        }.onFailure { it.reportAndLog("end abandoned active app flows") }.getOrNull() ?: 0

    override fun endWithReason(
        name: String,
        @AppFlowEndReason reason: Int,
        appLaunchId: String
    ): Int = end(name, appLaunchId, "end app flow with reason") {
        put(ApmAppFlowEntry.COLUMN_END_REASON, reason)
    }

    private inline fun end(
        name: String,
        appLaunchId: String,
        errorLogMessage: String,
        addToContentValues: ContentValues.() -> Unit = {}
    ): Int = runCatching {
        databaseWrapper.update(
            ApmAppFlowEntry.TABLE_NAME,
            ContentValues().apply {
                addToContentValues()
                put(ApmAppFlowEntry.COLUMN_IS_ENDED, true)
            },
            filterActiveAppFlowsByNameAndAppLaunchId,
            arrayOf(name, appLaunchId)
        )
    }.onFailure {
        it.reportAndLog(errorLogMessage)
    }.getOrNull() ?: 0

    override fun addAttribute(name: String, key: String, value: String, appLaunchId: String): Long =
        getMostRecentActiveAppFLowId(name, appLaunchId)?.let { appFlowId ->
            runCatching {
                databaseWrapper.insert(
                    ApmAppFlowTableAttributeEntry.TABLE_NAME,
                    null,
                    attributeContentValuesMapper.map(
                        AppFlowAttributeInsertionCacheModel(appFlowId, key, value)
                    )
                )
            }.onFailure {
                it.reportAndLog("add app flow attribute")
            }.getOrNull()
        } ?: -1

    override fun getAttributeCount(
        name: String,
        appLaunchId: String
    ): Int = runCatching {
        databaseWrapper.rawQuery(
            "SELECT COUNT(*) " +
                    "FROM ${ApmAppFlowTableAttributeEntry.TABLE_NAME} " +
                    "WHERE ${ApmAppFlowTableAttributeEntry.COLUMN_FLOW_ID} IN " +
                    "($queryMostRecentAppFlowIdWithName)",
            arrayOf(name, appLaunchId)
        )?.let { cursor ->
            try {
                cursor.takeIf { it.moveToFirst() }?.getInt(0) ?: 0
            } finally {
                cursor.close()
            }
        }
    }.onFailure {
        it.reportAndLog("getting attribute count")
    }.getOrNull() ?: 0

    override fun removeAttribute(name: String, key: String, appLaunchId: String): Int =
        runCatching {
            databaseWrapper.delete(
                ApmAppFlowTableAttributeEntry.TABLE_NAME,
                "${ApmAppFlowTableAttributeEntry.COLUMN_FLOW_ID} IN " +
                        "($queryMostRecentAppFlowIdWithName) AND " +
                        "${ApmAppFlowTableAttributeEntry.COLUMN_KEY} = ?",
                arrayOf(name, appLaunchId, key)
            )
        }.onFailure {
            it.reportAndLog("remove app flow attribute")
        }.getOrNull() ?: 0

    override fun updateAttributeValue(
        name: String,
        key: String,
        newValue: String,
        appLaunchId: String
    ): Int =
        runCatching {
            val whereStatement = "${ApmAppFlowTableAttributeEntry.COLUMN_FLOW_ID} IN " +
                    "($queryMostRecentAppFlowIdWithName) AND " +
                    "${ApmAppFlowTableAttributeEntry.COLUMN_KEY} = ?"

            val contentValues =
                ContentValues().apply { put(ApmAppFlowTableAttributeEntry.COLUMN_VALUE, newValue) }
            databaseWrapper.update(
                ApmAppFlowTableAttributeEntry.TABLE_NAME,
                contentValues,
                whereStatement,
                arrayOf(name, appLaunchId, key)
            )
        }.onFailure {
            it.reportAndLog("update app flow attribute")
        }.getOrNull() ?: 0

    override fun clear() {
        runCatching {
            databaseWrapper.execSQL(ApmAppFlowEntry.DELETE_ALL)
        }.onFailure {
            it.reportAndLog("clear app flows")
        }
    }

    override fun migrateActiveFlows(newSession: String, appLaunchId: String): Int =
        runCatching {
            databaseWrapper.update(
                ApmAppFlowEntry.TABLE_NAME,
                ContentValues().apply {
                    put(ApmAppFlowEntry.COLUMN_APM_SESSION_ID, newSession)
                },
                filterActiveFlowsWithLaunchId,
                arrayOf(appLaunchId)
            )
        }.onFailure {
            it.reportAndLog("migrate app flows")
        }.getOrNull() ?: 0

    override fun retrieve(sessionId: String): List<AppFlowCacheModel> =
        runCatching {
            val appFlowTable = ApmAppFlowEntry.TABLE_NAME
            val appFLowId = ApmAppFlowEntry.COLUMN_ID
            val appFLowSessionId = ApmAppFlowEntry.COLUMN_APM_SESSION_ID
            val attributesTable = ApmAppFlowTableAttributeEntry.TABLE_NAME
            val attributeFlowId = ApmAppFlowTableAttributeEntry.COLUMN_FLOW_ID
            val attributeKey = ApmAppFlowTableAttributeEntry.COLUMN_KEY
            val attributeValue = ApmAppFlowTableAttributeEntry.COLUMN_VALUE
            val cursor = databaseWrapper.rawQuery(
                "SELECT " +
                        "$appFlowTable.*, " +
                        "$attributesTable.$attributeKey, " +
                        "$attributesTable.$attributeValue " +
                        "FROM $appFlowTable " +
                        "LEFT JOIN $attributesTable " +
                        "ON $attributesTable.$attributeFlowId = $appFlowTable.$appFLowId " +
                        "WHERE $appFlowTable.$appFLowSessionId = ? " +
                        "ORDER BY $appFlowTable.$appFLowId",
                arrayOf(sessionId),
            )
            try {
                parser.parse(cursor)
            } finally {
                cursor?.close()
            }
        }.onFailure {
            it.reportAndLog("retrieve app flows")
        }.getOrNull() ?: emptyList()

    override fun getCount(sessionId: String): Int =
        "SELECT COUNT(*) FROM ${ApmAppFlowEntry.TABLE_NAME} WHERE ${ApmAppFlowEntry.COLUMN_APM_SESSION_ID} = ?"
            .runCatching {
                databaseWrapper.rawQuery(this, arrayOf(sessionId))?.run {
                    try {
                        takeIf { it.moveToFirst() }?.getInt(0) ?: 0
                    } finally {
                        close()
                    }
                }
            }.onFailure {
                it.reportAndLog("get app flows count")
            }.getOrNull() ?: 0

    override fun trimByRequestLimitForSession(sessionId: String, limit: Int): Int =
        runCatching {
            val queryFlowIdsWithinLimitForASession =
                "SELECT ${ApmAppFlowEntry.COLUMN_ID} FROM ${ApmAppFlowEntry.TABLE_NAME} " +
                        "WHERE ${ApmAppFlowEntry.COLUMN_APM_SESSION_ID} = ? " +
                        "ORDER BY ${ApmAppFlowEntry.COLUMN_ID} DESC " +
                        "LIMIT ?"
            databaseWrapper.delete(
                ApmAppFlowEntry.TABLE_NAME,
                "${ApmAppFlowEntry.COLUMN_APM_SESSION_ID} = ? AND " +
                        "${ApmAppFlowEntry.COLUMN_ID} NOT IN ($queryFlowIdsWithinLimitForASession)",
                arrayOf(sessionId, sessionId, limit.toString())
            )
        }.onFailure {
            it.reportAndLog("trim app flows")
        }.getOrNull() ?: 0

    override fun trimByStoreLimitExcludingSession(sessionId: String, limit: Int): Int =
        runCatching {
            val excludedFlowsWhereCondition = "(${ApmAppFlowEntry.COLUMN_APM_SESSION_ID} != ? OR " +
                    "${ApmAppFlowEntry.COLUMN_APM_SESSION_ID} IS NULL) "
            val queryFlowIdsWithinLimitExcludingASession =
                "SELECT ${ApmAppFlowEntry.COLUMN_ID} FROM ${ApmAppFlowEntry.TABLE_NAME} " +
                        "WHERE $excludedFlowsWhereCondition" +
                        "ORDER BY ${ApmAppFlowEntry.COLUMN_ID} DESC " +
                        "LIMIT ?"
            databaseWrapper.delete(
                ApmAppFlowEntry.TABLE_NAME,
                "$excludedFlowsWhereCondition AND " +
                        "${ApmAppFlowEntry.COLUMN_ID} NOT IN ($queryFlowIdsWithinLimitExcludingASession)",
                arrayOf(sessionId, sessionId, limit.toString())
            )
        }.onFailure {
            it.reportAndLog("trim app flows")
        }.getOrNull() ?: 0

    override fun dropDanglingFLowsExcludingAppLaunch(launchId: String): Int =
        runCatching {
            val whereQuery = "${ApmAppFlowEntry.COLUMN_APM_SESSION_ID} IS NULL AND " +
                    "(${ApmAppFlowEntry.COLUMN_APP_LAUNCH_ID} != ? OR " +
                    "${ApmAppFlowEntry.COLUMN_IS_ENDED} == 1)"
            databaseWrapper.delete(
                ApmAppFlowEntry.TABLE_NAME,
                whereQuery,
                arrayOf(launchId)
            )
        }.onFailure {
            it.reportAndLog("drop dangling flows")
        }.getOrNull() ?: 0

    private fun getMostRecentActiveAppFLowId(name: String, appLaunchId: String): Long? =
        runCatching {
            val cursor = databaseWrapper.rawQuery(
                queryMostRecentAppFlowIdWithName,
                arrayOf(name, appLaunchId),
            )
            try {
                cursor?.takeIf { it.moveToFirst() }
                    ?.getLong(cursor.getColumnIndexOrThrow(ApmAppFlowEntry.COLUMN_ID))
            } finally {
                cursor?.close()
            }
        }.onFailure {
            it.reportAndLog("get app flow by name")
        }.getOrNull()

    override fun setActiveFlowsEndReason(
        @AppFlowEndReason reason: Int,
        isBackgroundFlagOverride: Boolean?,
        appLaunchId: String
    ) = runCatching {
        val contentValues = ContentValues()
            .apply { put(ApmAppFlowEntry.COLUMN_END_REASON, reason) }
            .apply {
                isBackgroundFlagOverride?.let {
                    put(
                        ApmAppFlowEntry.COLUMN_IS_BACKGROUND,
                        it
                    )
                }
            }
        databaseWrapper.update(
            ApmAppFlowEntry.TABLE_NAME,
            contentValues,
            filterActiveFlowsWithLaunchId,
            arrayOf(appLaunchId)
        )
    }.onFailure { it.reportAndLog("set active app flows end reason") }
        .getOrNull() ?: 0

    override fun setActiveFlowsBackgroundState(
        isBackground: Boolean,
        appLaunchId: String
    ): Int = runCatching {
        val whereClause = "${ApmAppFlowEntry.COLUMN_IS_BACKGROUND} != ? AND " +
                filterActiveFlowsWithLaunchId
        databaseWrapper.update(
            ApmAppFlowEntry.TABLE_NAME,
            ContentValues().apply { put(ApmAppFlowEntry.COLUMN_IS_BACKGROUND, isBackground) },
            whereClause,
            arrayOf(isBackground.toSqlString(), appLaunchId)
        )
    }.onFailure { it.reportAndLog("set active flows background state") }
        .getOrDefault(0)

    override fun setActiveFlowsCoreSessionId(coreSessionId: String, appLaunchId: String) =
        runCatching {
            val whereClause = "${ApmAppFlowEntry.COLUMN_FIRST_CORE_SESSION_ID} IS NULL AND " +
                    filterActiveFlowsWithLaunchId
            databaseWrapper.update(
                ApmAppFlowEntry.TABLE_NAME,
                ContentValues().apply {
                    put(
                        ApmAppFlowEntry.COLUMN_FIRST_CORE_SESSION_ID,
                        coreSessionId
                    )
                },
                whereClause,
                arrayOf(appLaunchId)
            )
        }.onFailure { it.reportAndLog("set active flows coreSessionId") }
            .getOrDefault(0)

    override fun filterUnReadyCoreSessionIds(
        sessionIds: List<String>,
        launchId: String
    ): List<String> =
        mutableListOf<String>().apply {
            runCatching {
                queryAndFillUnreadySessionIds(
                    filterUnreadySessionsQueryStatement(sessionIds),
                    arrayOf(*sessionIds.toTypedArray(), launchId, SessionVersion.V3)
                )
            }.onFailure { it.reportAndLog("get unready core session ids") }
        }

    private fun MutableList<String>.queryAndFillUnreadySessionIds(
        query: String,
        whereArgs: Array<String>,
    ) = databaseWrapper.rawQuery(query, whereArgs)?.also { cursor ->
        try {
            if (cursor.moveToFirst()) {
                do {
                    add(cursor.getString(0))
                } while (cursor.moveToNext())
            }
        } finally {
            cursor.close()
        }
    }

    private fun filterUnreadySessionsQueryStatement(sessionIds: List<String>): String {
        val apmSessionTable = APMSessionEntry.TABLE_NAME
        val coreSessionId = "$apmSessionTable.${APMSessionEntry.COLUMN_CORE_ID}"
        val apmSessionId = "$apmSessionTable.${APMSessionEntry.COLUMN_ID}"
        val apmCoreSessionVersion = "$apmSessionTable.${APMSessionEntry.COLUMN_CORE_VERSION}"
        val appFlowApmSessionId = "$appFlowTable.${ApmAppFlowEntry.COLUMN_APM_SESSION_ID}"
        return "SELECT DISTINCT $coreSessionId FROM $apmSessionTable " +
                "INNER JOIN $appFlowTable ON $appFlowApmSessionId = $apmSessionId " +
                "WHERE $coreSessionId IN ${sessionIds.joinToArgs()} " +
                "AND $filterActiveFlowsWithLaunchId " +
                "AND $apmCoreSessionVersion = ? "
    }

    private fun Throwable.reportAndLog(message: String? = null) {
        val errorMessage = "APM app flow database error ${message?.let { ": $it" }}"
        logger.logSDKErrorProtected(errorMessage, this)
        IBGDiagnostics.reportNonFatal(this, errorMessage)
    }

    private fun Boolean.toSqlString() = if (this) "1" else "0"
}