package com.instabug.apm.webview.webview_trace.manager

import com.instabug.apm.configuration.APMConfigurationProvider
import com.instabug.apm.constants.ErrorMessages
import com.instabug.apm.di.Provider
import com.instabug.apm.logger.internal.Logger
import com.instabug.apm.webview.dispatch.WebViewEventDispatcher
import com.instabug.apm.webview.dispatch.WebViewEventListener
import com.instabug.apm.webview.webview_trace.configuration.WebViewTraceConfigurationProvider
import com.instabug.apm.webview.webview_trace.handler.WebViewTraceEventListener
import com.instabug.apm.webview.webview_trace.handler.WebViewTraceHandler
import java.util.Collections
import java.util.WeakHashMap
import java.util.concurrent.Executor

interface WebViewTraceManager {

    fun onStateChanged()
    fun setSDKEnabled(isEnabled: Boolean)
    fun registerWebViewEventListener(listener: WebViewTraceEventListener)
    fun unregisterWebViewEventListener(listener: WebViewTraceEventListener)
}

class WebViewTraceManagerImpl(
    private val webViewTraceConfigurations: WebViewTraceConfigurationProvider,
    private val apmConfigurations: APMConfigurationProvider,
    private val handler: Provider<WebViewTraceHandler?>,
    private val logger: Provider<Logger>,
    private val executor: Executor
) : WebViewTraceManager {

    private val listeners: MutableSet<WebViewEventListener> =
        Collections.synchronizedSet(
            Collections.newSetFromMap(WeakHashMap())
        )

    override fun onStateChanged() {
        executor.execute {
            onStateChangeSynchronous()
        }
    }

    private fun onStateChangeSynchronous() {
        if (!webViewTraceConfigurations.enabled) {
            handler()?.clearAll()
            listeners.forEach { WebViewEventDispatcher -= it }
            listeners.clear()
        }
    }

    override fun setSDKEnabled(isEnabled: Boolean) = executor.execute {
        when {
            !apmConfigurations.isAPMFeatureAvailable -> logger().e(ErrorMessages.WEB_VIEW_TRACE_APM_BE_DISABLED)
            !apmConfigurations.isAPMSdkEnabled -> logger().e(ErrorMessages.WEB_VIEW_TRACE_APM_SDK_DISABLED)
            !apmConfigurations.isUiTraceSdkEnabled -> logger().e(ErrorMessages.WEB_VIEW_TRACE_UI_TRACE_SDK_DISABLED)
            !apmConfigurations.isUiLoadingMetricsFeatureEnabled -> logger().e(ErrorMessages.WEB_VIEW_TRACE_FEATURE_BE_DISABLED)
            !apmConfigurations.isUiLoadingMetricsSdkEnabled -> logger().e(ErrorMessages.WEB_VIEW_TRACE_UI_LOADING_SDK_DISABLED)
            !webViewTraceConfigurations.featureEnabled -> logger().e(ErrorMessages.WEB_VIEW_TRACE_FEATURE_BE_DISABLED)
            else -> setWebViewTracesSdkEnabledValue(isEnabled)
        }
    }

    private fun setWebViewTracesSdkEnabledValue(isEnabled: Boolean) {
        webViewTraceConfigurations.sdkEnabled = isEnabled
        onStateChangeSynchronous()
    }

    override fun registerWebViewEventListener(listener: WebViewTraceEventListener) =
        executor.execute {
            if (webViewTraceConfigurations.enabled) {
                WebViewEventDispatcher += listener
                listeners += listener
            }
        }

    override fun unregisterWebViewEventListener(listener: WebViewTraceEventListener) =
        executor.execute {
            WebViewEventDispatcher -= listener
            listeners -= listener
        }
}