package com.instabug.apm.networking.mapping.networklog;

import android.text.TextUtils;

import androidx.annotation.NonNull;
import androidx.annotation.Nullable;

import com.instabug.apm.model.APMNetworkLog;
import com.instabug.library.diagnostics.IBGDiagnostics;

import org.json.JSONArray;
import org.json.JSONException;
import org.json.JSONObject;

import java.util.List;

public class APMNetworkLogMapperImpl implements APMNetworkLogMapper {

    @Override
    public JSONArray toJsonArray(@NonNull List<APMNetworkLog> apmNetworkLogList) throws JSONException {
        JSONArray jsonArray = new JSONArray();
        for (APMNetworkLog log : apmNetworkLogList) {
            JSONObject jsonObject = new JSONObject();
            if (log.getMethod() != null) {
                jsonObject.put(APMNetworkLogMappingKeys.METHOD, log.getMethod().toLowerCase());
            }
            if (log.getUrl() != null) {
                jsonObject.put(APMNetworkLogMappingKeys.URL, log.getUrl());
            }
            if (!TextUtils.isEmpty(log.getRadio())) {
                jsonObject.put(APMNetworkLogMappingKeys.RADIO, log.getRadio());
            }
            if (!TextUtils.isEmpty(log.getCarrier())) {
                jsonObject.put(APMNetworkLogMappingKeys.CARRIER, log.getCarrier());
            }
            int responseCode = log.getResponseCode();
            JSONObject grpcObject = getGrpcObject(log);
            boolean isGRPCRequest = grpcObject != null;
            if (grpcObject != null) {
                jsonObject.put(APMNetworkLogMappingKeys.GRPC_OBJECT, grpcObject);
            }
            JSONObject errorObject = getErrorObject(log);
            if (errorObject != null) {
                jsonObject.put(APMNetworkLogMappingKeys.ERROR_OBJECT, errorObject);
            } else if (responseCode >= 0) {
                if (isGRPCRequest || responseCode > 0) {
                    jsonObject.put(APMNetworkLogMappingKeys.STATUS_CODE, responseCode);
                }
            }
            JSONObject requestObject = getRequestObject(log);
            if (requestObject != null) {
                jsonObject.put(APMNetworkLogMappingKeys.REQUEST, requestObject);
            }
            JSONObject responseObject = getResponseObject(log);
            if (responseObject != null) {
                jsonObject.put(APMNetworkLogMappingKeys.RESPONSE, responseObject);
            }
            if (log.getTotalDuration() > 0) {
                jsonObject.put(APMNetworkLogMappingKeys.RESPONSE_TIME, log.getTotalDuration());
            }

            jsonObject.put(APMNetworkLogMappingKeys.BACKGROUND, log.getExecutedInBackground());
            if (log.getStartTime() != null) {
                jsonObject.put(APMNetworkLogMappingKeys.START_TIME, log.getStartTime());
            }
            if (log.getAttributes() != null) {
                jsonObject.put(APMNetworkLogMappingKeys.ATTRIBUTES, (new JSONObject(log.getAttributes())));
            }
            String graphQlQueryName = log.getGraphQlQueryName();
            if (graphQlQueryName != null) {
                JSONObject graphQlJsonObject = new JSONObject();
                graphQlJsonObject.put(APMNetworkLogMappingKeys.QUERY_NAME, graphQlQueryName);
                jsonObject.put(APMNetworkLogMappingKeys.GRAPH_QL, graphQlJsonObject);
            }
            String serverSideErrorMessage = log.getServerSideErrorMessage();
            if (serverSideErrorMessage != null) {
                JSONObject serverSideErrorJsonObject = new JSONObject();
                serverSideErrorJsonObject.put(APMNetworkLogMappingKeys.SERVER_ERROR_MESSAGE, serverSideErrorMessage);
                jsonObject.put(APMNetworkLogMappingKeys.SERVER_SIDE_ERROR, serverSideErrorJsonObject);
            }
            boolean userModified = log.isModified();
            jsonObject.put(APMNetworkLogMappingKeys.USER_MODIFIED_FLAG, userModified);
            jsonArray.put(jsonObject);
            JSONArray latencySpansObject = getLatencySpansObject(log);
            if (latencySpansObject != null) {
                jsonObject.put(APMNetworkLogMappingKeys.STAGES, latencySpansObject);
            }
            if (log.getExternalTraceId() != null) {
                jsonObject.put(
                        APMNetworkLogMappingKeys.EXTERNAL_TRACE_ID,
                        log.getExternalTraceId()
                );
            }
            if (log.getExternalTraceStartTimestampMillis() != null) {
                jsonObject.put(
                        APMNetworkLogMappingKeys.EXTERNAL_TRACE_START_TIME,
                        log.getExternalTraceStartTimestampMillis()
                );
            }
            if (log.isW3CTraceIdCaptured() != null){
                jsonObject.put(
                        APMNetworkLogMappingKeys.IS_W3C_EXTERNAL_TRACE_ID_CAPTURED,
                        log.isW3CTraceIdCaptured()
                );
            }
            if (log.getGeneratedW3CPid() != null){
                jsonObject.put(
                        APMNetworkLogMappingKeys.W3C_EXTERNAL_TRACE_ID_PID,
                        log.getGeneratedW3CPid()
                );
            }
            if (log.getGeneratedW3CTimestampSeconds() != null){
                jsonObject.put(
                        APMNetworkLogMappingKeys.W3C_EXTERNAL_TRACE_ID_TIMESTAMP,
                        log.getGeneratedW3CTimestampSeconds()
                );
            }
            if (log.getSyncableGeneratedW3CTraceId() != null){
                jsonObject.put(
                        APMNetworkLogMappingKeys.W3C_GENERATED_EXTERNAL_TRACE_ID,
                        log.getSyncableGeneratedW3CTraceId()
                );
            }
            if (log.getSyncableCapturedW3CTraceId() != null){
                jsonObject.put(
                        APMNetworkLogMappingKeys.W3C_CAPTURED_EXTERNAL_TRACE_ID,
                        log.getSyncableCapturedW3CTraceId()
                );
            }
        }
        return jsonArray;
    }

    private String sanitize(@NonNull String target) {
        return target.replaceAll("\\[", "")
                .replaceAll("]", "");
    }

    @Nullable
    private JSONObject getRequestObject(APMNetworkLog log) throws JSONException {
        if (log.getRequestBodySize() > 0
                || log.getRequestContentType() != null
                || log.getRequestHeaders() != null) {
            JSONObject requestObject = new JSONObject();
            if (log.getRequestBodySize() > 0) {
                requestObject.put(APMNetworkLogMappingKeys.PAYLOAD_SIZE, log.getRequestBodySize());
            }
            String headers = log.getRequestHeaders();
            if (headers != null) {
                headers = sanitize(headers);
                requestObject.put(APMNetworkLogMappingKeys.HEADERS, headers);
            }
            String contentType = log.getRequestContentType();
            if (contentType != null) {
                contentType = sanitize(contentType);
                requestObject.put(APMNetworkLogMappingKeys.CONTENT_TYPE, contentType);
            }
            return requestObject;
        }
        return null;
    }

    @Nullable
    private JSONObject getResponseObject(APMNetworkLog log) throws JSONException {
        if (log.getResponseBodySize() > 0
                || log.getResponseContentType() != null
                || log.getResponseHeaders() != null) {
            JSONObject responseObject = new JSONObject();
            if (log.getResponseBodySize() > 0) {
                responseObject.put(APMNetworkLogMappingKeys.PAYLOAD_SIZE, log.getResponseBodySize());
            }
            String headers = log.getResponseHeaders();
            if (headers != null) {
                headers = sanitize(headers);
                responseObject.put(APMNetworkLogMappingKeys.HEADERS, headers);
            }
            String contentType = log.getResponseContentType();
            if (contentType != null) {
                contentType = sanitize(contentType);
                responseObject.put(APMNetworkLogMappingKeys.CONTENT_TYPE, contentType);
            }
            return responseObject;
        }
        return null;
    }

    @Nullable
    private JSONObject getErrorObject(APMNetworkLog log) throws JSONException {
        String errorMessage = log.getErrorMessage();
        int errorCode = log.getClientErrorCode();
        if (errorMessage != null || errorCode > 0) {
            JSONObject errorObject = new JSONObject();
            if (errorMessage != null) {
                errorObject.put(APMNetworkLogMappingKeys.EXCEPTION, errorMessage);
            }
            if (errorCode > 0) {
                errorObject.put(APMNetworkLogMappingKeys.ERROR_CODE, errorCode);
            }
            return errorObject;
        }
        return null;
    }

    @Nullable
    private JSONObject getGrpcObject(APMNetworkLog log) throws JSONException {
        String grpcMethodName = log.getGrpcMethodName();
        if (grpcMethodName != null) {
            JSONObject grpcJsonObject = new JSONObject();
            grpcJsonObject.put(APMNetworkLogMappingKeys.GRPC_METHOD, grpcMethodName);
            return grpcJsonObject;
        }
        return null;
    }

    @Nullable
    private JSONArray getLatencySpansObject(APMNetworkLog log) {
        String spans = log.getLatencySpansJsonString();
        if (spans != null) {
            try {
                return new JSONArray(spans);
            } catch (JSONException jsonException) {
                IBGDiagnostics.reportNonFatal(jsonException, "Exception while parsing NetworkLogs latency spans");
            }
        }
        return null;
    }
}
