package com.instabug.apm.lifecycle

import com.instabug.apm.cache.model.AppLaunchCacheModel
import com.instabug.apm.constants.AppLaunchType
import com.instabug.apm.constants.Constants.AppLaunch
import com.instabug.apm.model.AppLaunchStage
import com.instabug.apm.model.AppLaunchStageDetails

interface AppLaunchModelFactory {
    fun createAppLaunchModelIfPossible(
        screenName: String,
        @AppLaunchType type: String,
        appLaunchDataRepository: AppLaunchDataRepository
    ): AppLaunchCacheModel?
}

class AppLaunchModelFactoryImpl : AppLaunchModelFactory {
    override fun createAppLaunchModelIfPossible(
        screenName: String,
        @AppLaunchType type: String,
        appLaunchDataRepository: AppLaunchDataRepository
    ): AppLaunchCacheModel? {
        val stagesMap = appLaunchDataRepository.appLaunchStages.toMap()
        if (stagesMap[AppLaunchStage.ACTIVITY_START] != null) {
            val appLaunchModel = AppLaunchCacheModel()
            appLaunchModel.screenName = screenName
            appLaunchModel.type = type
            when (type) {
                AppLaunchType.COLD -> updateColdAppLaunchModel(appLaunchModel, stagesMap, appLaunchDataRepository)
                AppLaunchType.WARM -> updateWarmAppLaunchModel(appLaunchModel, stagesMap, appLaunchDataRepository)
                AppLaunchType.HOT -> updateHotAppLaunchModel(appLaunchModel, stagesMap, appLaunchDataRepository)
            }
            return appLaunchModel
        }
        return null
    }

    private fun updateColdAppLaunchModel(
        model: AppLaunchCacheModel,
        stages: Map<AppLaunchStage, AppLaunchStageDetails>,
        appLaunchDataRepository: AppLaunchDataRepository
    ) {
        updateAppLaunchStartTimeAndDuration(
            model,
            startStage = stages[AppLaunchStage.APP_CREATION],
            endStage = stages[AppLaunchStage.ACTIVITY_START]
        )
        model.stages = getColdAppLaunchStages(stages, appLaunchDataRepository)
    }

    private fun updateWarmAppLaunchModel(
        model: AppLaunchCacheModel,
        stages: Map<AppLaunchStage, AppLaunchStageDetails>,
        appLaunchDataRepository: AppLaunchDataRepository
    ) {
        updateAppLaunchStartTimeAndDuration(
            model,
            startStage = stages[AppLaunchStage.ACTIVITY_CREATION],
            endStage = stages[AppLaunchStage.ACTIVITY_START]
        )
        model.stages = getWarmAppLaunchStages(stages, appLaunchDataRepository)
    }

    private fun updateHotAppLaunchModel(
        model: AppLaunchCacheModel,
        stages: Map<AppLaunchStage, AppLaunchStageDetails>,
        appLaunchDataRepository: AppLaunchDataRepository
    ) {
        updateAppLaunchStartTimeAndDuration(
            model,
            startStage = stages[AppLaunchStage.ACTIVITY_START],
            endStage = stages[AppLaunchStage.ACTIVITY_START]
        )
        model.stages = getHotAppLaunchStages(stages, appLaunchDataRepository)
    }

    private fun getHotAppLaunchStages(
        stages: Map<AppLaunchStage, AppLaunchStageDetails>,
        appLaunchDataRepository: AppLaunchDataRepository
    ): MutableMap<String, String> =
        HashMap<String, String>().apply {
            val activityStartStage = stages[AppLaunchStage.ACTIVITY_START]
            put(
                AppLaunch.ACTIVITY_ON_START_TIME_STAMP,
                (activityStartStage?.getStartTimeStampMicro() ?: 0).toString()
            )
            put(
                AppLaunch.ACTIVITY_ON_START_DURATION,
                (activityStartStage?.getDurationMicro() ?: 0).toString()
            )
            if (appLaunchDataRepository.isEndAppLaunchCalledEarly) {
                put(AppLaunch.END_APP_LAUNCH_DURATION, 0.toString())
            }
        }

    private fun getWarmAppLaunchStages(
        stages: Map<AppLaunchStage, AppLaunchStageDetails>,
        appLaunchDataRepository: AppLaunchDataRepository
    ): MutableMap<String, String> =
        getHotAppLaunchStages(stages, appLaunchDataRepository).apply {
            val activityCreateStage = stages[AppLaunchStage.ACTIVITY_CREATION]
            put(
                AppLaunch.ACTIVITY_ON_CREATE_TIME_STAMP,
                (activityCreateStage?.getStartTimeStampMicro() ?: 0).toString()
            )
            put(
                AppLaunch.ACTIVITY_ON_CREATE_DURATION,
                (activityCreateStage?.getDurationMicro() ?: 0).toString()
            )
        }

    private fun getColdAppLaunchStages(
        stages: Map<AppLaunchStage, AppLaunchStageDetails>,
        appLaunchDataRepository: AppLaunchDataRepository
    ): MutableMap<String, String> =
        getWarmAppLaunchStages(stages, appLaunchDataRepository).apply {
            val appCreateStage = stages[AppLaunchStage.APP_CREATION]
            put(
                AppLaunch.APP_ON_CREATE_START_TIMESTAMP,
                (appCreateStage?.getStartTimeStampMicro() ?: 0).toString()
            )
            put(
                AppLaunch.APP_ON_CREATE_DURATION,
                (appCreateStage?.getDurationMicro() ?: 0).toString()
            )
        }

    private fun updateAppLaunchStartTimeAndDuration(
        appLaunchModel: AppLaunchCacheModel,
        startStage: AppLaunchStageDetails?,
        endStage: AppLaunchStageDetails?
    ) {
        if (startStage != null && endStage != null) {
            appLaunchModel.startTime = startStage.getStartTimeStampMicro()
            appLaunchModel.duration = endStage.stageEndTimeMicro - startStage.stageStartTimeMicro
        }
    }
}