package com.instabug.apm.handler.uitrace.customuitraces;

import static com.instabug.apm.constants.ErrorMessages.UI_TRACE_STARTED_TWICE;
import static com.instabug.apm.util.ClassModuleHelper.getModuleName;

import android.annotation.SuppressLint;
import android.app.Activity;
import android.os.Build;
import android.os.Looper;
import android.view.Choreographer;

import androidx.annotation.NonNull;
import androidx.annotation.Nullable;
import androidx.annotation.RequiresApi;
import androidx.annotation.WorkerThread;

import com.instabug.apm.cache.handler.session.SessionMetaDataCacheHandler;
import com.instabug.apm.cache.handler.uitrace.UiTraceCacheHandler;
import com.instabug.apm.cache.model.UiTraceCacheModel;
import com.instabug.apm.configuration.APMConfigurationProvider;
import com.instabug.apm.di.ServiceLocator;
import com.instabug.apm.logger.internal.Logger;
import com.instabug.apm.model.EventTimeMetricCapture;
import com.instabug.apm.uitrace.activitycallbacks.APMUiTraceActivityCallbacks;
import com.instabug.apm.uitrace.di.UiTracesServiceLocator;
import com.instabug.apm.uitrace.uihangs.APMChoreographer;
import com.instabug.apm.uitrace.uihangs.FrameDropsCalculator;
import com.instabug.apm.util.device.APMDeviceStateProvider;
import com.instabug.apm.util.powermanagement.BatteryLevelChangeBroadcast;
import com.instabug.apm.util.powermanagement.PowerManagementCallback;
import com.instabug.apm.util.powermanagement.PowerSaveModeBroadcast;
import com.instabug.apm.util.view.InstabugViews;
import com.instabug.library.BuildFieldsProvider;
import com.instabug.library.diagnostics.IBGDiagnostics;
import com.instabug.library.model.common.Session;
import com.instabug.library.tracking.InstabugInternalTrackingDelegate;
import com.instabug.library.util.TimeUtils;

import java.util.concurrent.Executor;
import java.util.concurrent.TimeUnit;

/**
 * Created by Barakat on 25/07/2020
 */
@RequiresApi(api = Build.VERSION_CODES.JELLY_BEAN)
public class CustomUiTraceHandlerImpl implements CustomUiTraceHandler,
        Choreographer.FrameCallback,
        PowerManagementCallback, APMUiTraceActivityCallbacks {

    private final APMDeviceStateProvider deviceStateProvider;
    private final APMConfigurationProvider apmConfigurationProvider;
    private final Logger apmLogger;
    @Nullable
    private final APMChoreographer apmChoreographer;
    private final Executor customUiTraceThreadExecutor;

    @Nullable
    private UiTraceCacheModel cacheModel;
    UiTraceCacheHandler cacheHandler;
    @Nullable
    SessionMetaDataCacheHandler sessionMetaDataCacheHandler;
    @NonNull
    private final BatteryLevelChangeBroadcast batteryBroadcastReceiver;
    @NonNull
    private final PowerSaveModeBroadcast powerSaveModeBroadcast;
    private float uiTraceLargeDropThreshold = Float.MAX_VALUE;
    private float uiTraceSmallDropThreshold = Float.MAX_VALUE;
    private final FrameDropsCalculator frameDropsCalculator;

    public CustomUiTraceHandlerImpl(
            @NonNull BatteryLevelChangeBroadcast batteryBroadcastReceiver,
            @NonNull PowerSaveModeBroadcast powerSaveModeBroadcast,
            APMDeviceStateProvider deviceStateProvider,
            APMConfigurationProvider apmConfigurationProvider,
            Logger apmLogger,
            @NonNull FrameDropsCalculator frameDropsCalculator
    ) {
        this.deviceStateProvider = deviceStateProvider;
        this.apmConfigurationProvider = apmConfigurationProvider;
        this.apmLogger = apmLogger;
        this.batteryBroadcastReceiver = batteryBroadcastReceiver;
        this.powerSaveModeBroadcast = powerSaveModeBroadcast;

        apmChoreographer = UiTracesServiceLocator.INSTANCE.getApmChoreographer();
        this.frameDropsCalculator = frameDropsCalculator;
        this.cacheHandler = ServiceLocator.getUiTraceCacheHandler();
        this.sessionMetaDataCacheHandler = ServiceLocator.getSessionMetaDataCacheHandler();
        this.customUiTraceThreadExecutor = ServiceLocator.getSingleThreadExecutor("CustomUiTraceHandler");
    }

    @Override
    public void startUiTrace(@NonNull String name, Activity activity, @Nullable Looper callerThreadLooper) {
        customUiTraceThreadExecutor.execute(() -> {
            uiTraceLargeDropThreshold = apmConfigurationProvider.getUiTraceLargeDropThreshold();
            uiTraceSmallDropThreshold = apmConfigurationProvider.getUiTraceSmallDropThreshold();
            if (cacheModel != null) {
                apmLogger.logSDKProtected("Existing Ui trace " + getCurrentUiTrace() + " need to be ended first");
                if (getCurrentUiTrace() != null) {
                    apmLogger.logSDKWarning(UI_TRACE_STARTED_TWICE.replace("$s1", name).replace("$s2", getCurrentUiTrace()));
                }
                endUiTraceOnSameThread(activity, callerThreadLooper);
            }
            Session currentSession = ServiceLocator.getSessionHandler().getCurrentSession();
            if (currentSession == null) {
                return;
            }
            createCacheModel(name, activity, currentSession);
            registerBatteryLevelListener();
            registerPowerSaveModeListener();
            if (apmChoreographer != null) {
                frameDropsCalculator.reset();
                apmChoreographer.addCallback(this);
            }
            apmLogger.logSDKDebug("Custom UI Trace  \"" + name + "\" has started.");
        });
    }

    @WorkerThread
    private void endUiTraceOnSameThread(final Activity activity, @Nullable Looper callerThreadLooper) {
        apmLogger.logSDKProtected("Ui trace " + (cacheModel != null ? cacheModel.getName() : "") + " is ending in " + activity.getClass().getSimpleName());
        if (apmChoreographer != null) {
            apmChoreographer.removeCallback(this);
        }

        try {
            unregisterPowerSaveModeListener();
            unregisterBatteryLevelListener();

            updateCacheModel(activity);
            if (cacheModel == null) {
                apmLogger.logSDKProtected("uiTraceModel is null, can't insert to DB");
                return;
            }
            String sessionId = cacheModel.getSessionId();
            if (sessionId != null) {
                long id = cacheHandler.insert(cacheModel);
                if (id != -1) {
                    if (sessionMetaDataCacheHandler != null) {
                        sessionMetaDataCacheHandler.addToUITracesTotalCount(sessionId, 1);
                        int deletedTracesCount = cacheHandler.trimToLimit(sessionId,
                                apmConfigurationProvider.getUiTraceLimitPerRequest());
                        if (deletedTracesCount > 0) {
                            sessionMetaDataCacheHandler.addToUITracesDroppedCount(sessionId, deletedTracesCount);
                        }
                    }
                    cacheHandler.clearPreviousUnEndedTraces(sessionId);
                    cacheHandler.trimToLimit(apmConfigurationProvider.getUiTraceStoreLimit());
                }
                apmLogger.logSDKDebug("Custom UI Trace \"" + cacheModel.getName() + "\" has ended.\n"
                        + "Total duration: " + getTotalDurationSeconds(cacheModel) + " seconds\n"
                        + "Total hang duration: " + getTotalHangsDurationMillis(cacheModel) + " ms");
            } else {
                apmLogger.logSDKProtected("currentSession is null, can't insert to DB");
            }
            cacheModel = null;
        } catch (Exception e) {
            IBGDiagnostics.reportNonFatal(e, "Unable to end ui trace");
        }
    }

    @Override
    public void endUiTrace(final Activity activity, @Nullable Looper callerThreadLooper) {
        customUiTraceThreadExecutor.execute(() -> {
            endUiTraceOnSameThread(activity, callerThreadLooper);
        });

    }

    @Nullable
    @Override
    public String getCurrentUiTrace() {
        return cacheModel != null ? cacheModel.getName() : null;
    }

    @Override
    public void forceStop() {
        if (InstabugInternalTrackingDelegate.getInstance().getCurrentActivity() != null) {
            endUiTrace(InstabugInternalTrackingDelegate.getInstance().getCurrentActivity(), Looper.myLooper());
        }

    }

    @Override
    public void onActivityStarted(@NonNull Activity activity, @NonNull EventTimeMetricCapture timeMetric) {
        if (cacheModel != null && !isInstabugActivity(activity)) {
            apmLogger.logSDKProtected(String.format("New activity resumed while ui Trace %s is running, registering broadcast receivers…", getCurrentUiTrace()));
            customUiTraceThreadExecutor.execute(() -> {
                registerPowerSaveModeListener();
                registerBatteryLevelListener();
            });
        }
    }

    @Override
    public void onActivityStopped(@NonNull Activity activity, boolean isAppInBackground) {
        if (isInstabugActivity(activity)) return;
        if (cacheModel != null && isAppInBackground) {
            apmLogger.logSDKProtected(String.format("App went background while ui Trace %s is running, ending the trace…", getCurrentUiTrace()));
            endUiTrace(activity, Looper.myLooper());
        } else {
            customUiTraceThreadExecutor.execute(() -> {
                unregisterPowerSaveModeListener();
                unregisterBatteryLevelListener();
            });

        }
    }

    private boolean isInstabugActivity(@NonNull Activity activity) {
        return InstabugViews.isInstabugActivity(activity);
    }

    @Override
    public void doFrame(long frameTimeNanos) {
        Long frameDuration = frameDropsCalculator.calculateFrameDuration(frameTimeNanos, uiTraceSmallDropThreshold);
        if (frameDuration!=null)onFrameDrop(frameDuration);
    }

    public void onFrameDrop(long frameDurationMus) {
        if (cacheModel != null) {
            cacheModel.setSmallDropsDuration(cacheModel.getSmallDropsDuration() + frameDurationMus);
            if (frameDurationMus > uiTraceLargeDropThreshold) {
                cacheModel.setLargeDropsDuration(cacheModel.getLargeDropsDuration() + frameDurationMus);
            }
        }
    }

    @Override
    public void onBatteryLevelChanged(int level) {
        if (cacheModel != null) {
            if (cacheModel.getBatteryLevel() == -1) {
                cacheModel.setBatteryLevel(level);
            } else {
                cacheModel.setBatteryLevel(Math.min(level, cacheModel.getBatteryLevel()));
            }
        }
    }

    @Override
    public void onPowerSaveModeChanged(boolean enabled) {
        if (enabled && cacheModel != null) {
            cacheModel.setPowerSaveMode(enabled);
        }
    }

    private void createCacheModel(@NonNull String name, Activity activity, @NonNull Session currentSession) {
        cacheModel = new UiTraceCacheModel();
        cacheModel.setSessionId(currentSession.getId());
        cacheModel.setName(name);
        cacheModel.setStartTimestamp(TimeUnit.MILLISECONDS.toMicros(System.currentTimeMillis()));
        cacheModel.setStartTimeInMicros(TimeUtils.microTime());
        cacheModel.setBatteryLevel(deviceStateProvider.getBatteryLevel(activity));
        cacheModel.setPowerSaveMode(deviceStateProvider.isPowerSaveModeEnabled(activity));
        cacheModel.setOrientation(deviceStateProvider.getScreenOrientation(activity));
        cacheModel.setUserDefined(true);
        cacheModel.setId(System.currentTimeMillis());
    }

    private void updateCacheModel(Activity activity) {
        if (cacheModel != null) {
            cacheModel.setDuration(TimeUtils.microTime() - cacheModel.getStartTimeInMicros());
            if (activity != null) {
                cacheModel.setContainerName(activity.getClass().getSimpleName());
                if (activity.getTitle() != null) {
                    cacheModel.setScreenTitle(activity.getTitle().toString());
                }
                cacheModel.setModuleName(getModuleName(activity.getClass()));
            }
            cacheModel.setRefreshRate(deviceStateProvider.getRefreshRate(activity));
        }
    }

    /**
     * Observe battery level changes to capture the lowest value occurs during a ui trace
     * <p>
     * {@link UiTraceCacheModel} should hold the value of the lowest battery level occurs
     * during a ui trace, not the level at the ui trace start nor the ui trace end.
     * </p>
     */
    private void registerBatteryLevelListener() {
        batteryBroadcastReceiver.register(this);
    }


    /**
     * Observe if power save mode was enabled for once during ui trace
     * <p>
     * {@link UiTraceCacheModel} should hold a {@code true} if the power save mode
     * was enabled during a ui trace, not the value of the ui trace start nor the ui trace end.
     * </p>
     */
    @SuppressLint("NewApi")
    private void registerPowerSaveModeListener() {
        if (BuildFieldsProvider.INSTANCE.provideBuildVersion() >= Build.VERSION_CODES.LOLLIPOP) {
            powerSaveModeBroadcast.register(this);
        }
    }

    private void unregisterBatteryLevelListener() {
        batteryBroadcastReceiver.unregister(this);
    }

    @SuppressLint("NewApi")
    private void unregisterPowerSaveModeListener() {
        if (BuildFieldsProvider.INSTANCE.provideBuildVersion() >= Build.VERSION_CODES.LOLLIPOP) {
            powerSaveModeBroadcast.unregister(this);
        }
    }

    public long getTotalDurationSeconds(@NonNull UiTraceCacheModel model) {
        return TimeUnit.MICROSECONDS.toSeconds(model.getDuration());
    }

    public long getTotalHangsDurationMillis(@NonNull UiTraceCacheModel model) {
        return TimeUnit.MICROSECONDS.toMillis(model.getSmallDropsDuration() + model.getLargeDropsDuration());
    }
}
