package com.instabug.apm.handler.networklog;

import static com.instabug.apm.constants.Constants.NETWORK_LOG_CLEAN_EXECUTOR;

import androidx.annotation.NonNull;
import androidx.annotation.Nullable;
import androidx.annotation.WorkerThread;

import com.instabug.apm.cache.handler.networklog.DanglingNetworkLogCacheHandler;
import com.instabug.apm.cache.handler.networklog.NetworkLogCacheHandler;
import com.instabug.apm.cache.handler.session.SessionMetaDataCacheHandler;
import com.instabug.apm.configuration.APMConfigurationProvider;
import com.instabug.apm.constants.DefaultValues;
import com.instabug.apm.constants.ErrorMessages;
import com.instabug.apm.di.ServiceLocator;
import com.instabug.apm.logger.internal.Logger;
import com.instabug.apm.model.APMNetworkLog;
import com.instabug.library.Instabug;

import java.util.List;
import java.util.Map;
import java.util.concurrent.Executor;

@WorkerThread
public class NetworkLogHandlerImpl implements NetworkLogHandler {

    private final NetworkLogCacheHandler networkLogCacheHandler = ServiceLocator.getNetworkLogCacheHandler();
    private final DanglingNetworkLogCacheHandler danglingNetworkLogCacheHandler = ServiceLocator.getDanglingNetworkLogCacheHandler();
    private final Logger apmLogger = ServiceLocator.getApmLogger();
    @Nullable
    private final SessionMetaDataCacheHandler sessionMetaDataCacheHandler = ServiceLocator.getSessionMetaDataCacheHandler();


    @Override
    public long insertNetworkLog(@NonNull APMNetworkLog networkLog) {
        String sessionId = networkLog.getSessionId();
        long rowId = -1;
        if (Instabug.isBuilt()) {
            APMConfigurationProvider apmConfigurationProvider = ServiceLocator.getApmConfigurationProvider();
            if (apmConfigurationProvider.isNetworkEnabled()) {
                if (sessionId == null) {
                    rowId = danglingNetworkLogCacheHandler.insertNetworkLog(networkLog);
                    if (rowId != -1) {
                        apmLogger.d("Network request added to dangling table: " + networkLog.getUrl());
                        danglingNetworkLogCacheHandler.trimToLimit(apmConfigurationProvider.getNetworkLogsCacheLimit());
                    }
                } else {
                    rowId = networkLogCacheHandler.insertNetworkLog(sessionId, networkLog);
                    if (rowId != -1) {
                        apmLogger.d("Network request added to network table: " + networkLog.getUrl());
                        if (sessionMetaDataCacheHandler != null) {
                            sessionMetaDataCacheHandler.addToNetworkLogsTotalCount(sessionId, 1);
                            int deletedNetworkLogsCount =
                                    networkLogCacheHandler.trimToLimit(sessionId,
                                            apmConfigurationProvider.getNetworkLogsRequestLimit());
                            if (deletedNetworkLogsCount > 0) {
                                apmLogger.d("Network requests dropped count: " + deletedNetworkLogsCount);
                                sessionMetaDataCacheHandler.addToNetworkLogsDroppedCount(sessionId,
                                        deletedNetworkLogsCount);
                            }
                        }
                        networkLogCacheHandler.trimToLimit(apmConfigurationProvider.getNetworkLogsCacheLimit());
                    }
                }
                apmLogger.d("inserted network log, returning: " + rowId);
            } else {
                apmLogger.d("network log is not inserted apm network logs is disabled");
            }
        }
        return rowId;
    }

    @Override
    public void updateNetworkLog(APMNetworkLog networkLog) {
        if (Instabug.isBuilt()) {
            APMConfigurationProvider apmConfigurationProvider = ServiceLocator.getApmConfigurationProvider();
            if (apmConfigurationProvider.isNetworkEnabled()) {
                if (networkLog.getExecutedInBackground()) {
                    danglingNetworkLogCacheHandler.updateNetworkLog(networkLog);
                } else {
                    networkLogCacheHandler.updateNetworkLog(networkLog);
                }
            }
        }
    }

    @Override
    public void cleanUp() {
        Executor executor = ServiceLocator.getSingleThreadPoolExecutor(NETWORK_LOG_CLEAN_EXECUTOR);
        executor.execute(() -> {
            networkLogCacheHandler.cleanUp();
            danglingNetworkLogCacheHandler.cleanUp();
        });
    }

    @Override
    public void removeAll() {
        apmLogger.logSDKDebug("Clearing cached APM network logs");
        networkLogCacheHandler.removeAll();
        danglingNetworkLogCacheHandler.removeAll();
        if (sessionMetaDataCacheHandler != null) {
            sessionMetaDataCacheHandler.resetNetworkLogsCounts();
        }
    }

    @Override
    public void removeGraphQlData() {
        networkLogCacheHandler.removeGraphQlData();
        danglingNetworkLogCacheHandler.removeGraphQlData();
    }

    @Override
    public void removeGrpcData() {
        networkLogCacheHandler.removeGrpcData();
        danglingNetworkLogCacheHandler.removeGrpcData();
    }

    @Nullable
    @Override
    public List<APMNetworkLog> getEndedNetworkLogsForSession(String sessionID) {
        return networkLogCacheHandler.getEndedNetworkLogsForSession(sessionID);
    }

    @Override
    public void forceStop() {
        cleanUp();
    }

    @Override
    public boolean isValidAttribute(String traceName, String key, String value) {
        if (key == null || key.trim().isEmpty()) {
            apmLogger.logSDKError(ErrorMessages.ATTRIBUTE_KEY_NULL_OR_EMPTY.replace("$s", traceName));
            return false;
        }
        final String trimmedKey = key.trim();
        if (trimmedKey.length() > DefaultValues.Network.ATTRIBUTE_KEY_LENGTH) {
            apmLogger.logSDKError(ErrorMessages.ATTRIBUTE_KEY_INVALID_LENGTH
                    .replace("$s1", key)
                    .replace("$s2", traceName));
            return false;
        }
        if (value != null) {
            String trimmedValue = value.trim();
            if (trimmedValue.length() == 0) {
                apmLogger.logSDKError(ErrorMessages.ATTRIBUTE_VALUE_EMPTY
                        .replace("$s1", trimmedKey)
                        .replace("$s2", traceName));
                return false;
            }
            if (trimmedValue.length() > DefaultValues.Network.ATTRIBUTE_VALUE_LENGTH) {
                apmLogger.logSDKError(ErrorMessages.ATTRIBUTE_VALUE_INVALID_LENGTH
                        .replace("$s1", key)
                        .replace("$s2", traceName));
                return false;
            }
        }
        return true;
    }

    @Override
    public void addAttribute(long id, String name, boolean executedInBackground, String key, String value) {
        if (Instabug.isBuilt()) {
            APMConfigurationProvider apmConfigurationProvider = ServiceLocator.getApmConfigurationProvider();
            if (apmConfigurationProvider.isNetworkEnabled()) {
                if (executedInBackground) {
                    danglingNetworkLogCacheHandler.addAttribute(id, name, key, value);
                } else {
                    networkLogCacheHandler.addAttribute(id, name, key, value);
                }
            }
        }
    }

    @Override
    public Map<String, String> getTraceAttributes(long traceId) {
        return networkLogCacheHandler.getTraceAttributes(traceId);
    }

    @Override
    public void clearW3CExternalTraceIdCache() {
        networkLogCacheHandler.clearW3CExternalTraceIdCache();
        danglingNetworkLogCacheHandler.clearW3CExternalTraceIdCache();
    }

    @Override
    public void clearGeneratedW3CExternalTraceIdCache() {
        networkLogCacheHandler.clearGeneratedW3CExternalTraceIdCache();
        danglingNetworkLogCacheHandler.clearGeneratedW3CExternalTraceIdCache();
    }

    @Override
    public void clearCapturedW3CExternalTraceIdCache() {
        networkLogCacheHandler.clearCapturedW3CExternalTraceIdCache();
        danglingNetworkLogCacheHandler.clearCapturedW3CExternalTraceIdCache();
    }

    @Override
    public void clearNetworkSpansData() {
        networkLogCacheHandler.clearNetworkSpansData();
        danglingNetworkLogCacheHandler.clearNetworkSpansData();
    }
}
