package com.instabug.apm.handler.applaunch;

import androidx.annotation.NonNull;
import androidx.annotation.Nullable;
import androidx.annotation.WorkerThread;

import com.instabug.apm.cache.handler.applaunch.AppLaunchCacheHandler;
import com.instabug.apm.cache.handler.session.SessionMetaDataCacheHandler;
import com.instabug.apm.cache.model.AppLaunchCacheModel;
import com.instabug.apm.configuration.APMConfigurationProvider;
import com.instabug.apm.constants.AppLaunchType;
import com.instabug.apm.di.ServiceLocator;

import java.util.List;


@WorkerThread
public class AppLaunchesHandlerImpl implements AppLaunchesHandler {
    final private AppLaunchCacheHandler appLaunchCacheHandler = ServiceLocator.getAppLaunchCacheHandler();
    @Nullable
    final private SessionMetaDataCacheHandler sessionMetaDataCacheHandler = ServiceLocator.getSessionMetaDataCacheHandler();
    final private APMConfigurationProvider configurationProvider = ServiceLocator.getApmConfigurationProvider();

    @Override
    public long insertAppLaunch(String sessionID, AppLaunchCacheModel appLaunchCacheModel) {
        long id = appLaunchCacheHandler.insertAppLaunch(sessionID, appLaunchCacheModel);
        if (sessionMetaDataCacheHandler != null && id != -1)
            sessionMetaDataCacheHandler.addToAppLaunchesTotalCount(sessionID, 1);
        trimAppLaunch(sessionID, appLaunchCacheModel, id);
        return id;
    }

    private void trimAppLaunch(String sessionID, AppLaunchCacheModel appLaunchCacheModel, long id) {
        String appLaunchType = appLaunchCacheModel.getType();
        if (id != -1 && appLaunchType != null) {
            trimLaunchToLimitPerRequest(sessionID, appLaunchType);
            trimAppLaunchesToLimit(
                    configurationProvider.getAppLaunchStoreLimit(appLaunchType),
                    appLaunchType
            );
        }
    }

    private void trimLaunchToLimitPerRequest(String sessionID, String appLaunchType) {
        if (sessionMetaDataCacheHandler == null) return;
        int deletedLaunchesCount = trimAppLaunchesToLimit(
                sessionID,
                configurationProvider.getAppLaunchRequestLimit(appLaunchType),
                appLaunchType
        );
        if (deletedLaunchesCount > 0) {
            sessionMetaDataCacheHandler.addToAppLaunchesDroppedCount(sessionID, deletedLaunchesCount);
        }
    }

    private int trimAppLaunchesToLimit(String sessionID, long limit, @AppLaunchType String appLaunchType) {
        return appLaunchCacheHandler.trimAppLaunchesToLimit(sessionID, limit, appLaunchType);
    }

    /**
     * Gets list of cached app launches for passed session id
     *
     * @param sessionID id of the session to get cached app launches for
     * @return List of {@link AppLaunchCacheModel}
     */
    @Nullable
    @Override
    public List<AppLaunchCacheModel> getAppLaunchesForSession(String sessionID) {
        return appLaunchCacheHandler.getAppLaunchesForSession(sessionID);
    }

    private void trimAppLaunchesToLimit(long limit, @AppLaunchType String appLaunchType) {
        appLaunchCacheHandler.trimAppLaunchesToLimit(limit, appLaunchType);
    }

    @Override
    public void removeAppLaunches(@NonNull @AppLaunchType String type, boolean clearCounts) {
        appLaunchCacheHandler.removeAppLaunches(type);
        if (clearCounts) {
            clearCounts();
        }

    }

    @Override
    public void removeAll() {
        appLaunchCacheHandler.removeAll();
        clearCounts();
    }

    private void clearCounts() {
        if (sessionMetaDataCacheHandler != null) {
            sessionMetaDataCacheHandler.resetAppLaunchesCounts();
        }
    }

    @Override
    public void updateAppLaunch(AppLaunchCacheModel currentAppLaunch) {
        appLaunchCacheHandler.updateAppLaunch(currentAppLaunch);
    }
}
