package com.instabug.apm.cache.handler.uitrace;


import androidx.annotation.Nullable;

import com.instabug.apm.cache.model.UiTraceCacheModel;

import java.util.List;

public interface UiTraceCacheHandler {
    /**
     * Inserts ui frame drops  in cache
     *
     * @param uiTrace ui frame drops  to be cached
     * @return id of cached ui frame drops
     */
    long insert(UiTraceCacheModel uiTrace);

    /**
     * Update uiTrace in the database
     *
     * @param uiTrace to be updated
     * @return number of updated rows
     */
    int update(@Nullable UiTraceCacheModel uiTrace);

    /**
     * Insert ui loading model to database
     *
     * @param uiTrace containing UiLoadingModel
     * @return Id of the inserted uiLoading entry
     */
    long insertUiLoadingModel(UiTraceCacheModel uiTrace);

    /**
     * Removes invalid ui frame drops
     * Invalid ui frame drops
     */
    void cleanUp();

    /**
     * Keeps only last x number of records in cache and deletes other records
     *
     * @param limit number of logs to keep
     */
    void trimToLimit(long limit);

    /**
     * Removes oldest ui traces till
     * total ui traces count for the passed
     * sessionID reaches the limit passed
     *
     * @param sessionID Session id to limit its ui traces
     * @param limit     number of ui traces to be remained
     */
    int trimToLimit(String sessionID, long limit);

    /**
     * remove not ended ui traces at sessions excluding ui traces related to the session with the passed id
     *
     * @param sessionID for ui traces excluded from removal
     */
    int clearPreviousUnEndedTraces(String sessionID);

    /**
     * Gets list of cached ui frame drops by limit
     *
     * @param limit max count of ui frame drops s to be retrieved
     * @return list of cached ui frame drops. Empty list in case of no cached logs
     */
    @Nullable
    List<UiTraceCacheModel> getByLimit(int limit);

    /**
     * Removes cached ui frame drops s by limit
     *
     * @param limit max count of ui frame drops s to be removed
     */
    void deleteByLimit(int limit);

    /**
     * Removes all ui trace records
     */
    void removeAll();

    /**
     * Remove ui hangs data
     */
    void removeUiHangs();

    /**
     * Remove ui loading metrics
     */
    void removeUiLoadingMetrics();


    /**
     * Gets list of cached ui traces for passed session id
     *
     * @param sessionID id of the session to get cached ui
     *                  traces for
     * @return List of {@link ExecutionTraceCacheModel}
     */
    @Nullable
    List<UiTraceCacheModel> getUiTracesForSession(String sessionID);
}
