package com.instabug.apm.cache.handler.session;

import androidx.annotation.NonNull;
import androidx.annotation.Nullable;

import com.instabug.apm.cache.model.SessionCacheModel;
import com.instabug.apm.constants.Constants;
import com.instabug.library.model.common.Session;

import java.util.Collection;
import java.util.List;

public interface SessionCacheHandler {

    /**
     * Creates a {@link SessionCacheModel} from the passed core session and inserts it to the database
     * as a new record.
     *
     * @param coreSession the core session in which the APM session object will be created from
     * @return SessionCacheModel with a valid database ID
     */
    @Nullable
    SessionCacheModel insert(@NonNull Session coreSession);

    /**
     * Updates an existing session record in the database
     *
     * @param session to be appended to or updated in the database
     * @return number of affected rows
     */
    int update(@NonNull SessionCacheModel session);

    /**
     * Queries a session record in database given its {@code identifier}
     *
     * @param id apm or core session ID
     * @return the record represented by {@link SessionCacheModel} if found or null otherwise.
     */
    @Nullable
    SessionCacheModel query(@NonNull String id);

    /**
     * Queries all session records in the database without any size constrains.
     *
     * @return a collection of records represented by {@link SessionCacheModel} if found
     * or empty collection otherwise.
     */
    @NonNull
    Collection<SessionCacheModel> queryAll();

    /**
     * Queries all session records in the database limited by a specific number of records.
     *
     * @return a collection of records represented by {@link SessionCacheModel} if found
     * or empty collection otherwise.
     */
    @NonNull
    Collection<SessionCacheModel> queryAll(int limit);

    /**
     * Queries sessions by core session ids
     *
     * @param coreIds list of core sessions ids
     * @return a collection of records represented by {@link SessionCacheModel} if found
     * or empty collection otherwise.
     */
    @NonNull
    List<SessionCacheModel> queryByCoreIds(@NonNull List<String> coreIds);

    /**
     * Deletes the session/s associated with this ID from this database if it is in the given list.
     * Otherwise, the records remain unchanged.
     *
     * @param coreIds list of core session ID
     */
    void deleteByCoreIds(@NonNull List<String> coreIds);

    /**
     * Deletes the session/s associated with this ID from this database if it is present.
     * Otherwise, the records remain unchanged.
     *
     * @param id apm or core session ID
     * @return number of affected rows
     */
    int delete(@NonNull String id);

    /**
     * Get all APM sessions linked With V2 Sessions  which are marked as ready to be sent
     * see {@link com.instabug.apm.constants.Constants.SessionSyncStatus}
     *
     * @return list of ready to be sent sessions
     */
    @NonNull
    List<SessionCacheModel> getReadyToBeSentSessions();

    /**
     * Gets the session following to the current session id
     *
     * @param currentSessionID id of the current session
     * @return next session model if current session is not the last session, null otherwise
     */
    @Nullable
    SessionCacheModel getNextSession(String currentSessionID);

    /**
     * Changes sync status for list of sessions
     *
     * @param sessionsIDs list of sessions ids to change sync tatus for
     * @param syncStatus  new sync status. see {@link com.instabug.apm.constants.Constants.SessionSyncStatus}
     */
    void changeSessionSyncStatus(@NonNull List<String> sessionsIDs,
                                 @Constants.SessionSyncStatus int syncStatus);

    /**
     * Delete cached sessions according to its sync status.
     * see {@link com.instabug.apm.constants.Constants.SessionSyncStatus}
     *
     * @param syncStatus session sync status
     */
    void deleteSessionsBySyncStatus(@Constants.SessionSyncStatus int syncStatus);

    /**
     * Queries the previous session of the session with the passed id
     *
     * @return the session if found or null otherwise
     */
    @Nullable
    SessionCacheModel getPreviousSession(String sessionId);

    /**
     * Updates an existing session end reason in the database
     *
     * @param coreSessionId   id of core session to be updated
     * @param sessionDuration duration of session
     * @param reason          session end reason
     * @return number of affected rows
     */
    int updateEndReason(@NonNull String coreSessionId, long sessionDuration, @Constants.SessionEndStatusCode int reason);

    /**
     * Trim APM sessions table if rows count exceeds the passed max row count
     *
     * @return number of dropped sessions
     */
    int trimSessions(int maxRowCount);

}